#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import re
import os
import sys
import syslog
import requests
import subprocess
import datetime
import time
import json
import select


# ==================================================================================================================

InfraKey = "iddqd"

# ==================================================================================================================

ServiceDict = {
    "SolomonCloud": {
        "infra": "SolomonCloud",
        "fetcher": {
            "sas": {
                "servers": ["CG@cloud_prod_solomon-core_sas"],
                "z2": "SOLOMON_CLOUD_PROD_FETCHER",
                "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
            },
            "vla": {
                "servers": ["CG@cloud_prod_solomon-core_vla"],
                "z2": "SOLOMON_CLOUD_PROD_FETCHER",
                "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
            }
        },
        "gateway": {
            "servers": ["CG@cloud_prod_solomon-gateway"],
            "z2": "SOLOMON_CLOUD_PROD_GATEWAY",
            "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
        },
        "alert": {
            "servers": ["CG@cloud_prod_solomon-gateway"],
            "z2": "SOLOMON_CLOUD_PROD_GATEWAY",
            "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
        },
        "storage": {
            "sas": {
                "servers": ["CG@cloud_prod_solomon-stockpile_sas"],
                "z2": "SOLOMON_CLOUD_PROD_STORAGE_SAS",
                "z2key": "97ef4059-6366-48c2-83ce-a651469d7932"
            },
            "vla": {
                "servers": ["CG@cloud_prod_solomon-stockpile_vla"],
                "z2": "SOLOMON_CLOUD_PROD_STORAGE_VLA",
                "z2key": "97ef4059-6366-48c2-83ce-a651469d7932"
            }
        }
    },
    "Solomon": {
        "infra": "Solomon",
        "fetcher": {
            "sas": {
                "servers": ["CG@solomon_prod_fetcher_sas"],
                "z2": "SOLOMON_PROD_FETCHER_SAS",
                "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
            },
            "vla": {
                "servers": ["CG@solomon_prod_fetcher_vla"],
                "z2": "SOLOMON_PROD_FETCHER_VLA",
                "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
            }
        },
        "gateway": {
            "servers": ["CG@solomon_prod_kfront"],
            "z2": "SOLOMON_PROD_GATEWAY",
            "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
        },
        "alert": {
            "servers": ["CG@solomon_prod_kfront"],
            "z2": "SOLOMON_PROD_GATEWAY",
            "z2key": "220a0a97-6e05-4f3f-ac23-330f54a88efb"
        },
        "storage": {
            "sas": {
                "servers": ["CG@solomon_prod_storage_sas"],
                "z2": "SOLOMON_PROD_STORAGE_SAS",
                "z2key": "f3f97eba-26b0-49c0-b929-8eaafe5b4441"
            },
            "vla": {
                "servers": ["CG@solomon_prod_storage_vla"],
                "z2": "SOLOMON_PROD_STORAGE_VLA",
                "z2key": "f3f97eba-26b0-49c0-b929-8eaafe5b4441"
            }
        }
    }
}

# ==================================================================================================================

InfraServiceDict = {
    "Solomon": {
        "id": 4,
        "production": 10,
        "prestable": 11
    },
    "SolomonCloud": {
        "id": 309,
        "production": 428,
        "prestable": 429,
        "testing": 430,
        "dev": 445,
        "hwci": 464
    },
    "Grafana": {
        "id": 342,
        "production": 542
    }
}

# ==================================================================================================================


def Log(String):
    print("%s - %s" % (datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f"), String))


class Exec:
    def __init__(self, command):
        try:
            p = subprocess.Popen(command, env={"LANG": "C"}, stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True)
            self.out = p.stdout.read().strip()
            self.err = p.stderr.read().strip()
            p.wait()
            self.status = p.returncode
            self.all = "{'out': '%s', 'err': '%s'}" % (self.out, self.err)
        except OSError as e:
            Log("Got exception running %s: %s" % (command, e))
            raise


def AskYN(String, Default=True, Timeout=None):
    sys.stdout.write("%s? %s[%s]: " % (String, ("(timeout=%d) " % Timeout) if Timeout else "", "Y/n" if Default else "y/N"))
    sys.stdout.flush()
    if Timeout:
        L, _, _ = select.select([sys.stdout], [], [], Timeout)
    if Timeout is None or len(L) > 0:
        Result = os.read(sys.stdout.fileno(), 1024).strip()
        Result = (Result not in ["n", "N"]) if Default else (Result in ["y", "Y"])
    else:
        sys.stdout.write("\n")
        Result = Default
    return Result


def Wait():
    if AskYN("Break"):
        Log("Exiting...")
        exit(0)


def Sleep(Time):
    Log("Sleeping %.3f seconds" % Time)
    C = 0
    while Time > 0:
        if C%30 == 0:
            sys.stdout.write("%s  %7.3f seconds left ." % (("" if C == 0 else "\n"), Time))
        else:
            sys.stdout.write("%s." % ("  " if C%10 == 0 else " " if C%5 == 0 else ""))
        sys.stdout.flush()
        time.sleep(1)
        C += 1
        Time -= 1
    print("")
    Log("continue")


# ==================================================================================================================


reBracedDots = re.compile("^([0-9]+)\.\.([0-9]+)$")
def UnwindString(String):
    def OpenRange(Begin, End):
        Zero  = "0" + str(max(len(Begin), len(End))) if Begin[0] == "0" or End[0] == "0" else ""
        Begin = int(Begin)
        End   = int(End)
        Step  = 1 if Begin <= End else -1
        return ["{0:{1}}".format(N, Zero) for N in range(Begin, End + Step, Step)]
    def GetToken(String, Pointer=0, Depth=0):
        PointerMax = len(String)
        SubStringStart = Pointer
        MultStart = 0
        Array = ['']
        while Pointer < PointerMax:
            c = String[Pointer]
            if c == '{':
                SubString = String[SubStringStart:Pointer]
                Array = Array + [SubString] if MultStart >= len(Array) else Array[:MultStart] + [x + SubString for x in Array[MultStart:]]
                SubArray, Pointer = GetToken(String, Pointer + 1, Depth + 1)
                Array = Array + SubArray if MultStart >= len(Array) else Array[:MultStart] + [x + y for x in Array[MultStart:] for y in SubArray]
                SubStringStart = Pointer
            elif c == ',' and Depth > 0:
                SubString = String[SubStringStart:Pointer]
                Array = Array + [SubString] if MultStart >= len(Array) else Array[:MultStart] + [x + SubString for x in Array[MultStart:]]
                Pointer += 1
                MultStart = len(Array)
                SubStringStart = Pointer
            elif c == '}' and Depth > 0:
                SubString = String[SubStringStart:Pointer]
                MatchObject = reBracedDots.search(SubString)
                if MatchObject is not None:
                    return OpenRange(MatchObject.group(1), MatchObject.group(2)), Pointer + 1
                Array = Array + [SubString] if MultStart >= len(Array) else Array[:MultStart] + [x + SubString for x in Array[MultStart:]]
                if len(Array) == 1:
                    Array = ["{" + Array[0] + "}"]
                return Array, Pointer + 1
            else:
                Pointer += 1
        SubString = String[SubStringStart:]
        return [x + SubString for x in Array]
    return GetToken(String)


# ==================================================================================================================


def InfraAdd(Service, Environment, Title, Description, Tickets, DCDict, SendEmail=True, Type="maintenance", Severity="minor", Start=None, End=None, Diff=None):
    URL = "https://infra-api.yandex-team.ru/v1/events"
    if not AskYN("Create new infra event", Default=False):
        print("Not creating infra event")
        return GetInfraDict().get("id")
    Log("Sending infra event: %s %s - '%s'" % (Service, Environment, Title))
    if Start is None:
        Start = int(time.time())
    if End is None:
        End = int(Start + (7200 if Diff is None else Diff))
    JsonDict = {
        "title": Title,
        "description": Description,
        "environmentId": InfraServiceDict[Service][Environment],
        "serviceId": InfraServiceDict[Service]["id"],
        "startTime": Start,
        "finishTime": End,
        "type": Type,
        "severity": Severity,
        "tickets": Tickets,
        "sendEmailNotifications": SendEmail
    }
    JsonDict.update(DCDict)
    r = requests.post(URL, data=json.dumps(JsonDict), headers={"Content-Type": "application/json", "Authorization": "OAuth %s" % InfraKey})
    if r.status_code == 200:
        try:
            Id = json.loads(r.text)["id"]
            DumpDict("infra", {
                "id": Id,
                "title": JsonDict["title"],
                "type": JsonDict["type"],
                "severity": JsonDict["severity"],
                "service": Service,
                "environment": Environment,
                "startTime": time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(float(JsonDict["startTime"]))),
                "tickets": JsonDict["tickets"]
            })
        except Exception as e:
            Log("Got from infra: %s" % r.text)
            exit()
        return Id
    return None


def InfraStop(End=None):
    Id = GetInfraDict().get("id")
    if Id is None:
        Log("No valid ids for infra event provided")
        return
    InfraModEndTime(Id, End)


def InfraModEndTime(EventId, End=None):
    URL = "https://infra-api.yandex-team.ru/v1/events/%d" % EventId
    if End is None:
        End = int(time.time())
    Log("Updating infra event id=%d for finishTime=%s" % (EventId, time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(End))))
    JsonDict = {
        "finishTime": End
    }
    r = requests.put(URL, data=json.dumps(JsonDict), headers={"Content-Type": "application/json", "Authorization": "OAuth %s" % InfraKey})
    if r.status_code == 200:
        return json.loads(r.text)["id"]
    return None


# ==================================================================================================================


def Conductor(Macros):
    URL = "http://c.yandex-team.ru/api/groups2hosts/%s" % Macros
    try:
        return requests.get(URL, timeout=5).text.strip().split()
    except Exception as e:
        Log("Failed to resolve conductor macro '%s'. %s" % (Macros, e))
        exit(1)


def ExpandItem(Item):
    ResultList = []
    if isinstance(Item, str) or isinstance(Item, str):
        Item = [Item]
    for String in Item:
        for S in String.split():
            ResultList += UnwindString(S)
    return ResultList


def ExpandServers(ServersItem):
    ResultList = []
    List = ExpandItem(ServersItem)
    for S in List:
        ResultList += Conductor(S[3:]) if S.startswith("CG@") else [S]
    return ResultList


# ==================================================================================================================


def SSH(User, Server, JumpServer, Cmd):
    Command = [
        "/usr/bin/ssh",
        "-ACn",
        "-o", "BatchMode=yes",
        "-o", "ServerAliveInterval=2",
        "-o", "TCPKeepAlive=yes",
        "-o", "ConnectTimeout=15",
        "-l", User,
        Server,
        Cmd
    ]
    if JumpServer is not None:
        Command.insert(1, "-J")
        Command.insert(2, JumpServer)
    return subprocess.Popen(Command, env=os.environ, stdout=subprocess.PIPE, stderr=subprocess.STDOUT, close_fds=True)


def EpollSSH(User, ServerList, BunchSize, Cmd, JumpServer, ExecTimeout=600, Verbose=False):
    def StopJob(Dict, Epoll, TempDict, Code):
        Dict["FinishTime"] = time.time()
        Dict["ReturnCode"] = Code
        if Verbose:
            Output(Dict)
        ResultDict.update({Dict["Server"]: {F:Dict[F] for F in ["Data", "StartTime", "FinishTime", "ReturnCode"]}})
        Epoll.unregister(Fileno)
        del TempDict[Fileno]
    ResultDict = {}
    SrvList = [s for s in ServerList]
    SrvList.reverse()
    TempDict = {}
    PollTime = 0.1
    Epoll = select.epoll()
    while True:
        while len(TempDict) < BunchSize and len(SrvList) > 0:
            Server = SrvList.pop()
            SubProcess = SSH(User, Server, JumpServer, Cmd)
            Dict = {"Server": Server, "SubProcess": SubProcess, "Data": "", "StartTime": time.time(), "FinishTime": 0, "ReturnCode": 0}
            Fileno = SubProcess.stdout.fileno()
            TempDict[Fileno] = Dict
            Epoll.register(Fileno, select.EPOLLIN | select.EPOLLET)
        if len(TempDict) == 0:
            break
        Events = Epoll.poll(PollTime)
        for Fileno, Event in Events:
            Dict = TempDict[Fileno]
            Data = Dict["SubProcess"].stdout.read()
            if len(Data) == 0:
                Dict["SubProcess"].wait()
            else:
                Dict["Data"] += Data
            if Dict["SubProcess"].poll() is not None:
                StopJob(Dict, Epoll, TempDict, Dict["SubProcess"].returncode)
        if ExecTimeout > 0:
            Time = time.time()
            DeleteList = []
            for Fileno, Dict in list(TempDict.items()):
                if Time - Dict["StartTime"] > ExecTimeout:
                    DeleteList.append(Fileno)
            for Fileno in DeleteList:
                Dict["Data"] += "too long to execute: over %d sec" % ExecTimeout
                StopJob(Dict, Epoll, TempDict, 128)
    Epoll.close()
    return ResultDict


def Output(Dict):
    if len(Dict["Data"]) == 0:
        return
    print("--- %s [%.3fsec/%s-%s] %s---\n%s" % (
        Dict["Server"],
        Dict["FinishTime"] - Dict["StartTime"],
        time.strftime("%H:%M:%S", time.localtime(Dict["StartTime"])),
        time.strftime("%H:%M:%S", time.localtime(Dict["FinishTime"])),
        ("ERROR(%d) " % Dict["ReturnCode"]) if Dict["ReturnCode"] != 0 else "",
        Dict["Data"].decode('utf-8').rstrip('\n')
    ))


# ==================================================================================================================


def Z2Edit(Config, ApiKey, PkgList):
    URL = "https://z2.yandex-team.ru/api/v1/editItems"
    Data = {
        "configId": Config,
        "apiKey": ApiKey,
        "items": json.dumps([dict(list(zip(["name", "version"], S.split('=')))) for S in PkgList], separators=(',', ':'))
    }
    Log("Z2 edit config (%s -> %s):" % (Config, json.dumps(PkgList)))
    r = requests.post(URL, data=Data)
    Log("Z2 edit config code=%s, data=%s" % (r.status_code, r.text))
    Log("Done")
    return r.status_code


def Z2UpdateStart(Config, ApiKey):
    URL = "https://z2.yandex-team.ru/api/v1/update"
    Data = {
        "configId": Config,
        "apiKey": ApiKey
    }
    Log("Z2 run update (%s)" % (Config))
    r = requests.post(URL, data=Data)
    Log("Z2 start update code=%s, data=%s" % (r.status_code, r.text))
    return r.status_code


def Z2CheckUpdate(Config, ApiKey):
    URL = "https://z2.yandex-team.ru/api/v1/updateStatus?configId=%s&apiKey=%s" % (Config, ApiKey)
    r = requests.get(URL)
    return r.text


def Z2Update(Config, ApiKey):
    if Z2UpdateStart(Config, ApiKey) != 200:
        Log("Z2 failed to start update")
        return False
    while True:
        Dict = json.loads(Z2CheckUpdate(Config, ApiKey))
        if Dict["response"]["updateStatus"] == "FINISHED":
            if Dict["response"]["result"] == "SUCCESS":
                Log("Z2 Success for %s" % Config)
                return True
            else:
                Log("Z2 Success for %s: %s - failed workers: %s" % (Config, Dict["response"]["result"], " ".join(Dict["response"]["failedWorkers"])))
                Log("Check https://z2.yandex-team.ru/control_panel?configId=%s" % Config)
                if AskYN("Wait for success"):
                    continue
                return False
        Log("%s" % Dict["response"]["updateStatus"])
        time.sleep(5)


# ==================================================================================================================


def PackageVersionsInstalled(User, ServerList, PkgList, JumpServer):
    ResultPkgDict = {}
    NoVersionPkgList = [P.split('=')[0] for P in PkgList]
    Cmd = "sudo dpkg-query -W -f '${Package}=${Version}\n' %s 2>/dev/null || true" % (" ".join(NoVersionPkgList))
    R = EpollSSH(User, ServerList, 1000, Cmd, JumpServer, ExecTimeout=600)
    for S, D in list(R.items()):
        # ssh timed out
        if D["ReturnCode"] != 0:
            ResultPkgDict[S] = None
            continue
        InstalledPkgList = D["Data"].split()
        ResultPkgDict[S] = [p for p in InstalledPkgList if p.split('=')[0] in NoVersionPkgList]
    return ResultPkgDict


def PackagesNotInDB(User, ServerList, PkgList, JumpServer):
    ResultPkgDict = {}
    NoVersionPkgList = [P.split('=')[0] for P in PkgList]
    Cmd = "sudo apt-cache show %s 2>/dev/null | awk '/^Package:/ {printf \"%%s=\", $2} /^Version:/ {print $2}'" % (" ".join(NoVersionPkgList))
    R = EpollSSH(User, ServerList, 1000, Cmd, JumpServer, ExecTimeout=600)
    for S, D in list(R.items()):
        # ssh timed out
        if D["ReturnCode"] != 0:
            ResultPkgDict[S] = None
            continue
        CachedPkgList = D["Data"].split()
        ResultPkgDict[S] = [p for p in PkgList if p not in CachedPkgList]
    return ResultPkgDict


def AptUpdate(User, ServerList, JumpServer):
    ResultPkgDict = {}
    Cmd = "sudo apt-get update >/dev/null"
    R = EpollSSH(User, ServerList, 1000, Cmd, JumpServer, ExecTimeout=600)
    for S, D in list(R.items()):
        # ssh timed out
        if D["ReturnCode"] != 0:
            ResultPkgDict[S] = None
            continue
        ResultPkgDict[S] = D["Data"]
    return ResultPkgDict


def PackagesAptDownload(User, ServerList, PkgList, JumpServer):
    ResultPkgDict = {}
    Cmd = "sudo apt-get install -d -y %s | awk '/^Get:/ {print $4 \"=\" $5}'" % (" ".join(PkgList))
    R = EpollSSH(User, ServerList, 1000, Cmd, JumpServer, ExecTimeout=600)
    for S, D in list(R.items()):
        # ssh timed out
        if D["ReturnCode"] != 0:
            ResultPkgDict[S] = None
            continue
        ResultPkgDict[S] = D["Data"].split()
    return ResultPkgDict


def ServersPrepare(User, ServerList, PkgList, JumpServer):
    Log("Preparing servers:\n  %s\n\nPackage list:\n  + %s\n" % (
        "\n  ".join(ServerList),
        "\n  + ".join(PkgList)
    ))
    ServersStateDict = {}
    NoVersionPkgList = [P.split('=')[0] for P in PkgList]

    Log("Current state:")
    ActualPkgList = []
    PackageVersionsInstalledDict = PackageVersionsInstalled(User, ServerList, PkgList, JumpServer)
    for S, InstalledPkgList in list(PackageVersionsInstalledDict.items()):
        if InstalledPkgList is None:
            print("%s: bad host" % (S))
            ServerList.remove(S)
            ServersStateDict[S] = {
                "orig":   {
                    "removed":      [],
                    "installed":    []
                },
                "target": {
                    "installed":    [P for P in PkgList]
                }
            }
        else:
            NoVersionInstalledPkgList = [P.split('=')[0] for P in InstalledPkgList]
            ServersStateDict[S] = {
                "orig":   {
                    "removed":      [P for P in NoVersionPkgList if P not in NoVersionInstalledPkgList],
                    "installed":    [P for P in InstalledPkgList if P not in PkgList]
                },
                "target": {
                    "installed":    [P for P in PkgList if P not in InstalledPkgList]
                }
            }
            for P in ServersStateDict[S]["target"]["installed"]:
                if P not in ActualPkgList:
                    ActualPkgList.append(P)
            if len(ServersStateDict[S]["target"]["installed"]) == 0:
                print("%s: - all packages are as required" % (S))
                ServerList.remove(S)
            else:
                print("%s:\n  + %s" % (S, "\n  + ".join(ServersStateDict[S]["target"]["installed"])))
    print("")

    Log("Dumping restore point")
    DumpDict("pkgStates", ServersStateDict)
    print("")

    Log("Not in db packages:")
    PackagesNotInDBDict = PackagesNotInDB(User, ServerList, ActualPkgList, JumpServer)
    ServersForUpdateList = [s for s in ServerList]
    for S, PList in list(PackagesNotInDBDict.items()):
        if PList is None:
            print("%s: bad host" % (S))
            ServersForUpdateList.remove(S)
        elif len(PList) == 0:
            print("%s: all packages are in DB" % (S))
            ServersForUpdateList.remove(S)
        else:
            print("%s: %s" % (S, " ".join(PList)))
    print("")

    if len(ServersForUpdateList) > 0:
        Log("Update result:")
        UpdateResult = AptUpdate(User, ServersForUpdateList, JumpServer)
        for S, R in list(UpdateResult.items()):
            print("%s: %s" % (S, R))
    else:
        Log("No need to update packge database.")
    print("")

    if len(ServersForUpdateList) > 0:
        Log("Download result:")
        DownloadResult = PackagesAptDownload(User, ServersForUpdateList, ActualPkgList, JumpServer)
        for S, R in list(DownloadResult.items()):
            try:
                print("%s: %s" % (S, " ".join(R)))
            except TypeError:
                print("%s: %s" % (S, R))
    else:
        Log("No need to download packages.")
    print("")

    return ServerList


# ==================================================================================================================


'''
JsonDB = {
    "pkgStates": {
        timestamp1: ServersStateDict1,
        timestamp2: ServersStateDict2,
        ...
    }
    "infra": {
        timestamp1: InfraDict1,
        timestamp2: InfraDict2,
        ...
    }
}
'''

def DumpDict(Key, ValueDict):
    JsonDB = {}
    MaxTime = 36*3600
    Time = time.time()
    try:
        LocalDBFile = open(LocalDBFileName, "r")
        JsonDB = json.load(LocalDBFile)
        LocalDBFile.close()
    except Exception as e:
        print("error while reading DB file: %s" % e)
    KeyDict = {k: v for k, v in list(JsonDB.get(Key, {}).items()) if Time - float(k) <= MaxTime}
    KeyDict.update({Time: ValueDict})
    JsonDB.update({Key: KeyDict})
    try:
        LocalDBFile = open(LocalDBFileName, "w")
        json.dump(JsonDB, LocalDBFile)
        LocalDBFile.close()
    except Exception as e:
        print("error while writing DB file: %s" % e)


def LoadDict(Key):
    JsonDB = {}
    try:
        LocalDBFile = open(LocalDBFileName, "r")
        JsonDB = json.load(LocalDBFile)
        LocalDBFile.close()
    except Exception as e:
        print("error while reading DB file: %s" % e)
    return JsonDB.get(Key, {})


reNumber = re.compile("^[0-9]+$")
def SelectFromDict(RestorePointsDict):
    if len(RestorePointsDict) == 0:
        print("Cannot find any restore points for this set of servers")
        return {}
    Counter = 0
    CountedPointsDict = {}
    print("Select one of the following restore points:")
    for P in sorted(RestorePointsDict):
        print("  %2d. %s ~ %s" % (Counter, time.strftime("%Y-%m-%d %H:%M:%S", time.localtime(float(P))), json.dumps(RestorePointsDict[P], ensure_ascii=False)))
        CountedPointsDict[Counter] = P
        Counter += 1
    Counter -= 1
    while True:
        sys.stdout.write("please select one, type 'q' to quit [%d]: " % (Counter))
        sys.stdout.flush()
        N = os.read(sys.stdout.fileno(), 1024).strip()
        if len(N) == 0:
            N = Counter
            break
        elif N.startswith("q") or N.startswith("Q"):
            print("quitting...")
            return {}
        elif reNumber.match(N) and int(N) <= Counter:
            N = int(N)
            break
        else:
            Log("Bad selection: %s. Try again ..." % N)
    ResultDict = RestorePointsDict[CountedPointsDict[N]]
    print("Selected %s" % (json.dumps(ResultDict, ensure_ascii=False)))
    return ResultDict


def GetRestorePoint(ServerList):
    ServersSet = set(ServerList)
    PkgStatesDict = LoadDict("pkgStates")
    RestorePointDict = {}
    for Time, ServerState in list(PkgStatesDict.items()):
        SrvSet = set([str(S) for S in ServerState])
        if ServersSet == SrvSet:
            for S in ServerState:
                if len(ServerState[S]["orig"]["installed"]) > 0:
                    RestorePointDict[Time] = ServerState[S]["orig"]
                    break
    return SelectFromDict(RestorePointDict)


def GetInfraDict():
    InfraDict = LoadDict("infra")
    return SelectFromDict(InfraDict)


# ==================================================================================================================


def ServersUpdate(User, ServerList, BunchSize, JumpServer, UpdateCmd, PkgList):
    R = EpollSSH(User, ServerList, BunchSize, "__func__() { \n%s\n } ; ( flock -w 1 9 || exit 1 ; __func__ %s ) 9>%s &" % (UpdateCmd, " ".join(PkgList), RemoteUpdateLockFile), JumpServer, Verbose=True)
    return R


def ServersRollback(User, ServerList, BunchSize, JumpServer):
    RestorePointDict = GetRestorePoint(ServerList)

    if AskYN("Do you want slow rollback", Default=False):
        return RestorePointDict

    RollbackCmd   = "echo 'Begin rollback' ;"
    RemovedPkgs   = " ".join(RestorePointDict["removed"])
    InstalledPkgs = " ".join(RestorePointDict["installed"])

    if len(RemovedPkgs) > 0:
        RollbackCmd += "sudo apt-get purge -y %s 2>&1 ;" % (RemovedPkgs)
    if len(InstalledPkgs) > 0:
        RollbackCmd += "sudo apt-get install -y --force-yes %s 2>&1 ;" % (InstalledPkgs)
    RollbackCmd += "echo 'End rollback' ;"
    EpollSSH(User, ServerList, BunchSize, "__func__() { \n%s\n } ; ( flock -w 1 9 || exit 1 ; __func__ ) 9>%s &" % (RollbackCmd, RemoteUpdateLockFile), JumpServer, Verbose=True)
    return {}


def ServersCancel(User, ServerList, BunchSize, JumpServer):
    EpollSSH(User, ServerList, BunchSize, "sudo fuser -k %s" % (RemoteUpdateLockFile), JumpServer, Verbose=True)


# ==================================================================================================================


def SolomonFreezeShards(User, Server, JumpServer=None):
    R = EpollSSH(User, [Server], 1, "curl -qs -L 'http://localhost:4500/balancer/allNodeFreeze?flag=true' -o /dev/null -w '%{http_code}'", JumpServer, Verbose=False)
    for S, SDict in list(R.items()):
        print("  Freeze shards: (%s/%s) %s" % (S, SDict["ReturnCode"], SDict["Data"]))


def SolomonUnFreezeShards(User, Server, JumpServer=None):
    R = EpollSSH(User, [Server], 1, "curl -qs -L 'http://localhost:4500/balancer/allNodeFreeze?flag=false' -o /dev/null -w '%{http_code}'", JumpServer, Verbose=False)
    for S, SDict in list(R.items()):
        print("  Unfreeze shards: (%s/%s) %s" % (S, SDict["ReturnCode"], SDict["Data"]))


def SolomonStockpileHandle(User, ServerList, JumpServer=None):
    Bean    = "ru.yandex.stockpile.server.shard.StockpileShardLoader"
    Method  = "forceShardsLogsSnapshot"
    Desc    = "(J)Ljava/lang/String;"
    URL     = "http://localhost:4500/manager?bean=%s&method=%s&desc=%s&invoke=1&p0=200" % (Bean, Method, Desc)
    R = EpollSSH(User, ServerList, 1, "curl -qs '%s' -o /dev/null -w '%%{http_code}'" % URL, JumpServer, Verbose=False)
    for S, SDict in list(R.items()):
        print("  %s: (%s) %s" % (S, SDict["ReturnCode"], SDict["Data"]))


def UpdateStockpile(Service, DC, Ticket, Pkgs, PurgePkgs=""):
    PkgList = ExpandItem(Pkgs)
    PurgePkgList = ExpandItem(PurgePkgs)

    InfraTitle = "Обновление Stockpile"
    InfraText = "Запланированный релиз Stockpile. Должно пройти незаметно для пользователей."
    InfraTicket = Ticket

    ServerList = ExpandServers(ServiceDict[Service]["storage"][DC]["servers"])
    Z2Config = ServiceDict[Service]["storage"][DC]["z2"]
    Z2ApiKey = ServiceDict[Service]["storage"][DC]["z2key"]
    ServerList.sort()

    User = os.getlogin()
    JumpServer = "bastion.cloud.yandex.net" if "Cloud" in Service else None

    Rollback = False
    if AskYN("Cancel current update", Default=False):
        ServersCancel(User, ServerList, 1000, JumpServer)
    if AskYN("Rollback current update", Default=False):
        RollbackDict = ServersRollback(User, ServerList, 1000, JumpServer)
        if len(RollbackDict) == 0:
            return
        Rollback = True
        PkgList = RollbackDict["installed"]
        PurgePkgList = RollbackDict["removed"]

    if not Rollback:
        InfraId = InfraAdd(ServiceDict[Service]["infra"], "production", InfraTitle, InfraText, InfraTicket, {DC: True})

    ServerList = ServersPrepare(User, ServerList, PkgList, JumpServer)
    print("Servers prepared\n")

    print("Check: https://solomon.yandex-team.ru/?project=solomon&cluster=prestable&service=stockpile&dashboard=stockpile-cross-dc-summary2\n")

    if AskYN("Freeze shards"):
        SolomonFreezeShards(User, ServerList[0], JumpServer)
        print("")

    if AskYN("Pull stockpile handle"):
        SolomonStockpileHandle(User, ServerList, JumpServer)
        print("")

    UpdateCmd = """
            if [ -n "$1" ] ; then
                sudo apt-get purge -y $1 2>&1
            fi
            if [ -n "$2" ] ; then
                sudo apt-get install -y {0} $2 2>&1
            fi
        """.format("--force-yes" if Rollback else "")


    print("Ready to update!")
    Wait()
    ServersUpdate(User, [ServerList[0]], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % " ".join(PkgList)])
    print("")

    if not Rollback:
        Wait()
    ServersUpdate(User, ServerList[1:], 1000, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % " ".join(PkgList)])
    print("")

    print("Ready to update Z2!")
    Wait()
    Z2Edit(Z2Config, Z2ApiKey, PkgList)
    print("")

    if AskYN("Unfreeze shards in 5 min"):
        Sleep(300)
        Log("Unfreeze shards")
        SolomonUnFreezeShards(User, ServerList[0], JumpServer)

    if AskYN("About to close infra event. Continue"):
        if InfraId is None:
            InfraStop(time.time() + 300)
        else:
            InfraModEndTime(InfraId, time.time() + 300)



# ==================================================================================================================


def UpdateAlerting(Service, Ticket, Pkgs, PurgePkgs=""):
    PkgList = ExpandItem(Pkgs)
    PurgePkgList = ExpandItem(PurgePkgs)

    InfraTitle = "Обновление Solomon Alerting"
    InfraText = "Обновление на новую версию. Пользователей может затронуть в виде прихода алертов из 10 минутного прошлого"
    InfraTicket = Ticket

    ServerList = ExpandServers(ServiceDict[Service]["alert"]["servers"])
    Z2Config = ServiceDict[Service]["alert"]["z2"]
    Z2ApiKey = ServiceDict[Service]["alert"]["z2key"]
    ServerList.sort()

    User = os.getlogin()
    JumpServer = "bastion.cloud.yandex.net" if "Cloud" in Service else None

    Rollback = False
    if AskYN("Cancel current update", Default=False):
        ServersCancel(User, ServerList, 1000, JumpServer)
    if AskYN("Rollback current update", Default=False):
        RollbackDict = ServersRollback(User, ServerList, 1000, JumpServer)
        if len(RollbackDict) == 0:
            return
        Rollback = True
        PkgList = RollbackDict["installed"]
        PurgePkgList = RollbackDict["removed"]

    Master = ""
    Log("Looking for the alerting master")
    R = EpollSSH(User, ServerList, 1000, "/usr/bin/curl -qso /dev/null -m 2 -w %{http_code} 'http://localhost:8608/balancer'", JumpServer, Verbose=False)
    for S, Dict in list(R.items()):
        if Dict["Data"] == "200":
            Master = S
            break
    if len(Master) == 0:
        Log("Cannot find alerting master!")
        return
    else:
        Log("Alerting master is %s" % Master)
        ServerList.remove(Master)
        ServerList.append(Master)

    if not Rollback:
        InfraId = InfraAdd(ServiceDict[Service]["infra"], "production", InfraTitle, InfraText, InfraTicket, {"sas": True, "vla": True})

    ServerList = ServersPrepare(User, ServerList, PkgList, JumpServer)
    Log("Servers prepared\n")

    print("Check: https://solomon.yandex-team.ru/?project=solomon&cluster=production&dashboard=alerting-projects-assignment-summary-dashboard&b=1h&e=\n")

    UpdateCmd = """
            if [ -n "$1" ] ; then
                sudo apt-get purge -y $1 2>&1
            fi
            if [ -n "$2" ] ; then
                sudo apt-get install -y {0} $2 2>&1
            fi
        """.format("--force-yes" if Rollback else "")

    Wait()
    ServersUpdate(User, [ServerList[0]], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % " ".join(PkgList)])
    print("")

    if not Rollback:
        Wait()
    for Server in ServerList[1:]:
        Sleep(150)
        ServersUpdate(User, [Server], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % " ".join(PkgList)])
        print("")

    Wait()
    Z2Edit(Z2Config, Z2ApiKey, PkgList)
    print("")

    if AskYN("About to close infra event. Continue"):
        if InfraId is None:
            InfraStop(time.time() + 300)
        else:
            InfraModEndTime(InfraId, time.time() + 300)


# ==================================================================================================================


def UpdateFetchers(Service, DC, Ticket, Pkgs, PurgePkgs=""):
    PkgList = ExpandItem(Pkgs)
    PurgePkgList = ExpandItem(PurgePkgs)

    InfraTitle = "Обновление Solomon"
    InfraText = "Обновление на новую версию. Для пользователей должно пройти незаметно"
    InfraTicket = Ticket

    ServerList = ExpandServers(ServiceDict[Service]["fetcher"][DC]["servers"])
    Z2Config = ServiceDict[Service]["fetcher"][DC]["z2"]
    Z2ApiKey = ServiceDict[Service]["fetcher"][DC]["z2key"]
    ServerList.sort()

    User = os.getlogin()
    JumpServer = "bastion.cloud.yandex.net" if "Cloud" in Service else None

    Rollback = False
    if AskYN("Cancel current update", Default=False):
        ServersCancel(User, ServerList, 1000, JumpServer)
    if AskYN("Rollback current update", Default=False):
        RollbackDict = ServersRollback(User, ServerList, 1000, JumpServer)
        if len(RollbackDict) == 0:
            return
        Rollback = True
        PkgList = RollbackDict["installed"]
        PurgePkgList = RollbackDict["removed"]

    if not Rollback:
        InfraId = InfraAdd(ServiceDict[Service]["infra"], "production", InfraTitle, InfraText, InfraTicket, {"sas": (DC == "sas"), "vla": (DC == "vla")})

    ServerList = ServersPrepare(User, ServerList, PkgList, JumpServer)
    Log("Servers prepared\n")

    if len(ServerList) > 0:
        print("Check: https://solomon.yandex-team.ru/?cluster=production&l.host=%s&project=%s&l.projectId=total&l.sensor=engine.shard.count&service=coremon&l.shardId=total&l.state=LOADING&graph=auto&b=30m&e=\n"
              "       https://solomon.yandex-team.ru/?project=solomon&cluster=production&service=coremon&host=solomon-fetcher-%s-*&dashboard=solomon-coremon-shards-load&b=1h&e=\n"
              "       https://solomon.yandex-team.ru/?cluster=production&host=%s&project=solomon&service=coremon&dashboard=solomon-data-process-per-host&b=30m&e=\n"
                % ("Vla" if DC == "vla" else "Sas", "solomon_cloud" if "Cloud" in Service else "solomon",
                    DC,
                    ServerList[0].split('.')[0])
             )

        PkgsNoCoremon = " ".join([P for P in PkgList if not "coremon" in P])
        PkgsCoremon = " ".join([P for P in PkgList if "coremon" in P])

        UpdateCmd = """
            if [ -n "$1" ] ; then
                sudo apt-get purge -y $1 2>&1
            fi
            if [ -n "$2" ] ; then
                sudo s stop fetcher\$
                sudo apt-get install -y {0} $2 2>&1
            fi
            if [ -n "$3" ] ; then
                sudo apt-get install -y {0} $3 2>&1
            else
                sudo s start fetcher\$
            fi
        """.format("--force-yes" if Rollback else "")

        Wait()
        ServersUpdate(User, [ServerList[0]], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % PkgsCoremon, "'%s'" % PkgsNoCoremon])
        print("")

        if not Rollback:
            Wait()
        for Server in ServerList[1:]:
            ServersUpdate(User, [Server], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % PkgsCoremon, "'%s'" % PkgsNoCoremon])
            print("")

    Wait()
    Z2Edit(Z2Config, Z2ApiKey, PkgList)
    print("")

    if AskYN("About to close infra event. Continue"):
        if InfraId is None:
            InfraStop(time.time() + 300)
        else:
            InfraModEndTime(InfraId, time.time() + 300)


# ==================================================================================================================


def UpdateGateway(Service, Ticket, Pkgs, PurgePkgs=""):
    PkgList = ExpandItem(Pkgs)
    PurgePkgList = ExpandItem(PurgePkgs)

    InfraTitle = "Обновление Gateway"
    InfraText = "Запланированный релиз Gateway. Должно пройти незаметно для пользователей."
    InfraTicket = Ticket

    ServerList = ExpandServers(ServiceDict[Service]["gateway"]["servers"])
    Z2Config = ServiceDict[Service]["gateway"]["z2"]
    Z2ApiKey = ServiceDict[Service]["gateway"]["z2key"]
    ServerList.sort()

    User = os.getlogin()
    JumpServer = "bastion.cloud.yandex.net" if "Cloud" in Service else None

    Rollback = False
    if AskYN("Cancel current update", Default=False):
        ServersCancel(User, ServerList, 1000, JumpServer)
    if AskYN("Rollback current update", Default=False):
        RollbackDict = ServersRollback(User, ServerList, 1000, JumpServer)
        if len(RollbackDict) == 0:
            return
        Rollback = True
        PkgList = RollbackDict["installed"]
        PurgePkgList = RollbackDict["removed"]

    if not Rollback:
        InfraId = InfraAdd(ServiceDict[Service]["infra"], "production", InfraTitle, InfraText, InfraTicket, {"sas": True, "vla": True})

    ServerList = ServersPrepare(User, ServerList, PkgList, JumpServer)
    print("Servers prepared\n")

    UpdateCmd = """
            if [ -n "$1" ] ; then
                sudo apt-get purge -y $1 2>&1
            fi
            if [ -n "$2" ] ; then
                sudo apt-get install -y {0} $2 2>&1
            fi
        """.format("--force-yes" if Rollback else "")

    Wait()
    ServersUpdate(User, [ServerList[0]], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % " ".join(PkgList)])
    print("")

    if not Rollback:
        Wait()
    for Server in ServerList[1:]:
        Sleep(10)
        ServersUpdate(User, [Server], 1, JumpServer, UpdateCmd, ["'%s'" % " ".join(PurgePkgList), "'%s'" % " ".join(PkgList)])
        print("")

    Z2Edit(Z2Config, Z2ApiKey, PkgList)
    print("")

    if AskYN("About to close infra event. Continue"):
        if InfraId is None:
            InfraStop(time.time() + 300)
        else:
            InfraModEndTime(InfraId, time.time() + 300)


# ==================================================================================================================


# global variables
LocalDBFileName = "/dev/shm/server_updater.db"
RemoteUpdateLockFile = "/var/lock/remote_update.lock"


# ==================================================================================================================

'''
Service = "Solomon"
Z2Config = ServiceDict[Service]["gateway"]["z2"]
Z2ApiKey = ServiceDict[Service]["gateway"]["z2key"]
PkgList = ExpandItem("yandex-solomon-web=5032358.stable-2019-05-13 yandex-solomon-gateway=5034746.stable-2019-05-13 yandex-solomon-gateway-conf=5028686.stable-2019-05-13 yandex-solomon-common-conf=5035645.stable-2019-05-13")
Z2Edit(Z2Config, Z2ApiKey, PkgList)
'''

# ==================================================================================================================

#DC = "sas"
DC = "vla"

Ticket = "SOLOMON-4133"
#Ticket = "SOLOMON-4134"

#Service = "SolomonCloud"
Service = "Solomon"

StockpilePkgs = """
yandex-solomon-stockpile=5074835.stable-2019-05-27
# yandex-solomon-stockpile-conf=5074835.stable-2019-05-27
yandex-solomon-stockpile-conf-cloud=5074835.stable-2019-05-27
yandex-solomon-common-conf=5074835.stable-2019-05-27
"""
FetchersPkgs = """
# yandex-solomon-common-conf=5028686.stable-2019-05-13
yandex-solomon-fetcher=5081912.stable-2019-05-27
yandex-solomon-fetcher-conf=5081912.stable-2019-05-27
# yandex-solomon-fetcher-conf-cloud=5050018.stable-2019-05-13
# yandex-solomon-coremon=5028686.stable-2019-05-13
# yandex-solomon-coremon-conf-cloud=5050018.stable-2019-05-13
"""
GatewayPkgs = """
# yandex-solomon-nginx-conf=5049978.stable-2019-05-13
# yandex-solomon-nginx-conf-cloud=5050004.stable-2019-05-13
yandex-solomon-web=5064106.trunk
yandex-solomon-gateway=5064106.trunk
# yandex-solomon-gateway-conf=5028686.stable-2019-05-13
# yandex-solomon-gateway-conf-cloud=5050004.stable-2019-05-13
# yandex-solomon-common-conf=5035645.stable-2019-05-13
"""
AlertingPkgs = """
yandex-solomon-common-conf=5035645.stable-2019-05-13
yandex-solomon-alerting=5035645.stable-2019-05-13
yandex-solomon-alerting-conf-cloud=5050004.stable-2019-05-13
"""

UpdateStockpile(Service, DC, Ticket, re.sub("#.+(\n|$)", "", StockpilePkgs))
#UpdateFetchers(Service, DC, Ticket, re.sub("#.+(\n|$)", "", FetchersPkgs))
#UpdateGateway(Service, Ticket, re.sub("#.+(\n|$)", "", GatewayPkgs))
#UpdateAlerting(Service, Ticket, re.sub("#.+(\n|$)", "", AlertingPkgs))
