package ru.yandex.solomon.alert;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.MoreObjects;
import com.google.common.base.Strings;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class EvaluationStatus {
    private static final String DESCRIPTION_KEY = "description";

    public static final EvaluationStatus NO_DATA = Code.NO_DATA.toStatus();
    public static final EvaluationStatus OK = Code.OK.toStatus();
    public static final EvaluationStatus ERROR = Code.ERROR.toStatus();
    public static final EvaluationStatus DEADLINE = ERROR.withErrorCode(ErrorCode.DEADLINE);
    public static final EvaluationStatus UNAVAILABLE = ERROR.withErrorCode(ErrorCode.UNAVAILABLE);
    public static final EvaluationStatus ALARM = Code.ALARM.toStatus();
    public static final EvaluationStatus WARN = Code.WARN.toStatus();

    private final Code code;
    private final Map<String, String> annotations;
    private final Map<String, String> serviceProviderAnnotations;
    private final ErrorCode errorCode;
    private final List<ScalarValue> scalarValues;

    private EvaluationStatus(
            Code code,
            Map<String, String> annotations,
            Map<String, String> serviceProviderAnnotations,
            ErrorCode errorCode,
            List<ScalarValue> scalarValues)
    {
        this.code = code;
        this.annotations = annotations;
        this.serviceProviderAnnotations = serviceProviderAnnotations;
        this.errorCode = errorCode;
        this.scalarValues = scalarValues;
    }

    private EvaluationStatus(Code code, Map<String, String> annotations, Map<String, String> serviceProviderAnnotations) {
        this(code, annotations, serviceProviderAnnotations, ErrorCode.UNSPECIFIED, List.of());
    }

    public Code getCode() {
        return code;
    }

    public ErrorCode getErrorCode() {
        return errorCode;
    }

    /**
     * Details about status code
     */
    public String getDescription() {
        return annotations.getOrDefault(DESCRIPTION_KEY, "");
    }

    public Map<String, String> getAnnotations() {
        return annotations;
    }

    public Map<String, String> getServiceProviderAnnotations() {
        return serviceProviderAnnotations;
    }

    public List<ScalarValue> getScalarValues() {
        return scalarValues;
    }

    public EvaluationStatus withDescription(@Nullable String description) {
        if (Strings.isNullOrEmpty(description)) {
            return this;
        }

        if (annotations.isEmpty()) {
            return new EvaluationStatus(code, Map.of(DESCRIPTION_KEY, description),Map.copyOf(serviceProviderAnnotations), errorCode, scalarValues);
        }

        var extendedAnnotations = new HashMap<>(annotations);
        extendedAnnotations.put(DESCRIPTION_KEY, description);

        return new EvaluationStatus(code, Map.copyOf(extendedAnnotations), Map.copyOf(serviceProviderAnnotations), errorCode, scalarValues);
    }

    public EvaluationStatus withAnnotations(Map<String, String> annotations) {
        return new EvaluationStatus(code, Map.copyOf(annotations), serviceProviderAnnotations, errorCode, scalarValues);
    }

    public EvaluationStatus withServiceProviderAnnotations(Map<String, String> serviceProviderAnnotations) {
        return new EvaluationStatus(code, annotations, Map.copyOf(serviceProviderAnnotations), errorCode, scalarValues);
    }

    public EvaluationStatus withErrorCode(EvaluationStatus.ErrorCode errorCode) {
        return new EvaluationStatus(code, annotations, serviceProviderAnnotations, errorCode, scalarValues);
    }

    public EvaluationStatus withScalars(List<ScalarValue> scalars) {
        return new EvaluationStatus(code, annotations, serviceProviderAnnotations, errorCode, scalars);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        EvaluationStatus that = (EvaluationStatus) o;

        return (code == that.code) && (errorCode == that.errorCode);
    }

    @Override
    public int hashCode() {
        return code.hashCode() * 31 + errorCode.hashCode();
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .omitNullValues()
                .add("code", code)
                .add("annotations", annotations)
                .add("serviceProviderAnnotations", serviceProviderAnnotations)
                .add("errorCode", errorCode)
                .toString();
    }

    /**
     * List of available statue for alert, sorted by priority.
     *
     * @author Vladimir Gordiychuk
     */
    public enum Code {
        NO_DATA(0),
        OK(1),
        WARN(6),
        ERROR(3),
        ALARM(4),
        ;

        private final int number;

        Code(int number) {
            this.number = number;
        }

        public int getNumber() {
            return number;
        }

        public EvaluationStatus toStatus() {
            return toStatus(ErrorCode.UNSPECIFIED);
        }

        public EvaluationStatus toStatus(ErrorCode errorCode) {
            return new EvaluationStatus(this, Collections.emptyMap(), Collections.emptyMap(), errorCode, List.of());
        }

        public EvaluationStatus toStatus(String description) {
            if (Strings.isNullOrEmpty(description)) {
                return new EvaluationStatus(this, Map.of(), Map.of());
            }

            return new EvaluationStatus(this, Map.of(DESCRIPTION_KEY, description), Map.of());
        }

        public EvaluationStatus toStatus(Map<String, String> annotations, Map<String, String> serviceProviderAnnotations) {
            return new EvaluationStatus(this, Map.copyOf(annotations), Map.copyOf(serviceProviderAnnotations));
        }

        public static Code forNumber(int num) {
            for (Code code : values()) {
                if (code.getNumber() == num) {
                    return code;
                }
            }

            throw new IllegalArgumentException("Not found code with number: "+ num);
        }

        public static Code parse(String value) {
            for (Code code : values()) {
                if (code.name().equals(value)) {
                    return code;
                }
            }

            throw new IllegalArgumentException("Not found code by name: "+ value);
        }
    }

    public enum ErrorCode {
        UNSPECIFIED(0),
        DEADLINE(2),
        UNAVAILABLE(5),
        ;

        private final int number;

        ErrorCode(int number) {
            this.number = number;
        }

        public int getNumber() {
            return number;
        }

        public static ErrorCode forNumber(int num) {
            for (ErrorCode errorCode : values()) {
                if (errorCode.getNumber() == num) {
                    return errorCode;
                }
            }

            throw new IllegalArgumentException("Not found errorCode with number: "+ num);
        }
    }

    public enum ScalarType {
        STRING,
        DOUBLE,
        BOOLEAN,
    }

    public record ScalarValue(String name, @Nullable Object value, ScalarType type) {}
}
