package ru.yandex.solomon.alert.ambry;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.ambry.AmbryClient;
import ru.yandex.ambry.dto.LastUpdatedResponse;
import ru.yandex.ambry.dto.YasmAlertDto;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.api.AlertingException;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlert;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.util.WindowRateLimit;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.solomon.yasm.alert.converter.YasmAlertConverter;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AmbryPoller implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(AmbryPoller.class);
    private static final Duration deadline = Duration.ofSeconds(30);
    private static final long FORCE_REFRESH_MILLIS = Duration.ofMinutes(10).toMillis();
    private static final int ALERTING_LIST_PAGE_SIZE = 1000;

    public static final String SYNC_USER = "robot-yasm-golovan";

    private final YasmAlertConverter converter;

    private final FeatureFlagsHolder featureFlags;
    private final AlertApi alertApi;
    private final AmbryClient ambryClient;

    private final AmbryPollerMetrics metrics;

    private final ScheduledExecutorService executor;
    private final ActorWithFutureRunner actor;
    private final DistributedLock lock;

    private final WindowRateLimit pollLimiter;

    private final AtomicLong lastAmbryUpdatedMillis = new AtomicLong(0);
    private final AtomicLong lastProcessedMillis = new AtomicLong(0);
    private final Clock clock;

    @Nullable
    private ScheduledFuture<?> scheduledFuture;

    public AmbryPoller(
            MetricRegistry registry,
            FeatureFlagsHolder featureFlags,
            String yasmItypeProjectPrefix,
            AlertApi alertApi,
            AmbryClient ambryClient,
            ScheduledExecutorService executor,
            DistributedLock lock)
    {
        this(Clock.systemUTC(), registry, featureFlags, yasmItypeProjectPrefix, alertApi, ambryClient, executor, lock);
    }

    @VisibleForTesting
    public AmbryPoller(
            Clock clock,
            MetricRegistry registry,
            FeatureFlagsHolder featureFlags,
            String yasmItypeProjectPrefix,
            AlertApi alertApi,
            AmbryClient ambryClient,
            ScheduledExecutorService executor,
            DistributedLock lock)
    {
        this.clock = clock;
        this.pollLimiter = new WindowRateLimit(clock, 1, TimeUnit.MINUTES);
        this.featureFlags = featureFlags;
        this.converter = new YasmAlertConverter(yasmItypeProjectPrefix);
        this.alertApi = alertApi;
        this.ambryClient = ambryClient;
        this.executor = executor;
        this.actor = new ActorWithFutureRunner(this::act, executor);

        this.metrics = new AmbryPollerMetrics(registry);

        this.lock = lock;

        schedule(1_000);
    }

    private void schedule(long delayMillis) {
        if (delayMillis == 0) {
            actor.schedule();
            return;
        }
        scheduledFuture = executor.schedule(actor::schedule, DurationUtils.randomize(delayMillis), TimeUnit.MILLISECONDS);
    }

    private CompletableFuture<?> act() {
        return CompletableFutures.safeCall(this::checkRevisionUpdates)
                .whenComplete((r, t) -> schedule(5_000));
    }

    private CompletableFuture<?> checkRevisionUpdates() {
        if (!lock.isLockedByMe()) {
            return completedFuture(null);
        }

        return ambryClient.lastUpdatedTotal()
                .thenCompose(this::checkUpdated);
    }

    private CompletableFuture<?> checkUpdated(LastUpdatedResponse lastUpdatedResponse) {
        long prevUpdatedMillis = lastAmbryUpdatedMillis.longValue();
        long currUpdatedMillis = lastUpdatedResponse.response.lastUpdated.toEpochMilli();

        if (currUpdatedMillis < prevUpdatedMillis) {
            return completedFuture(null);
        }

        boolean forceUpdate = clock.millis() > lastProcessedMillis.get() + FORCE_REFRESH_MILLIS;
        if (currUpdatedMillis == prevUpdatedMillis && !forceUpdate) {
            return completedFuture(null);
        }

        if (currUpdatedMillis > prevUpdatedMillis) {
            logger.info("Alert list updated on " + lastUpdatedResponse.response.lastUpdated);
        } else {
            logger.info("Time to force update alert list");
        }

        if (!pollLimiter.attempt()) {
            logger.warn("Rate-limiting loading alert list");
            return completedFuture(null);
        }

        metrics.listUpdated();

        return ambryClient.list(AmbryClient.TagFormat.DYNAMIC)
                .thenCompose(this::processList)
                .thenAccept(aVoid -> {
                    lastAmbryUpdatedMillis.set(currUpdatedMillis);
                    lastProcessedMillis.set(clock.millis());
                });
    }

    private CompletableFuture<Void> processList(List<YasmAlertDto> alertList) {
        Map<String, List<YasmAlertDto>> alertsPerItype = alertList.stream()
                .filter(alert -> {
                    if (alert.tags.get("itype") == null) {
                        logger.error("alert without itype: " + alert.name);
                        metrics.invalidAlert();
                        return false;
                    }
                    List<String> itypes = alert.tags.get("itype");
                    if (itypes.size() != 1) {
                        logger.error("alert without single itype: " + alert.name);
                        metrics.invalidAlert();
                        return false;
                    }
                    return true;
                })
                .collect(Collectors.groupingBy(alert -> alert.tags.get("itype").get(0)));

        List<CompletableFuture<Void>> futures = new ArrayList<>();

        for (var entry : alertsPerItype.entrySet()) {
            String itype = entry.getKey();
            var alerts = entry.getValue();

            String project = converter.projectFromItype(itype);

            if (!featureFlags.hasFlag(FeatureFlag.SYNC_AMBRY_ALERTS, project)) {
                logger.info("Skipping alerts in " + project + ", syncing flag is not set");
                continue;
            }

            futures.add(processProject(project, alerts));
        }

        return CompletableFutures.allOfVoid(futures);
    }

    private CompletableFuture<Void> processProject(String project, List<YasmAlertDto> alerts) {
        logger.info("Syncing " + alerts.size() + " alerts in " + project);

        List<TAlert.Builder> fresh = alerts.stream()
                .flatMap(yasmAlert -> {
                    try {
                        TAlert.Builder alert = converter.convertAlert(yasmAlert);
                        metrics.convertedSuccessfully(project);
                        return Stream.of(alert);
                    } catch (Exception e) {
                        logger.error("Failed to convert alert " + yasmAlert.name + " in project " + project + ": " +
                                e.getMessage());
                        metrics.failedToConvert(project);
                        return Stream.empty();
                    }
                })
                .collect(Collectors.toList());

        return listProjectAlerts(project)
                .thenApply(list -> list.stream()
                        .filter(alert -> alert.getName().startsWith(YasmAlertConverter.NAME_PREFIX))
                        .map(TListAlert::getId)
                        .collect(Collectors.toSet())
                )
                .whenComplete((r, t) -> {
                    if (t != null) {
                        logger.error("Fetching existing alerts failed in project " + project, t);
                    } else {
                        logger.info("There are " + r.size() + " existing alerts in project " + project);
                    }
                })
                .thenCompose(existingAlertIds -> syncAlerts(project, existingAlertIds, fresh));
    }

    private CompletableFuture<Pair<String, List<TListAlert>>> listProjectAlertsPage(String project, String token) {
        return alertApi.listAlerts(TListAlertRequest.newBuilder()
                .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                .setProjectId(project)
                .setFilterByName(YasmAlertConverter.NAME_PREFIX)
                .setPageSize(ALERTING_LIST_PAGE_SIZE)
                .setPageToken(token)
                .build())
                .thenApply(response -> {
                    if (response.getRequestStatus() != ERequestStatusCode.OK) {
                        throw new AlertingException(response.getRequestStatus(), response.getStatusMessage());
                    }

                    return Pair.of(response.getNextPageToken(), response.getAlertsList());
                });
    }

    private CompletableFuture<List<TListAlert>> listProjectAlertsRec(String project, List<TListAlert> accumulator, String token) {
        return listProjectAlertsPage(project, token)
                .thenCompose(tokenAndList -> {
                    String nextToken = tokenAndList.getKey();
                    List<TListAlert> page = tokenAndList.getValue();
                    accumulator.addAll(page);
                    if (nextToken.isEmpty() || page.size() < ALERTING_LIST_PAGE_SIZE) {
                        return completedFuture(accumulator);
                    }
                    return listProjectAlertsRec(project, accumulator, nextToken);
                });
    }

    private CompletableFuture<List<TListAlert>> listProjectAlerts(String project) {
        return listProjectAlertsRec(project, new ArrayList<>(), "");
    }

    private CompletableFuture<Void> syncAlerts(String project, Set<String> existingAlertIds, List<TAlert.Builder> fresh) {
        // Resolve alertIds to alerts, some may fail or be hand-modified

        return existingAlertIds.stream()
                .map(alertId -> alertApi.readAlert(TReadAlertRequest.newBuilder()
                        .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                        .setProjectId(project)
                        .setAlertId(alertId)
                        .build())
                        .handle((response, t) -> {
                            if (t != null) {
                                logger.error("Exception while reading alert " + alertId + " in project " + project, t);
                                metrics.readException(project);
                                return null;
                            }
                            if (response.getRequestStatus() != ERequestStatusCode.OK) {
                                logger.error("Failed to read alert " + alertId + " in project " + project +
                                        " with status " + response.getRequestStatus() + ": " +
                                        response.getStatusMessage());
                                metrics.readFailed(project);
                                return null;
                            }
                            if (!response.getAlert().getUpdatedBy().equals(SYNC_USER)) {
                                logger.error("Alert " + alertId + " in project " + project + " has conflicting updates by " +
                                        response.getAlert().getUpdatedBy());
                                metrics.readConflict(project);
                                return null;
                            }
                            return response.getAlert();
                        }))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOf))
                .thenCompose(resolvedExistingAlerts -> syncAlerts(project, existingAlertIds, resolvedExistingAlerts, fresh));
    }

    private CompletableFuture<Void> syncAlerts(
            String project,
            Set<String> existingAlertIds,
            List<TAlert> resolvedExistingAlerts,
            List<TAlert.Builder> fresh)
    {
        /*
         *       existing      fresh
         *     ,----------. ,----------.
         *    /            X            \
         *   /            / \            \
         *  /   delete   /upd\            \
         * (            ( ate )   create   )
         *  \------------\---/            /
         *   \    failed  \ /            /
         *    \  conflict  X            /
         *     `----------' `----------'
         */

        // Steps:
        // 1. Create (fresh \ existing)
        // 2. Delete (resolved \ fresh)
        // 3. Update (resolved * fresh)

        Map<String, TAlert.Builder> freshById = fresh.stream()
                .collect(Collectors.toMap(TAlert.Builder::getId, Function.identity()));
        List<TAlert> resolved = resolvedExistingAlerts.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        final List<Pair<TAlert, TAlert.Builder>> modified = new ArrayList<>();
        final List<TAlert> removed = new ArrayList<>();

        resolved.forEach(alert -> {
            var freshAlert = freshById.get(alert.getId());
            if (freshAlert == null) {
                removed.add(alert);
            } else {
                modified.add(Pair.of(alert, freshAlert));
            }
        });

        final List<TAlert.Builder> added = fresh.stream()
                .filter(alert -> !existingAlertIds.contains(alert.getId()))
                .collect(Collectors.toList());

        logger.info("Project " + project +
                ": delete " + removed.size() + ", create " + added.size() + ", update " + modified.size() + " alerts");

        return CompletableFuture.allOf(delete(project, removed), create(project, added), update(project, modified));
    }

    private CompletableFuture<Void> delete(String project, List<TAlert> removed) {
        return removed.stream()
                .map(alert -> alertApi.deleteAlert(TDeleteAlertRequest.newBuilder()
                        .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                        .setProjectId(project)
                        .setAlertId(alert.getId())
                        .build())
                        .handle((response, t) -> {
                            if (t != null) {
                                logger.error("Exception while deleting alert " + alert.getId() + " in project " + project, t);
                                metrics.deleteException(project);
                                return null;
                            }
                            metrics.deleteStatus(project, response.getRequestStatus());
                            if (response.getRequestStatus() != ERequestStatusCode.OK) {
                                logger.error("Failed to delete alert " + alert.getId() + " in project " + project +
                                        " with status " + response.getRequestStatus() + ": " +
                                        response.getStatusMessage());
                                return null;
                            }
                            return response;
                        }))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }

    private CompletableFuture<Void> create(String project, List<TAlert.Builder> added) {
        return added.stream()
                .map(alert -> alertApi.createAlert(TCreateAlertRequest.newBuilder()
                        .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                        .setAlert(alert
                                .setCreatedBy(SYNC_USER)
                                .setUpdatedBy(SYNC_USER))
                        .build())
                        .handle((response, t) -> {
                            if (t != null) {
                                logger.error("Exception while creating alert " + alert.getId() + " in project " + project, t);
                                metrics.createException(project);
                                return null;
                            }
                            metrics.createStatus(project, response.getRequestStatus());
                            if (response.getRequestStatus() != ERequestStatusCode.OK) {
                                logger.error("Failed to create alert " + alert.getId() + " in project " + project +
                                        " with status " + response.getRequestStatus() + ": " +
                                        response.getStatusMessage());
                                return null;
                            }
                            return response;
                        }))
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }

    private CompletableFuture<Void> update(String project, List<Pair<TAlert, TAlert.Builder>> modified) {
        return modified.stream()
                .map(oldAndNew -> {
                    TAlert old = oldAndNew.getLeft();
                    TAlert.Builder alert = oldAndNew.getRight();
                    return alertApi.updateAlert(TUpdateAlertRequest.newBuilder()
                            .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                            .setAlert(alert
                                    .setVersion(old.getVersion())
                                    .setCreatedBy(SYNC_USER)
                                    .setUpdatedBy(SYNC_USER))
                            .build())
                            .handle((response, t) -> {
                                if (t != null) {
                                    logger.error("Exception while updating alert " + alert.getId() + " in project " + project, t);
                                    metrics.updateException(project);
                                    return null;
                                }
                                metrics.updateStatus(project, response.getRequestStatus());
                                if (response.getRequestStatus() != ERequestStatusCode.OK) {
                                    logger.error("Failed to update alert " + alert.getId() + " in project " + project +
                                            " with status " + response.getRequestStatus() + ": " +
                                            response.getStatusMessage());
                                    return null;
                                }
                                return response;
                            });
                })
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }

    @VisibleForTesting
    public long getLastAmbryUpdatedMillis() {
        return lastAmbryUpdatedMillis.get();
    }

    @Override
    public void close() {
        var future = scheduledFuture;
        if (future != null) {
            future.cancel(true);
        }
    }
}
