package ru.yandex.solomon.alert.ambry;

import java.util.EnumMap;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.selfmon.counters.EnumMetrics;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AmbryPollerMetrics {
    private final MetricRegistry registry;

    private final Rate listUpdated;
    private final Rate invalidAlert;

    private final ConcurrentHashMap<String, ProjectMetrics> perProjectMetrics = new ConcurrentHashMap<>();
    private final ProjectMetrics total;

    public AmbryPollerMetrics(MetricRegistry registry) {
        this.registry = registry;

        listUpdated = registry.rate("ambryPoller.listUpdated");
        invalidAlert = registry.rate("ambryPoller.invalidAlert");

        total = new ProjectMetrics(registry.subRegistry("projectId", "total"));
    }

    public void listUpdated() {
        listUpdated.inc();
    }

    public void invalidAlert() {
        invalidAlert.inc();
    }

    private static class ProjectMetrics {
        private final Rate convertedSuccessfully;
        private final Rate failedToConvert;
        private final Rate readFailed;
        private final Rate readConflict;
        private final Rate readException;

        private final Rate createException;
        private final Rate deleteException;
        private final Rate updateException;

        private final EnumMap<ERequestStatusCode, Rate> createStatus;
        private final EnumMap<ERequestStatusCode, Rate> deleteStatus;
        private final EnumMap<ERequestStatusCode, Rate> updateStatus;

        public ProjectMetrics(MetricRegistry subRegistry) {
            convertedSuccessfully = subRegistry.rate("ambryPoller.convertedSuccessfully");
            failedToConvert = subRegistry.rate("ambryPoller.failedToConvert");

            readFailed = subRegistry.rate("ambryPoller.readFailed");
            readConflict = subRegistry.rate("ambryPoller.readConflict");
            readException = subRegistry.rate("ambryPoller.readException");

            createException = subRegistry.rate("ambryPoller.createException");
            deleteException = subRegistry.rate("ambryPoller.deleteException");
            updateException = subRegistry.rate("ambryPoller.updateException");

            createStatus = EnumMetrics.rates(ERequestStatusCode.class, subRegistry, "ambryPoller.createStatus", "status");
            deleteStatus = EnumMetrics.rates(ERequestStatusCode.class, subRegistry, "ambryPoller.deleteStatus", "status");
            updateStatus = EnumMetrics.rates(ERequestStatusCode.class, subRegistry, "ambryPoller.updateStatus", "status");
        }
    }

    private ProjectMetrics getProjectMetrics(String project) {
        var projectMetrics = perProjectMetrics.get(project);
        if (projectMetrics != null) {
            return projectMetrics;
        }
        return perProjectMetrics.computeIfAbsent(project, ignore ->
                new ProjectMetrics(registry.subRegistry("projectId", project)));
    }

    public void convertedSuccessfully(String project) {
        getProjectMetrics(project).convertedSuccessfully.inc();
        total.convertedSuccessfully.inc();
    }

    public void failedToConvert(String project) {
        getProjectMetrics(project).failedToConvert.inc();
        total.failedToConvert.inc();
    }

    public void readFailed(String project) {
        getProjectMetrics(project).readFailed.inc();
        total.readFailed.inc();
    }

    public void readConflict(String project) {
        getProjectMetrics(project).readConflict.inc();
        total.readConflict.inc();
    }

    public void readException(String project) {
        getProjectMetrics(project).readException.inc();
        total.readException.inc();
    }

    public void createException(String project) {
        getProjectMetrics(project).createException.inc();
        total.createException.inc();
    }

    public void deleteException(String project) {
        getProjectMetrics(project).deleteException.inc();
        total.deleteException.inc();
    }

    public void updateException(String project) {
        getProjectMetrics(project).updateException.inc();
        total.updateException.inc();
    }

    public void updateStatus(String project, ERequestStatusCode status) {
        getProjectMetrics(project).updateStatus.get(status).inc();
        total.updateStatus.get(status).inc();
    }

    public void createStatus(String project, ERequestStatusCode status) {
        getProjectMetrics(project).createStatus.get(status).inc();
        total.createStatus.get(status).inc();
    }

    public void deleteStatus(String project, ERequestStatusCode status) {
        getProjectMetrics(project).deleteStatus.get(status).inc();
        total.deleteStatus.get(status).inc();
    }
}
