package ru.yandex.solomon.alert.api;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.api.validators.ValidationException;
import ru.yandex.solomon.alert.dao.ConcurrentAlertModificationException;
import ru.yandex.solomon.alert.dao.DuplicateKeyException;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.labels.query.SelectorsException;

/**
 * @author Vladimir Gordiychuk
 */
public final class ErrorClassificatory {
    private static final Logger logger = LoggerFactory.getLogger(ErrorClassificatory.class);

    private ErrorClassificatory() {
    }

    public static ERequestStatusCode classifyError(Throwable throwable) {
        Throwable cause = CompletableFutures.unwrapCompletionException(throwable);

        if (cause instanceof ValidationException) {
            return ERequestStatusCode.INVALID_REQUEST;
        }

        if (cause instanceof SelectorsException) {
            return ERequestStatusCode.INVALID_REQUEST;
        }

        if (cause instanceof DuplicateKeyException) {
            return ERequestStatusCode.INVALID_REQUEST;
        }

        if (cause instanceof ConcurrentAlertModificationException) {
            return ERequestStatusCode.CONCURRENT_MODIFICATION;
        }

        if (cause instanceof AlertingException alertingException) {
            return alertingException.getCode();
        }

        if (cause instanceof StatusRuntimeException statusRuntimeException) {
            return classifyGrpcStatus(statusRuntimeException.getStatus());
        }

        return ERequestStatusCode.INTERNAL_ERROR;
    }

    private static ERequestStatusCode classifyGrpcStatus(Status status) {
        return switch (status.getCode()) {
            case OK -> ERequestStatusCode.OK;
            case DEADLINE_EXCEEDED -> ERequestStatusCode.DEADLINE_EXCEEDED;
            case INVALID_ARGUMENT, ALREADY_EXISTS -> ERequestStatusCode.INVALID_REQUEST;
            case NOT_FOUND -> ERequestStatusCode.NOT_FOUND;
            case RESOURCE_EXHAUSTED -> ERequestStatusCode.RESOURCE_EXHAUSTED;
            case UNAVAILABLE -> ERequestStatusCode.NODE_UNAVAILABLE;
            default -> ERequestStatusCode.INTERNAL_ERROR;
        };
    }

    public static <T> T throwExceptionAsGrpc(Throwable throwable) {
        throw convertExceptionToGrpc(throwable);
    }

    public static StatusRuntimeException convertExceptionToGrpc(Throwable throwable) {
        Throwable cause = CompletableFutures.unwrapCompletionException(throwable);

        if (cause instanceof StatusRuntimeException sre) {
            return sre;
        }

        if (cause instanceof ValidationException || cause instanceof DuplicateKeyException) {
            return Status.INVALID_ARGUMENT.withDescription(cause.getMessage()).asRuntimeException();
        }

        if (cause instanceof ConcurrentAlertModificationException) {
            return Status.FAILED_PRECONDITION.withDescription(cause.getMessage()).asRuntimeException();
        }

        if (cause instanceof AlertingException alertingException) {
            return alertingExceptionToGrpc(alertingException);
        }

        logger.error("Unexpected exception type", cause);

        return Status.INTERNAL.withCause(cause).asRuntimeException();
    }

    private static StatusRuntimeException alertingExceptionToGrpc(AlertingException alertingException) {
        Status status = switch (alertingException.getCode()) {
            case OK -> Status.OK;
            case INVALID_REQUEST -> Status.INVALID_ARGUMENT;
            case NOT_FOUND -> Status.NOT_FOUND;
            case RESOURCE_EXHAUSTED -> Status.RESOURCE_EXHAUSTED;
            case DEADLINE_EXCEEDED -> Status.DEADLINE_EXCEEDED;
            case NODE_UNAVAILABLE, SHARD_NOT_INITIALIZED -> Status.UNAVAILABLE;
            case CONCURRENT_MODIFICATION -> Status.FAILED_PRECONDITION;
            case NOT_AUTHORIZED -> Status.PERMISSION_DENIED;
            default -> Status.INTERNAL;
        };

        return status.withDescription(alertingException.getMessage()).asRuntimeException();
    }
}
