package ru.yandex.solomon.alert.api.converters;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Interner;
import com.google.protobuf.util.Timestamps;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AbstractAlertBuilder;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertInterner;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.AlertingLabelsAllocator;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.domain.NoPointsPolicy;
import ru.yandex.solomon.alert.domain.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.domain.StringInterner;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.threshold.Compare;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.domain.threshold.ThresholdType;
import ru.yandex.solomon.alert.protobuf.AlertFromTemplate;
import ru.yandex.solomon.alert.protobuf.AlertParameter;
import ru.yandex.solomon.alert.protobuf.AlertTemplateStatus;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.Expression;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TChannelConfig;
import ru.yandex.solomon.alert.protobuf.TDefaultChannelConfig;
import ru.yandex.solomon.alert.protobuf.TEvaluationState;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TExpression;
import ru.yandex.solomon.alert.protobuf.TNotificationChannelOptions;
import ru.yandex.solomon.alert.protobuf.TSubAlert;
import ru.yandex.solomon.alert.protobuf.TThreshold;
import ru.yandex.solomon.alert.protobuf.Threshold;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.alert.template.domain.AbstractAlertTemplateBuilder;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.alert.template.domain.expression.ExpressionAlertTemplate;
import ru.yandex.solomon.alert.template.domain.threshold.TemplatePredicateRule;
import ru.yandex.solomon.alert.template.domain.threshold.ThresholdAlertTemplate;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
public final class AlertConverter {
    private AlertConverter() {
    }

    public static Alert protoToAlert(TAlert proto) {
        return protoToAlertWithPrevState(proto, null);
    }

    public static Alert protoToAlertWithPrevState(TAlert proto, Alert prevState) {
        final AbstractAlertBuilder<?, ?> builder = switch (proto.getTypeCase()) {
            case THRESHOLD -> protoToThreshold(proto.getThreshold());
            case EXPRESSION -> protoToExpression(proto.getExpression());
            case ALERT_FROM_TEMPLATE -> protoToAlertFromTemplate(proto, prevState);
            default -> throw new UnsupportedOperationException("Unsupported alert type: " + proto);
        };

        builder
                .setId("".equals(proto.getId()) ? UUID.randomUUID().toString() : proto.getId())
                .setProjectId(proto.getProjectId())
                .setFolderId(proto.getFolderId())
                .setName(proto.getName())
                .setDescription(proto.getDescription())
                .setVersion(proto.getVersion())
                .setCreatedBy(proto.getCreatedBy())
                .setCreatedAt(proto.getCreatedAt())
                .setUpdatedBy(proto.getUpdatedBy())
                .setUpdatedAt(proto.getUpdatedAt())
                .setState(AlertState.valueOf(proto.getState().name()))
                .setGroupByLabels(proto.getGroupByLabelsList())
                .setAnnotations(proto.getAnnotationsMap())
                .setPeriod(Duration.ofMillis(proto.getPeriodMillis()))
                .setDelaySeconds(proto.getDelaySeconds())
                .setResolvedEmptyPolicy(ResolvedEmptyPolicy.forNumber(proto.getResolvedEmptyPolicy().getNumber()))
                .setNoPointsPolicy(NoPointsPolicy.forNumber(proto.getNoPointsPolicy().getNumber()))
                .setSeverity(AlertSeverity.forNumber(proto.getSeverity().getNumber()))
                .setLabels(proto.getLabelsMap())
                .setObtainedFromTemplate(proto.getObtainedFromTemplate())
                .setServiceProviderAnnotations(proto.getServiceProviderAnnotationsMap())
                .setEscalations(new HashSet<>(proto.getEscalationsList()));


        if (proto.getConfiguredNotificationChannelsCount() > 0) {
            builder.setNotificationChannels(proto.getConfiguredNotificationChannelsMap().entrySet().stream()
                    .collect(Collectors.toUnmodifiableMap(
                        Map.Entry::getKey,
                        e -> protoToChannelConfig(e.getValue())
                    )));
        } else {
            builder.setNotificationChannels(proto.getNotificationChannelIdsList());
        }

        return builder.build();
    }

    public static Alert protoToAlert(TAlert proto, Interner<Alert> interner) {
        return interner.intern(protoToAlert(proto));
    }

    private static ThresholdAlert.Builder protoToThreshold(TThreshold proto) {
        ThresholdAlert.Builder builder = ThresholdAlert.newBuilder()
                .setTransformations(proto.getTransformations())
                .setSelectors(LabelSelectorConverter.protoToSelectors(proto.getNewSelectors()))
                .setPredicateRules(Stream.of(PredicateRule
                        .onThreshold(proto.getThreshold())
                        .withThresholdType(ThresholdType.forNumber(proto.getThresholdTypeValue()))
                        .withComparison(Compare.forNumber(proto.getComparisonValue()))));

        if (proto.getPredicateRulesCount() > 0) {
            builder.setPredicateRules(proto.getPredicateRulesList().stream().map(PredicateRule::fromProto));
        }

        return builder;
    }

    private static ExpressionAlert.Builder protoToExpression(TExpression proto) {
        return ExpressionAlert.newBuilder()
                .setProgram(proto.getProgram())
                .setCheckExpression(proto.getCheckExpression());
    }

    private static AlertFromTemplatePersistent.Builder protoToAlertFromTemplate(TAlert alert, Alert prevState) {
        AlertFromTemplate proto = alert.getAlertFromTemplate();
        var serviceProvider = alert.getServiceProvider();
        if (StringUtils.isEmpty(serviceProvider) && prevState instanceof AlertFromTemplatePersistent alertPrev) {
            serviceProvider = alertPrev.getServiceProvider();
        }
        return AlertFromTemplatePersistent.newBuilder()
                .setTemplateId(proto.getTemplateId())
                .setTemplateVersionTag(proto.getTemplateVersionTag())
                .setServiceProvider(serviceProvider)
                .setParameters(proto.getAlertParametersList().stream()
                        .map(AlertConverter::protoToParameter)
                        .collect(Collectors.toList()))
                .setThresholds(proto.getAlertThresholdsList().stream()
                        .map(AlertConverter::protoToParameter)
                        .collect(Collectors.toList()));
    }

    public static ru.yandex.solomon.alert.domain.template.AlertParameter protoToParameter(AlertParameter alertParameter) {
        return switch (alertParameter.getParameterCase()) {
            case DOUBLE_PARAMETER_VALUE -> new ru.yandex.solomon.alert.domain.template.AlertParameter.DoubleParameterValue(
                    alertParameter.getDoubleParameterValue().getValue(),
                    alertParameter.getDoubleParameterValue().getName());
            case INTEGER_PARAMETER_VALUE -> new ru.yandex.solomon.alert.domain.template.AlertParameter.IntegerParameterValue(
                    (int) alertParameter.getIntegerParameterValue().getValue(),
                    alertParameter.getIntegerParameterValue().getName());
            case TEXT_PARAMETER_VALUE -> new ru.yandex.solomon.alert.domain.template.AlertParameter.TextParameterValue(
                    alertParameter.getTextParameterValue().getValue(),
                    alertParameter.getTextParameterValue().getName());
            case TEXT_LIST_PARAMETER_VALUE -> new ru.yandex.solomon.alert.domain.template.AlertParameter.TextListParameterValue(
                    alertParameter.getTextListParameterValue().getValuesList(),
                    alertParameter.getTextListParameterValue().getName());
            case LABEL_LIST_PARAMETER_VALUE -> new ru.yandex.solomon.alert.domain.template.AlertParameter.LabelListParameterValue(
                    alertParameter.getLabelListParameterValue().getValuesList(),
                    alertParameter.getLabelListParameterValue().getName());
            case PARAMETER_NOT_SET -> throw new UnsupportedOperationException("Unsupported parameter type: " + alertParameter.getParameterCase());
        };
    }

    public static ru.yandex.solomon.alert.protobuf.TAlert alertToProto(Alert alert) {
        final ru.yandex.solomon.alert.protobuf.TAlert.Builder builder =
                ru.yandex.solomon.alert.protobuf.TAlert.newBuilder();
        switch (alert.getAlertType()) {
            case THRESHOLD -> builder.setThreshold(thresholdToProto((ThresholdAlert) alert));
            case EXPRESSION -> builder.setExpression(expressionToProto((ExpressionAlert) alert));
            case FROM_TEMPLATE -> builder.setAlertFromTemplate(alertFromTemplateToProto((AlertFromTemplatePersistent) alert))
                    .setServiceProvider(((AlertFromTemplatePersistent) alert).getServiceProvider())
                    .putAllServiceProviderLabels(((AlertFromTemplatePersistent) alert).getServiceProviderLabels());
            default -> throw new UnsupportedOperationException("Unsupported alert type: " + alert);
        }

        return builder
                .setId(alert.getId())
                .setProjectId(alert.getProjectId())
                .setFolderId(alert.getFolderId())
                .setName(alert.getName())
                .setDescription(alert.getDescription())
                .setCreatedBy(alert.getCreatedBy())
                .setCreatedAt(alert.getCreatedAt())
                .setUpdatedBy(alert.getUpdatedBy())
                .setUpdatedAt(alert.getUpdatedAt())
                .setState(ru.yandex.solomon.alert.protobuf.EAlertState.valueOf(alert.getState().name()))
                .setVersion(alert.getVersion())
                .addAllNotificationChannelIds(alert.getNotificationChannels().keySet().stream()
                    .sorted()
                    .collect(Collectors.toList()))
                .putAllConfiguredNotificationChannels(alert.getNotificationChannels().entrySet().stream()
                        .collect(Collectors.toUnmodifiableMap(
                            Map.Entry::getKey,
                            e -> channelConfigToProto(e.getValue())
                        )))
                .addAllGroupByLabels(alert.getGroupByLabels())
                .putAllAnnotations(alert.getAnnotations())
                .putAllServiceProviderAnnotations(alert.getServiceProviderAnnotations())
                .addAllEscalations(alert.getEscalations())
                .setPeriodMillis(alert.getPeriod().toMillis())
                .setDelaySeconds(alert.getDelaySeconds())
                .setResolvedEmptyPolicyValue(alert.getResolvedEmptyPolicy().getNumber())
                .setNoPointsPolicyValue(alert.getNoPointsPolicy().getNumber())
                .setSeverityValue(alert.getSeverity().getNumber())
                .putAllLabels(alert.getLabels())
                .setObtainedFromTemplate(alert.isObtainedFromTemplate())
                .build();
    }

    public static TNotificationChannelOptions channelConfigToProto(ChannelConfig config) {
        if (config == ChannelConfig.EMPTY) {
            return TNotificationChannelOptions.newBuilder()
                .setDefaultChannelConfig(TDefaultChannelConfig.newBuilder())
                .build();
        }
        return TNotificationChannelOptions.newBuilder()
            .setChannelConfig(TChannelConfig.newBuilder()
                .setRepeatNotifyDelayMillis(config.getRepeatNotificationDelayOrDefault(Duration.ZERO).toMillis())
                .addAllNotifyAboutStatuses(config.getNotifyAboutStatusesOrDefault(Set.of()).stream()
                        .map(code -> TEvaluationStatus.ECode.forNumber(code.getNumber()))
                        .collect(Collectors.toList())))
            .build();
    }

    private static ChannelConfig protoToChannelConfig(TNotificationChannelOptions options) {
        return switch (options.getOptionsCase()) {
            case DEFAULTCHANNELCONFIG -> ChannelConfig.EMPTY;
            case CHANNELCONFIG -> new ChannelConfig(
                    options.getChannelConfig().getNotifyAboutStatusesList().stream()
                            .map(ecode -> EvaluationStatus.Code.forNumber(ecode.getNumber()))
                            .collect(Collectors.toUnmodifiableSet()),
                    Duration.ofMillis(options.getChannelConfig().getRepeatNotifyDelayMillis())
            );
            default -> throw new UnsupportedOperationException("Unsupported channel options: " + options);
        };
    }

    public static TSubAlert subAlertToProto(SubAlert alert) {
        return TSubAlert.newBuilder()
                .setId(alert.getId())
                .setProjectId(alert.getProjectId())
                .setParent(alertToProto(alert.getParent()))
                .addAllGroupKey(LabelConverter.labelsToProtoList(alert.getGroupKey()))
                .build();
    }

    public static SubAlert protoToSubAlert(TSubAlert proto) {
        return protoToSubAlert(proto, AlertInterner.NOOP);
    }

    public static SubAlert protoToSubAlert(TSubAlert proto, Interner<Alert> interner) {
        return SubAlert.newBuilder()
            .setParent(protoToAlert(proto.getParent(), interner))
            .setGroupKey(LabelConverter.protoToLabels(proto.getGroupKeyList(), AlertingLabelsAllocator.I))
            .setProjectId(proto.getProjectId())
            .setId(proto.getId())
            .build();
    }

    public static AlertType protoToType(EAlertType proto) {
        return AlertType.values()[proto.getNumber()];
    }

    public static EAlertType typeToProto(AlertType type) {
        return EAlertType.forNumber(type.ordinal());
    }

    public static EnumSet<AlertType> protosToTypes(List<EAlertType> protos) {
        EnumSet<AlertType> result = EnumSet.noneOf(AlertType.class);
        for (EAlertType proto : protos) {
            result.add(protoToType(proto));
        }
        return result;
    }

    public static AlertState protoToAlertState(EAlertState proto) {
        return AlertState.values()[proto.getNumber()];
    }

    public static EAlertState alertStateToProto(AlertState state) {
        return EAlertState.forNumber(state.ordinal());
    }

    public static EnumSet<AlertState> protosToAlertState(List<EAlertState> protos) {
        EnumSet<AlertState> result = EnumSet.noneOf(AlertState.class);
        for (EAlertState proto : protos) {
            result.add(protoToAlertState(proto));
        }
        return result;
    }

    private static TThreshold thresholdToProto(ThresholdAlert alert) {
        TThreshold.Builder builder = TThreshold.newBuilder()
                .setNewSelectors(LabelSelectorConverter.selectorsToNewProto(alert.getSelectors()))
                .setTransformations(alert.getTransformations());

        PredicateRule predicateRule = alert.getPredicateRule();

        builder
                .setThreshold(predicateRule.getThreshold())
                .setThresholdType(EThresholdType.forNumber(predicateRule.getThresholdType().getNumber()))
                .setComparison(ECompare.forNumber(predicateRule.getComparison().getNumber()));

        alert.getPredicateRules().stream()
                .map(PredicateRule::toProto)
                .forEach(builder::addPredicateRules);

        return builder.build();
    }

    private static TExpression expressionToProto(ExpressionAlert alert) {
        return TExpression.newBuilder()
                .setProgram(alert.getProgram())
                .setCheckExpression(alert.getCheckExpression())
                .build();
    }

    private static AlertFromTemplate alertFromTemplateToProto(AlertFromTemplatePersistent alert) {
        return AlertFromTemplate.newBuilder()
                .setTemplateId(alert.getTemplateId())
                .setTemplateVersionTag(alert.getTemplateVersionTag())
                .addAllAlertParameters(alert.getParameters().stream()
                        .map(ru.yandex.solomon.alert.domain.template.AlertParameter::toProto)
                        .collect(Collectors.toList()))
                .addAllAlertThresholds(alert.getThresholds().stream()
                        .map(ru.yandex.solomon.alert.domain.template.AlertParameter::toProto)
                        .collect(Collectors.toList()))
                .build();
    }

    public static TEvaluationState stateToProto(@Nullable EvaluationState state) {
        if (state == null) {
            return TEvaluationState.getDefaultInstance();
        }

        return TEvaluationState.newBuilder()
                .setAlertId(state.getKey().getAlertId())
                .setProjectId(state.getKey().getProjectId())
                .setParentId(state.getKey().getParentId())
                .setAlertVersion(state.getAlertVersion())
                .setStatus(statusToProto(state.getStatus()))
                .setSinceMillis(state.getSince().toEpochMilli())
                .setLatestEvalMillis(state.getLatestEval().toEpochMilli())
                .setPreviousStatus(statusToProto(state.getPreviousStatus()))
                .build();
    }

    @Nullable
    public static EvaluationState protoToState(TEvaluationState proto) {
        if (Objects.equals(TEvaluationState.getDefaultInstance(), proto)) {
            return null;
        }

        AlertKey key = new AlertKey(
            StringInterner.I.intern(proto.getProjectId()),
            StringInterner.I.intern(proto.getParentId()),
            StringInterner.I.intern(proto.getAlertId())
        );

        return EvaluationState.newBuilder()
                .setAlertKey(key)
                .setAlertVersion(proto.getAlertVersion())
                .setStatus(protoToStatus(proto.getStatus()))
                .setSince(Instant.ofEpochMilli(proto.getSinceMillis()))
                .setLatestEval(Instant.ofEpochMilli(proto.getLatestEvalMillis()))
                .setPreviousStatus(protoToStatus(proto.getPreviousStatus()))
                .build();
    }

    public static TEvaluationStatus statusToProto(EvaluationStatus status) {
        return TEvaluationStatus.newBuilder()
                .setCode(statusCodeToProto(status.getCode()))
                .setDesctiption(status.getDescription())
                .putAllAnnotations(status.getAnnotations())
                .putAllServiceProviderAnnotations(status.getServiceProviderAnnotations())
                .addAllScalarValues(status.getScalarValues().stream()
                        .map(AlertConverter::toProto)
                        .collect(Collectors.toList()))
                .setErrorCode(TEvaluationStatus.EErrorCode.forNumber(status.getErrorCode().getNumber()))
                .build();
    }

    private static TEvaluationStatus.ScalarValue toProto(EvaluationStatus.ScalarValue scalarValue) {
        var builder = TEvaluationStatus.ScalarValue.newBuilder()
                .setName(scalarValue.name());
        switch (scalarValue.type()) {
            case STRING -> builder.setStringValue(Nullables.orEmpty((String) scalarValue.value()));
            case DOUBLE -> builder.setDoubleValue(Nullables.orZero((Double) scalarValue.value()));
            case BOOLEAN -> builder.setBoolValue(Nullables.orFalse((Boolean) scalarValue.value()));
        }
        return builder.build();
    }

    public static TEvaluationStatus.ECode statusCodeToProto(EvaluationStatus.Code code) {
        return TEvaluationStatus.ECode.forNumber(code.getNumber());
    }

    public static ru.yandex.solomon.alert.protobuf.AlertMuteStatus muteStatusToProto(@Nullable AlertMuteStatus muteStatus) {
        if (muteStatus == null) {
            return ru.yandex.solomon.alert.protobuf.AlertMuteStatus.getDefaultInstance();
        }

        var builder = ru.yandex.solomon.alert.protobuf.AlertMuteStatus.newBuilder()
                .setCode(MuteConverter.alertMuteStatusCodeToProto(muteStatus.statusCode()));
        muteStatus.affectingMutes().forEach(affectingMute -> {
            builder.addAffectingMutes(MuteConverter.affectingMuteToProto(affectingMute));
        });
        return builder.build();
    }

    public static EvaluationStatus protoToStatus(TEvaluationStatus proto) {
        var annotations = ImmutableMap.<String, String>builderWithExpectedSize(proto.getAnnotationsCount());
        for (var entry : proto.getAnnotationsMap().entrySet()) {
            annotations.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        var serviceProviderAnnotations = ImmutableMap.<String, String>builderWithExpectedSize(proto.getServiceProviderAnnotationsCount());
        for (var entry : proto.getServiceProviderAnnotationsMap().entrySet()) {
            serviceProviderAnnotations.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        return protoToStatusCode(proto.getCode())
                .toStatus(annotations.build(), serviceProviderAnnotations.build())
                .withErrorCode(EvaluationStatus.ErrorCode.forNumber(proto.getErrorCodeValue()));
    }

    public static EvaluationStatus.Code protoToStatusCode(TEvaluationStatus.ECode proto) {
        return EvaluationStatus.Code.forNumber(proto.getNumber());
    }

    public static AlertMuteStatus protoToMuteStatus(ru.yandex.solomon.alert.protobuf.AlertMuteStatus proto) {
        return new AlertMuteStatus(
                MuteConverter.protoToAlertMuteStatusCode(proto.getCode()),
                proto.getAffectingMutesList().stream()
                        .map(MuteConverter::protoToAffectingMute)
                        .collect(Collectors.toUnmodifiableList()));
    }

    public static ru.yandex.solomon.alert.protobuf.AlertTemplate alertTemplateToProto(AlertTemplate alertTemplate, AlertTemplateStatus status) {
        final ru.yandex.solomon.alert.protobuf.AlertTemplate.Builder builder =
                ru.yandex.solomon.alert.protobuf.AlertTemplate.newBuilder();
        switch (alertTemplate.getAlertTemplateType()) {
            case THRESHOLD -> builder.setThreshold(thresholdToProto((ThresholdAlertTemplate) alertTemplate));
            case EXPRESSION -> builder.setExpression(expressionToProto((ExpressionAlertTemplate) alertTemplate));
            default -> throw new UnsupportedOperationException("Unsupported alert template type: " + alertTemplate);
        }
        return builder
                .setId(alertTemplate.getId())
                .setTemplateVersionTag(alertTemplate.getTemplateVersionTag())
                .setServiceProviderId(alertTemplate.getServiceProviderId())
                .setName(alertTemplate.getName())
                .setDescription(alertTemplate.getDescription())
                .setCreatedAt(Timestamps.fromMillis(alertTemplate.getCreatedAt().toEpochMilli()))
                .setModifiedAt(Timestamps.fromMillis(alertTemplate.getUpdatedAt().toEpochMilli()))
                .setCreatedBy(alertTemplate.getCreatedBy())
                .setModifiedBy(alertTemplate.getUpdatedBy())
                .addAllGroupByLabels(alertTemplate.getGroupByLabels())
                .putAllAnnotations(alertTemplate.getAnnotations())
                .setPeriodMillis(alertTemplate.getPeriodMillis())
                .setDelaySeconds(alertTemplate.getDelaySeconds())
                .setResolvedEmptyPolicyValue(alertTemplate.getResolvedEmptyPolicy().getNumber())
                .setNoPointsPolicyValue(alertTemplate.getNoPointsPolicy().getNumber())
                .setSeverityValue(alertTemplate.getSeverity().getNumber())
                .putAllLabels(alertTemplate.getLabels())
                .setAlertTemplateStatus(status)
                .setDefault(alertTemplate.isDefaultTemplate())
                .addAllAlertTemplateParameters(alertTemplate.getParameters()
                        .stream().map(AlertTemplateParameter::toProto)
                        .collect(Collectors.toList()))
                .addAllAlertTemplateThresholds(alertTemplate.getThresholds()
                        .stream().map(AlertTemplateParameter::toProto)
                        .collect(Collectors.toList()))
                .build();
    }

    private static Threshold thresholdToProto(ThresholdAlertTemplate template) {
        Threshold.Builder builder = Threshold.newBuilder()
                .setSelectors(template.getSelectors())
                .setTransformations(template.getTransformations());

        template.getPredicateRules().stream()
                .map(TemplatePredicateRule::toProto)
                .forEach(builder::addPredicateRules);

        return builder.build();
    }

    private static Expression expressionToProto(ExpressionAlertTemplate template) {
        return Expression.newBuilder()
                .setProgram(template.getProgram())
                .build();
    }

    public static AlertTemplate alertTemplateFromProto(ru.yandex.solomon.alert.protobuf.AlertTemplate alertTemplate) {
        AbstractAlertTemplateBuilder<?, ?> builder;
        switch (alertTemplate.getTypeCase()) {
            case THRESHOLD ->  builder = thresholdFromProto(alertTemplate);
            case EXPRESSION -> builder = expressionFromProto(alertTemplate);
            default -> throw new UnsupportedOperationException("Unsupported alert template type: " + alertTemplate);
        }
        return builder
                .setId(alertTemplate.getId())
                .setTemplateVersionTag(alertTemplate.getTemplateVersionTag())
                .setServiceProviderId(alertTemplate.getServiceProviderId())
                .setName(alertTemplate.getName())
                .setDescription(alertTemplate.getDescription())
                .setCreatedAt(Instant.ofEpochSecond(alertTemplate.getCreatedAt().getSeconds(), alertTemplate.getCreatedAt().getNanos()))
                .setUpdatedAt(Instant.ofEpochSecond(alertTemplate.getModifiedAt().getSeconds(), alertTemplate.getModifiedAt().getNanos()))
                .setCreatedBy(alertTemplate.getCreatedBy())
                .setUpdatedBy(alertTemplate.getModifiedBy())
                .setGroupByLabels(alertTemplate.getGroupByLabelsList())
                .setAnnotations(alertTemplate.getAnnotationsMap())
                .setPeriodMillis((int) alertTemplate.getPeriodMillis())
                .setDelaySeconds(alertTemplate.getDelaySeconds())
                .setResolvedEmptyPolicy(ResolvedEmptyPolicy.forNumber(alertTemplate.getResolvedEmptyPolicy().getNumber()))
                .setNoPointsPolicy(NoPointsPolicy.forNumber(alertTemplate.getNoPointsPolicy().getNumber()))
                .setAlertSeverity(AlertSeverity.forNumber(alertTemplate.getSeverity().getNumber()))
                .setLabels(alertTemplate.getLabelsMap())
                .setDefaultTemplate(alertTemplate.getDefault())
                .setParameters(alertTemplate.getAlertTemplateParametersList()
                        .stream().map(AlertTemplateParameter::fromProto)
                        .collect(Collectors.toList()))
                .setThresholds(alertTemplate.getAlertTemplateThresholdsList()
                        .stream().map(AlertTemplateParameter::fromProto)
                        .collect(Collectors.toList()))
                .build();
    }

    private static AbstractAlertTemplateBuilder<?,?> thresholdFromProto(ru.yandex.solomon.alert.protobuf.AlertTemplate template) {
        return ThresholdAlertTemplate.newBuilder()
                .setSelectors(template.getThreshold().getSelectors())
                .setTransformations(template.getThreshold().getTransformations())
                .setPredicateRules(template.getThreshold().getPredicateRulesList().stream()
                        .map(TemplatePredicateRule::fromProto));
    }

    private static AbstractAlertTemplateBuilder<?,?> expressionFromProto(ru.yandex.solomon.alert.protobuf.AlertTemplate alertTemplate) {
        return ExpressionAlertTemplate.newBuilder()
                .setProgram(alertTemplate.getExpression().getProgram());
    }

    public static List<AlertFromTemplatePersistent> protoToAlerts(long now, CreateAlertsFromTemplateRequest request) {
        var result = new ArrayList<AlertFromTemplatePersistent>(request.getTemplateIdsCount() * request.getResourcesCount());
        var templateConfigs = request.getTemplateConfigsList();
        var defaultChannels = request.getChannelsMap();
        var escalations = request.getEscalationsList();
        if (!templateConfigs.isEmpty()) {
            for (CreateAlertsFromTemplateRequest.TemplateConfigs templateConfig : templateConfigs) {
                String templateId = templateConfig.getTemplateId();
                var priorityChannels = templateConfig.getChannelsMap();
                var chosenChannels = (!priorityChannels.isEmpty() ? priorityChannels : defaultChannels);
                var thresholds = templateConfig.getAlertThresholdsList();
                var chosenEscalations = templateConfig.getEscalationsList().isEmpty()
                        ? escalations
                        : templateConfig.getEscalationsList();
                for (CreateAlertsFromTemplateRequest.Resource resourceId : request.getResourcesList()) {
                    addAlert(result, templateId, request, resourceId, chosenChannels, now, thresholds, request.getServiceProviderId(), chosenEscalations);
                }
            }
        } else {
            for (String templateId : request.getTemplateIdsList()) {
                for (CreateAlertsFromTemplateRequest.Resource resourceId : request.getResourcesList()) {
                    addAlert(result, templateId, request, resourceId, defaultChannels, now, List.of(), "", escalations);
                }
            }
        }
        return result;
    }

    private static void addAlert(
            ArrayList<AlertFromTemplatePersistent> result,
            String templateId,
            CreateAlertsFromTemplateRequest request,
            CreateAlertsFromTemplateRequest.Resource resourceId,
            Map<String, TNotificationChannelOptions> chosenChannels,
            long now,
            List<AlertParameter> thresholds,
            String serviceProviderId,
            List<String> escalations)
    {
        var builder = AlertFromTemplatePersistent.newBuilder()
                .setTemplateId(templateId)
                .setTemplateVersionTag("")
                .setName("")
                .setDescription("")
                .setId(UUID.randomUUID().toString())
                .setProjectId(request.getProjectId())
                .setState(AlertState.ACTIVE)
                .setLabels(resourceId.getResourceParametersMap())
                .setNotificationChannels(chosenChannels.entrySet().stream()
                        .collect(Collectors.toUnmodifiableMap(
                                Map.Entry::getKey,
                                e -> protoToChannelConfig(e.getValue())
                        )))
                .setEscalations(new HashSet<>(escalations))
                .setServiceProvider(serviceProviderId)
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setCreatedBy(request.getCreatedBy())
                .setUpdatedBy(request.getCreatedBy())
                .setVersion(1);
        if (thresholds != null && !thresholds.isEmpty()) {
            builder.setThresholds(formatAlertParameters(thresholds));
        }
        var alert = builder.build();
        result.add(alert);
    }

    private static List<ru.yandex.solomon.alert.domain.template.AlertParameter> formatAlertParameters(List<AlertParameter> thresholds) {
        List<ru.yandex.solomon.alert.domain.template.AlertParameter> result = new ArrayList<>();
        for (var threshold : thresholds) {
            switch (threshold.getParameterCase()) {
                case DOUBLE_PARAMETER_VALUE -> result.add(
                        new ru.yandex.solomon.alert.domain.template.AlertParameter.DoubleParameterValue(
                                threshold.getDoubleParameterValue().getValue(),
                                threshold.getDoubleParameterValue().getName())
                );
                case INTEGER_PARAMETER_VALUE -> result.add(
                        new ru.yandex.solomon.alert.domain.template.AlertParameter.IntegerParameterValue(
                                (int) threshold.getIntegerParameterValue().getValue(),
                                threshold.getIntegerParameterValue().getName()
                        )
                );
                case TEXT_PARAMETER_VALUE -> result.add(
                        new ru.yandex.solomon.alert.domain.template.AlertParameter.TextParameterValue(
                                threshold.getTextParameterValue().getValue(),
                                threshold.getTextParameterValue().getName()
                        )
                );
                case TEXT_LIST_PARAMETER_VALUE -> result.add(
                        new ru.yandex.solomon.alert.domain.template.AlertParameter.TextListParameterValue(
                                threshold.getTextListParameterValue().getValuesList(),
                                threshold.getTextListParameterValue().getName()
                        )
                );
                case LABEL_LIST_PARAMETER_VALUE -> result.add(
                        new ru.yandex.solomon.alert.domain.template.AlertParameter.LabelListParameterValue(
                                threshold.getLabelListParameterValue().getValuesList(),
                                threshold.getLabelListParameterValue().getName()
                        )
                );
            }
        }
        return result;
    }
}
