package ru.yandex.solomon.alert.api.converters;

import java.time.Instant;
import java.util.UUID;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.mute.domain.AffectingMute;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.mute.domain.MuteStatus;
import ru.yandex.solomon.alert.mute.domain.SelectorsMute;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class MuteConverter {
    public static final MuteConverter INSTANCE = new MuteConverter();

    private MuteConverter() {
    }

    public Mute protoToMute(ru.yandex.solomon.alert.protobuf.mute.Mute proto) {
        var builder = switch (proto.getTypeCase()) {
            case SELECTORS -> protoToSelectorsMute(proto.getSelectors());
            default -> throw new IllegalArgumentException("Unsupported mute type: " + proto.getTypeCase());
        };

        builder
                .setId(proto.getId().isEmpty() ? UUID.randomUUID().toString() : proto.getId())
                .setProjectId(proto.getProjectId())
                .setFolderId(proto.getFolderId())
                //
                .setName(proto.getName())
                .setDescription(proto.getDescription())
                .setFrom(Instant.ofEpochMilli(proto.getFromMillis()))
                .setTo(Instant.ofEpochMilli(proto.getToMillis()))
                .setTicketId(proto.getTicketId())
                //
                .setVersion(proto.getVersion())
                .setCreatedAt(Instant.ofEpochMilli(proto.getCreatedAt()))
                .setUpdatedAt(Instant.ofEpochMilli(proto.getUpdatedAt()))
                .setCreatedBy(proto.getCreatedBy())
                .setUpdatedBy(proto.getUpdatedBy());

        if (proto.getTtlBase() != 0) {
            builder.setTtlBase(Instant.ofEpochMilli(proto.getTtlBase()));
        } else {
            builder.setTtlBase(Instant.ofEpochMilli(proto.getToMillis()));
        }

        return builder.build();
    }

    private SelectorsMute.Builder protoToSelectorsMute(ru.yandex.solomon.alert.protobuf.mute.SelectorsType selectors) {
        return SelectorsMute.newBuilder()
                .setAlertSelector(LabelSelectorConverter.protoToSelector(selectors.getAlertSelector()))
                .setLabelSelectors(LabelSelectorConverter.protoToSelectors(selectors.getLabelSelectors()))
                ;
    }

    public ru.yandex.solomon.alert.protobuf.mute.Mute muteToProto(Mute mute, MuteStatus status) {
        var builder = ru.yandex.solomon.alert.protobuf.mute.Mute.newBuilder();

        switch (mute.getType()) {
            case BY_SELECTORS -> builder.setSelectors(muteSelectorsToProto((SelectorsMute) mute));
            default -> throw new IllegalArgumentException("Unsupported mute type: " + mute.getType());
        }

        if (status != MuteStatus.UNKNOWN) {
            builder.setStatus(muteStatusCodeToProto(status));
        }

        builder
                .setId(mute.getId())
                .setProjectId(mute.getProjectId())
                .setFolderId(mute.getFolderId())
                //
                .setName(mute.getName())
                .setDescription(mute.getDescription())
                .setFromMillis(mute.getFrom().toEpochMilli())
                .setToMillis(mute.getTo().toEpochMilli())
                .setTtlBase(mute.getTtlBase().toEpochMilli())
                .setTicketId(mute.getTicketId())
                //
                .setVersion(mute.getVersion())
                .setCreatedAt(mute.getCreatedAt())
                .setUpdatedAt(mute.getUpdatedAt())
                .setCreatedBy(mute.getCreatedBy())
                .setUpdatedBy(mute.getUpdatedBy());

        return builder.build();
    }

    private ru.yandex.solomon.alert.protobuf.mute.SelectorsType.Builder muteSelectorsToProto(SelectorsMute mute) {
        return ru.yandex.solomon.alert.protobuf.mute.SelectorsType.newBuilder()
                .setAlertSelector(LabelSelectorConverter.selectorToProto(mute.getAlertSelector()))
                .setLabelSelectors(LabelSelectorConverter.selectorsToNewProto(mute.getLabelSelectors()))
                ;
    }

    static ru.yandex.solomon.alert.protobuf.AffectingMute.Builder affectingMuteToProto(AffectingMute affectingMute) {
        return ru.yandex.solomon.alert.protobuf.AffectingMute.newBuilder()
                .setId(affectingMute.muteId())
                .setStatus(muteStatusCodeToProto(affectingMute.status()));
    }

    public static ru.yandex.solomon.alert.protobuf.MuteStatus muteStatusCodeToProto(MuteStatus status) {
        return switch (status) {
            case UNKNOWN -> ru.yandex.solomon.alert.protobuf.MuteStatus.UNKNOWN;
            case PENDING -> ru.yandex.solomon.alert.protobuf.MuteStatus.PENDING;
            case ACTIVE -> ru.yandex.solomon.alert.protobuf.MuteStatus.ACTIVE;
            case EXPIRED -> ru.yandex.solomon.alert.protobuf.MuteStatus.EXPIRED;
            case ARCHIVED -> ru.yandex.solomon.alert.protobuf.MuteStatus.ARCHIVED;
        };
    }

    public static MuteStatus protoToMuteStatusCode(ru.yandex.solomon.alert.protobuf.MuteStatus status) {
        return MuteStatus.forNumber(status.getNumber());
    }

    static ru.yandex.solomon.alert.protobuf.AlertMuteStatus.Code alertMuteStatusCodeToProto(AlertMuteStatus.MuteStatusCode muteStatusCode) {
        return switch (muteStatusCode) {
            case UNSPECIFIED -> ru.yandex.solomon.alert.protobuf.AlertMuteStatus.Code.UNKNOWN;
            case NOT_MUTED -> ru.yandex.solomon.alert.protobuf.AlertMuteStatus.Code.NOT_MUTED;
            case PARTLY_MUTED -> ru.yandex.solomon.alert.protobuf.AlertMuteStatus.Code.PARTLY_MUTED;
            case MUTED -> ru.yandex.solomon.alert.protobuf.AlertMuteStatus.Code.MUTED;
        };
    }

    static AlertMuteStatus.MuteStatusCode protoToAlertMuteStatusCode(ru.yandex.solomon.alert.protobuf.AlertMuteStatus.Code code) {
        return AlertMuteStatus.MuteStatusCode.forNumber(code.getNumber());
    }

    static AffectingMute protoToAffectingMute(ru.yandex.solomon.alert.protobuf.AffectingMute affectingMute) {
        return new AffectingMute(
                affectingMute.getId(),
                protoToMuteStatusCode(affectingMute.getStatus())
        );
    }
}
