package ru.yandex.solomon.alert.api.converters;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.notification.NotificationKey;
import ru.yandex.solomon.alert.notification.NotificationState;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.telegram.ChatIdStorage;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.PhoneNotification;
import ru.yandex.solomon.alert.notification.domain.email.CloudEmailNotification;
import ru.yandex.solomon.alert.notification.domain.email.DatalensEmailNotification;
import ru.yandex.solomon.alert.notification.domain.email.EmailNotification;
import ru.yandex.solomon.alert.notification.domain.juggler.JugglerNotification;
import ru.yandex.solomon.alert.notification.domain.push.CloudPushNotification;
import ru.yandex.solomon.alert.notification.domain.sms.CloudSmsNotification;
import ru.yandex.solomon.alert.notification.domain.sms.SmsNotification;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.alert.notification.domain.webhook.WebhookNotification;
import ru.yandex.solomon.alert.notification.domain.yachats.YaChatsNotification;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TNotificationState;
import ru.yandex.solomon.alert.protobuf.TNotificationStatus;
import ru.yandex.solomon.alert.protobuf.notification.ENotificationChannelType;
import ru.yandex.solomon.alert.protobuf.notification.PhoneType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudPushType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TDatalensEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TEmailType;
import ru.yandex.solomon.alert.protobuf.notification.THeader;
import ru.yandex.solomon.alert.protobuf.notification.TJugglerType;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.alert.protobuf.notification.TNotificationDetails;
import ru.yandex.solomon.alert.protobuf.notification.TSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TTelegramType;
import ru.yandex.solomon.alert.protobuf.notification.TWebhookType;
import ru.yandex.solomon.alert.protobuf.notification.TYaChatsType;
import ru.yandex.solomon.util.collection.Nullables;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

/**
 * @author Vladimir Gordiychuk
 */
public final class NotificationConverter {

    private final ChatIdStorage chatIdResolver;

    public NotificationConverter(ChatIdStorage chatIdResolver) {
        this.chatIdResolver = chatIdResolver;
    }

    public Notification protoToNotification(TNotification proto) {
        final Notification.Builder<?, ?> builder = switch (proto.getTypeCase()) {
            case EMAIL -> protoToEmail(proto.getEmail());
            case WEBKOOK -> protoToWebhook(proto.getWebkook());
            case PHONE -> protoToPhone(proto.getPhone());
            case JUGGLER -> protoToJuggler(proto.getJuggler());
            case SMS -> protoToSms(proto.getSms());
            case TELEGRAM -> protoToTelegram(proto.getTelegram());
            case CLOUDEMAIL -> protoToCloudEmail(proto.getCloudEmail());
            case CLOUDSMS -> protoToCloudSms(proto.getCloudSms());
            case YACHATS -> protoToYaChats(proto.getYaChats());
            case DATALENSEMAIL -> protoToDatalensEmail(proto.getDatalensEmail());
            case CLOUDPUSH -> protoToCloudPush(proto.getCloudPush());
            default -> throw new UnsupportedOperationException("Unsupported notification type: " + proto);
        };

        return builder.setId("".equals(proto.getId()) ? UUID.randomUUID().toString() : proto.getId())
                .setProjectId(proto.getProjectId())
                .setFolderId(proto.getFolderId())
                .setNotifyAboutStatus(proto.getNotifyAboutStatusesList()
                        .stream()
                        .map(status -> EvaluationStatus.Code.valueOf(status.name()))
                        .collect(Collectors.toSet()))
                .setVersion(proto.getVersion())
                .setName(proto.getName())
                .setDescription(proto.getDescription())
                .setCreatedBy(proto.getCreatedBy())
                .setCreatedAt(Instant.ofEpochMilli(proto.getCreatedAt()))
                .setUpdatedBy(proto.getUpdatedBy())
                .setUpdatedAt(Instant.ofEpochMilli(proto.getUpdatedAt()))
                .setRepeatNotifyDelay(Duration.ofMillis(proto.getRepeatNotifyDelayMillis()))
                .setLabels(proto.getLabelsMap())
                .setDefaultForProject(proto.getDefaultForProject())
                .setDefaultForSeverity(proto.getDefaultForAlertSeverityList().stream()
                        .map(severity -> AlertSeverity.forNumber(severity.getNumber()))
                        .collect(toSet()))
                .build();
    }

    private EmailNotification.Builder protoToEmail(TEmailType proto) {
        return EmailNotification.newBuilder()
                .setRecipients(proto.getRecipientsList())
                .setContentTemplate(proto.getContentTemplate())
                .setSubjectTemplate(proto.getSubjectTemplate());
    }

    private CloudEmailNotification.Builder protoToCloudEmail(TCloudEmailType proto) {
        return CloudEmailNotification.newBuilder()
                .setRecipients(proto.getRecipientsList());
    }

    private CloudSmsNotification.Builder protoToCloudSms(TCloudSmsType proto) {
        return CloudSmsNotification.newBuilder()
                .setRecipients(proto.getRecipientsList());
    }

    private CloudPushNotification.Builder protoToCloudPush(TCloudPushType proto) {
        return CloudPushNotification.newBuilder()
                .setRecipients(proto.getRecipientsList());
    }

    private WebhookNotification.Builder protoToWebhook(TWebhookType proto) {
        return WebhookNotification.newBuilder()
                .setUrl(proto.getUrl())
                .setTemplate(proto.getTemplate())
                .setHeaders(proto.getHeadersList()
                        .stream()
                        .collect(toMap(THeader::getName, THeader::getValue)));
    }

    private PhoneNotification.Builder protoToPhone(PhoneType proto) {
        switch (proto.getReceiverCase()) {
            case DUTY -> {
                return PhoneNotification.newBuilder()
                        .setDuty(new PhoneNotification.AbcDuty(proto.getDuty().getAbcService(), proto.getDuty().getDutySlug()));
            }
            case LOGIN -> {
                return PhoneNotification.newBuilder()
                        .setLogin(proto.getLogin());
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + proto.getReceiverCase());
    }

    private JugglerNotification.Builder protoToJuggler(TJugglerType proto) {
        return JugglerNotification.newBuilder()
                .setHost(proto.getHost())
                .setService(proto.getService())
                .setInstance(proto.getInstance())
                .setJugglerDescription(proto.getDescription())
                .setTags(proto.getTagsList());
    }

    private SmsNotification.Builder protoToSms(TSmsType proto) {
        SmsNotification.Builder builder = SmsNotification.newBuilder()
                .setTextTemplate(proto.getTextTemplate());

        switch (proto.getTypeCase()) {
            case LOGIN:
                builder.setLogin(proto.getLogin());
                break;
            case PHONE:
                builder.setPhone(proto.getPhone());
                break;
            default:
                throw new UnsupportedOperationException("Unsupported type: " + proto.getTypeCase());
        }

        return builder;
    }

    private TelegramNotification.Builder protoToTelegram(TTelegramType proto) {
        TelegramNotification.Builder builder = TelegramNotification.newBuilder();
        switch (proto.getReceiverCase()) {
            case LOGIN:
                builder.setLogin(proto.getLogin());
                break;
            case GROUPTITLE:
                long chatId = chatIdResolver.getChatIdByGroupTitle(proto.getGroupTitle());
                if (chatId == 0) {
                    throw new IllegalArgumentException("don't have group with title:" + proto.getGroupTitle());
                }
                builder.setChatId(chatId);
                break;
            default:
                throw new UnsupportedOperationException("Unsupported type: " + proto.getReceiverCase());
        }
        return builder
                .setTextTemplate(proto.getTextTemplate())
                .setSendScreenshot(proto.getSendScreenshot());
    }

    private YaChatsNotification.Builder protoToYaChats(TYaChatsType proto) {
        YaChatsNotification.Builder builder = YaChatsNotification.newBuilder();
        switch (proto.getReceiverCase()) {
            case LOGIN:
                builder.setLogin(proto.getLogin());
                break;
            case GROUPID:
                builder.setGroupId(proto.getGroupId());
                break;
            default:
                throw new UnsupportedOperationException("Unsupported type: " + proto.getReceiverCase());
        }
        return builder
                .setTextTemplate(proto.getTextTemplate());
    }

    private DatalensEmailNotification.Builder protoToDatalensEmail(TDatalensEmailType proto) {
        return DatalensEmailNotification.newBuilder()
                .setRecipients(proto.getRecipientsList());
    }

    public TNotification notificationToProto(Notification target) {
        final TNotification.Builder builder = TNotification.newBuilder();
        switch (target.getType()) {
            case EMAIL -> builder.setEmail(emailToProto((EmailNotification) target));
            case WEBHOOK -> builder.setWebkook(webhookToProto((WebhookNotification) target));
            case PHONE_CALL -> builder.setPhone(phoneToProto((PhoneNotification) target));
            case JUGGLER -> builder.setJuggler(jugglerToProto((JugglerNotification) target));
            case SMS -> builder.setSms(smsToProto((SmsNotification) target));
            case TELEGRAM -> builder.setTelegram(telegramToProto((TelegramNotification) target));
            case CLOUD_EMAIL -> builder.setCloudEmail(cloudEmailToProto((CloudEmailNotification) target));
            case CLOUD_SMS -> builder.setCloudSms(cloudSmsToProto((CloudSmsNotification) target));
            case YA_CHATS -> builder.setYaChats(yaChatsToProto((YaChatsNotification) target));
            case DATALENS_EMAIL -> builder.setDatalensEmail(datalensEmailToProto((DatalensEmailNotification) target));
            case CLOUD_PUSH -> builder.setCloudPush(cloudPushToProto((CloudPushNotification) target));
            default -> throw new UnsupportedOperationException("Unsupported notification type: " + target);
        }

        return builder
                .setId(target.getId())
                .setProjectId(target.getProjectId())
                .setFolderId(target.getFolderId())
                .setName(target.getName())
                .setDescription(target.getDescription())
                .setCreatedBy(target.getCreatedBy())
                .setCreatedAt(target.getCreatedAt())
                .setUpdatedBy(target.getUpdatedBy())
                .setUpdatedAt(target.getUpdatedAt())
                .setVersion(target.getVersion())
                .addAllNotifyAboutStatuses(target.getNotifyAboutStatus()
                        .stream()
                        .map(code -> TEvaluationStatus.ECode.forNumber(code.getNumber()))
                        .sorted()
                        .collect(Collectors.toList()))
                .setRepeatNotifyDelayMillis(target.getRepeatNotifyDelay().toMillis())
                .putAllLabels(target.getLabels())
                .setDefaultForProject(target.isDefaultForProject())
                .addAllDefaultForAlertSeverityValue(target.getDefaultForSeverity().stream()
                        .map(AlertSeverity::getNumber)
                        .collect(toList()))
                .build();
    }

    public TNotificationDetails notificationToDetailsProto(Notification notification) {
        final TNotificationDetails.Builder builder = TNotificationDetails.newBuilder()
                .setNotificationId(notification.getId())
                .setProjectId(notification.getProjectId())
                .setFolderId(notification.getFolderId())
                .setName(notification.getName());

        switch (notification.getType()) {
            case EMAIL -> builder
                    .setType(ENotificationChannelType.EMAIL)
                    .addAllRecipients(emailRecipients((EmailNotification) notification));
            case WEBHOOK -> builder
                    .setType(ENotificationChannelType.WEBHOOK)
                    .addAllRecipients(webhookRecipients((WebhookNotification) notification));
            case PHONE_CALL -> builder
                    .setType(ENotificationChannelType.PHONE_CALL)
                    .addAllRecipients(phoneRecipients((PhoneNotification) notification));
            case JUGGLER -> builder
                    .setType(ENotificationChannelType.JUGGLER)
                    .addAllRecipients(jugglerRecipients((JugglerNotification) notification));
            case SMS -> builder
                    .setType(ENotificationChannelType.SMS)
                    .addAllRecipients(smsRecipients((SmsNotification) notification));
            case TELEGRAM -> builder
                    .setType(ENotificationChannelType.TELEGRAM)
                    .addAllRecipients(telegramRecipients((TelegramNotification) notification));
            case CLOUD_EMAIL -> builder
                    .setType(ENotificationChannelType.CLOUD_EMAIL)
                    .addAllRecipients(cloudEmailRecipients((CloudEmailNotification) notification));
            case CLOUD_SMS -> builder
                    .setType(ENotificationChannelType.CLOUD_SMS)
                    .addAllRecipients(cloudSmsRecipients((CloudSmsNotification) notification));
            case YA_CHATS -> builder
                    .setType(ENotificationChannelType.YA_CHATS)
                    .addAllRecipients(yaChatsRecipients((YaChatsNotification) notification));
            case DATALENS_EMAIL -> builder
                    .setType(ENotificationChannelType.DATALENS_EMAIL)
                    .addAllRecipients(datalensEmailRecipients((DatalensEmailNotification) notification));
            case CLOUD_PUSH -> builder
                    .setType(ENotificationChannelType.CLOUD_PUSH)
                    .addAllRecipients(cloudPushRecipients((CloudPushNotification) notification));
            default -> throw new UnsupportedOperationException("Unsupported notification type: " + notification);
        }

        return builder.build();
    }

    private Iterable<String> telegramRecipients(TelegramNotification notification) {
        String recipient = Nullables.orEmpty(notification.getLogin());
        if (recipient.isEmpty()) {
            recipient = Nullables.orEmpty(chatIdResolver.resolveGroupTitle(notification.getChatId()));
        }
        return List.of(recipient);
    }

    private Iterable<String> yaChatsRecipients(YaChatsNotification notification) {
        return List.of(notification.getLogin());
    }

    private Iterable<String> smsRecipients(SmsNotification notification) {
        String recipient = notification.getLogin();
        if (recipient.isEmpty()) {
            recipient = notification.getPhone();
        }
        return List.of(recipient);
    }

    private Iterable<String> jugglerRecipients(JugglerNotification notification) {
        return List.of(notification.getService() + "@" + notification.getHost());
    }

    private Iterable<String> webhookRecipients(WebhookNotification notification) {
        return List.of(notification.getUrl());
    }

    private Iterable<String> phoneRecipients(PhoneNotification notification) {
        return List.of(notification.toDisplayText());
    }

    private Iterable<String> emailRecipients(EmailNotification notification) {
        return notification.getRecipients();
    }

    private Iterable<String> cloudEmailRecipients(CloudEmailNotification notification) {
        return notification.getRecipients();
    }

    private Iterable<String> datalensEmailRecipients(DatalensEmailNotification notification) {
        return notification.getRecipients();
    }

    private Iterable<String> cloudSmsRecipients(CloudSmsNotification notification) {
        return notification.getRecipients();
    }

    private Iterable<String> cloudPushRecipients(CloudPushNotification notification) {
        return notification.getRecipients();
    }

    private TEmailType emailToProto(EmailNotification target) {
        return TEmailType.newBuilder()
                .addAllRecipients(target.getRecipients())
                .setContentTemplate(target.getContentTemplate() != null ? target.getContentTemplate() : "")
                .setSubjectTemplate(target.getSubjectTemplate() != null ? target.getSubjectTemplate() : "")
                .build();
    }

    private TCloudEmailType cloudEmailToProto(CloudEmailNotification target) {
        return TCloudEmailType.newBuilder()
                .addAllRecipients(target.getRecipients())
                .build();
    }

    private TDatalensEmailType datalensEmailToProto(DatalensEmailNotification target) {
        return TDatalensEmailType.newBuilder()
                .addAllRecipients(target.getRecipients())
                .build();
    }

    private TCloudSmsType cloudSmsToProto(CloudSmsNotification target) {
        return TCloudSmsType.newBuilder()
                .addAllRecipients(target.getRecipients())
                .build();
    }

    private TCloudPushType cloudPushToProto(CloudPushNotification target) {
        return TCloudPushType.newBuilder()
                .addAllRecipients(target.getRecipients())
                .build();
    }

    private TWebhookType webhookToProto(WebhookNotification target) {
        return TWebhookType.newBuilder()
                .setUrl(target.getUrl())
                .setTemplate(target.getTemplate() != null ? target.getTemplate() : "")
                .addAllHeaders(target.getHeaders()
                        .entrySet()
                        .stream()
                        .map(entry -> THeader.newBuilder()
                                .setName(entry.getKey())
                                .setValue(entry.getValue())
                                .build())
                        .collect(toList()))
                .build();
    }

    private PhoneType phoneToProto(PhoneNotification target) {
        if (target.isLogin()) {
            return PhoneType.newBuilder()
                    .setLogin(target.getLogin())
                    .build();
        }
        return PhoneType.newBuilder()
                .setDuty(PhoneType.Duty.newBuilder()
                        .setAbcService(target.getDuty().abcService())
                        .setDutySlug(target.getDuty().dutySlug())
                        .build())
                .build();
    }

    private TJugglerType jugglerToProto(JugglerNotification target) {
        return TJugglerType.newBuilder()
                .setHost(target.getHost())
                .setService(target.getService())
                .setInstance(target.getInstance())
                .setDescription(target.getJugglerDescription())
                .addAllTags(target.getTags())
                .build();
    }

    private TSmsType smsToProto(SmsNotification target) {
        TSmsType.Builder builder = TSmsType.newBuilder()
                .setTextTemplate(target.getTextTemplate());

        if (!"".equals(target.getLogin())) {
            builder.setLogin(target.getLogin());
        } else {
            builder.setPhone(target.getPhone());
        }

        return builder.build();
    }

    private TTelegramType telegramToProto(TelegramNotification target) {
        TTelegramType.Builder builder = TTelegramType.newBuilder()
                .setTextTemplate(target.getTextTemplate())
                .setSendScreenshot(target.isSendScreenshot());

        if (target.getLogin() != null && !"".equals(target.getLogin())) {
            builder.setLogin(target.getLogin());
        } else {
            String chatId = chatIdResolver.resolveGroupTitle(target.getChatId());
            if (chatId != null) {
                builder.setGroupTitle(chatId);
            }
        }
        return builder.build();
    }

    private TYaChatsType yaChatsToProto(YaChatsNotification target) {
        TYaChatsType.Builder builder = TYaChatsType.newBuilder()
                .setTextTemplate(target.getTextTemplate());

        if (target.getLogin() != null && !"".equals(target.getLogin())) {
            builder.setLogin(target.getLogin());
        } else if (target.getGroupId() != null) {
            builder.setGroupId(target.getGroupId());
        }

        return builder.build();
    }

    public TNotificationStatus statusToProto(NotificationStatus status) {
        return TNotificationStatus.newBuilder()
                .setCodeValue(status.getCode().getNumber())
                .setDesctiption(status.getDescription())
                .build();
    }

    public TNotificationState stateToProto(NotificationState state) {
        NotificationKey key = state.getKey();
        return TNotificationState.newBuilder()
                .setAlertId(key.getAlertKey().getAlertId())
                .setProjectId(key.getProjectId())
                .setNotificationChannelId(key.getNotificationId())
                .setLatestEvalMillis(state.getLatestEval().toEpochMilli())
                .setLatestSuccessMillis(state.getLatestSuccessNotify().toEpochMilli())
                .setStatus(statusToProto(state.getLatestStatus()))
                .build();
    }

    public NotificationStatus protoToStatus(TNotificationStatus proto) {
        return protoToStatusCode(proto.getCode()).toStatus(proto.getDesctiption());
    }

    public NotificationStatus.Code protoToStatusCode(TNotificationStatus.ECode proto) {
        return NotificationStatus.Code.forNumber(proto.getNumber());
    }
}
