package ru.yandex.solomon.alert.api.validators;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.base.Throwables;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.protobuf.AlertFromTemplate;
import ru.yandex.solomon.alert.protobuf.AlertTemplate;
import ru.yandex.solomon.alert.protobuf.AlertTemplateParameter;
import ru.yandex.solomon.alert.protobuf.AlertTemplateStatus;
import ru.yandex.solomon.alert.protobuf.CreateAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.DeleteAlertTemplatePublicationRequest;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.Expression;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsRequest;
import ru.yandex.solomon.alert.protobuf.NoPointsPolicy;
import ru.yandex.solomon.alert.protobuf.PredicateRule;
import ru.yandex.solomon.alert.protobuf.PublishAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ReadAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExpression;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSubAlert;
import ru.yandex.solomon.alert.protobuf.TThreshold;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TemplateDeployPolicy;
import ru.yandex.solomon.alert.protobuf.Threshold;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.rule.ProgramCompiler;
import ru.yandex.solomon.expression.SelParser;
import ru.yandex.solomon.expression.analytics.PreparedProgram;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.ast.AstAnonymous;
import ru.yandex.solomon.expression.ast.AstCall;
import ru.yandex.solomon.expression.ast.AstStatement;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsValidator;
import ru.yandex.solomon.model.protobuf.Selector;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Vladimir Gordiychuk
 */
public final class AlertValidator {
    private AlertValidator() {
    }

    public static void ensureValid(TCreateAlertRequest request) {
        if (!request.hasAlert()) {
            throw invalid("Request doesn't have alert to create: %s", request);
        }

        ensureValidAlert(request.getAlert(), true);
    }

    public static void ensureValid(TUpdateAlertRequest request) {
        if (!request.hasAlert()) {
            throw invalid("Request doesn't have alert to update:\n%s", request);
        }

        ensureValidAlert(request.getAlert(), false);
    }

    public static void ensureValid(TDeleteAlertRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TDeletionNotificationRequest request) {
        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadAlertRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadAlertInterpolatedRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadSubAlertRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }

        if ("".equals(request.getParentId())) {
            throw invalid("Request doesn't have parent id:\n%s", request);
        }
    }

    public static void ensureValid(TListAlertRequest request) {
        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TListSubAlertRequest request) {
        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }

        if ("".equals(request.getParentId())) {
            throw invalid("ParentId not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadEvaluationStateRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadEvaluationStatsRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadNotificationStateRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TReadNotificationStatsRequest request) {
        if ("".equals(request.getAlertId())) {
            throw invalid("Request doesn't have alert id:\n%s", request);
        }

        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(TExplainEvaluationRequest request) {
        switch (request.getCheckCase()) {
            case REFERENCE: {
                TExplainEvaluationRequest.TReferenceToAlert reference = request.getReference();
                if ("".equals(reference.getAlertId())) {
                    throw invalid("Request doesn't have alert id:\n%s", request);
                }

                if ("".equals(reference.getProjectId())) {
                    throw invalid("Project not specified for request:\n%s", request);
                }

                break;
            }
            case SUBALERT: {
                TSubAlert subAlert = request.getSubAlert();
                ensureValidAlert(subAlert.getParent(), false);
                break;
            }
            case ALERT: {
                ensureValidAlert(request.getAlert(), false);
                break;
            }
            default: {
                invalid("Unsupported alert check: \n%s", request);
            }
        }

        if (request.getEvaluationTimeMillis() == 0) {
            throw invalid("Not specified evaluation time:\n%s", request);
        }
    }


    public static void ensureValid(TSimulateEvaluationRequest request) {
        switch (request.getCheckCase()) {
            case REFERENCE: {
                TExplainEvaluationRequest.TReferenceToAlert reference = request.getReference();
                if ("".equals(reference.getAlertId())) {
                    throw invalid("Request doesn't have alert id:\n%s", request);
                }

                if ("".equals(reference.getProjectId())) {
                    throw invalid("Project not specified for request:\n%s", request);
                }

                break;
            }
            case SUBALERT: {
                TSubAlert subAlert = request.getSubAlert();
                ensureValidAlert(subAlert.getParent(), false);
                break;
            }
            case ALERT: {
                ensureValidAlert(request.getAlert(), false);
                break;
            }
            default: {
                invalid("Unsupported alert check: \n%s", request);
            }
        }

        if (request.getSimulationTimeBeginMillis() == 0) {
            throw invalid("Not specified simulation begin time:\n%s", request);
        }

        if (request.getSimulationTimeEndMillis() == 0) {
            throw invalid("Not specified simulation end time:\n%s", request);
        }

        if (request.getGridMillis() == 0) {
            throw invalid("Not specified simulation grid spacing:\n%s", request);
        }

        if (request.getSimulationTimeBeginMillis() >= request.getSimulationTimeEndMillis()) {
            throw invalid("Simulation begin time should be less than the end time");
        }
    }

    public static void ensureValid(TReadProjectStatsRequest request) {
        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    private static void ensureValidAlert(TAlert alert, boolean strictValidation) {
        if ("".equals(alert.getProjectId())) {
            throw invalid("Project not specified for alert:\n%s", alert);
        }

        IdValidator.ensureValid(alert.getId(), "alert", strictValidation);

        if (StringUtils.isBlank(alert.getName())) {
            throw invalid("Alert name is blank");
        }

        ensureGroupLabelsValid(alert.getGroupByLabelsList());

        switch (alert.getTypeCase()) {
            case THRESHOLD:
                validatePeriodAndDelay(alert);
                ensureThresholdNoDataPolicyValid(alert.getResolvedEmptyPolicy(), alert.getNoPointsPolicy());
                ensureThresholdAlertValid(alert.getThreshold(), alert.getPeriodMillis());
                break;
            case EXPRESSION:
                validatePeriodAndDelay(alert);
                ensureExpressionNoDataPolicyValid(alert.getResolvedEmptyPolicy(), alert.getNoPointsPolicy());
                ensureExpressionAlertValid(alert.getExpression(), alert.getPeriodMillis());
                break;
            case ALERT_FROM_TEMPLATE:
                validateTemplateAlert(alert, alert.getAlertFromTemplate());
                break;
            default:
                throw invalid("Unsupported alert type:\n%s", alert);
        }
    }

    private static void validateTemplateAlert(TAlert alert, AlertFromTemplate alertFromTemplate) {
        validateTemplateId(alert, alertFromTemplate);
        if (alert.getGroupByLabelsCount() > 0) {
            throw invalid("Can't specify group by label for template alert, use alert template for that.\n(%s)", alert);
        }
        if (alert.getPeriodMillis() > 0) {
            throw invalid("Can't specify period for template alert, use alert template for that.\n(%s)", alert);
        }
        if (alert.getDelaySeconds() > 0) {
            throw invalid("Can't specify evaluation delay seconds for template alert, use alert template for that.\n(%s)", alert);
        }
        if (alert.getResolvedEmptyPolicy() != ResolvedEmptyPolicy.RESOLVED_EMPTY_DEFAULT) {
            throw invalid("Can't specify resolvedEmptyPolicy for template alert, use alert template for that.\n(%s)", alert);
        }
        if (alert.getNoPointsPolicy() != NoPointsPolicy.NO_POINTS_DEFAULT) {
            throw invalid("Can't specify noPointsPolicy for template alert, use alert template for that.\n(%s)", alert);
        }
        if (alert.getSeverity().getNumber() > 0) {
            throw invalid("Can't specify severity for template alert, use alert template for that.\n(%s)", alert);
        }
    }

    private static void validateTemplateId(TAlert alert, AlertFromTemplate alertFromTemplate) {
        if (StringUtils.isEmpty(alertFromTemplate.getTemplateId())) {
            throw invalid("TemplateId not specified, for alert %s", alert);
        }
        if (StringUtils.isEmpty(alertFromTemplate.getTemplateVersionTag())) {
            throw invalid("TemplateVersionTag not specified, for alert %s", alert);
        }
    }

    private static void validatePeriodAndDelay(TAlert alert) {
        if (alert.getPeriodMillis() <= 0) {
            throw invalid("Period should be greater then 0, for alert %s", alert.getId());
        }

        if (alert.getDelaySeconds() < 0) {
            throw invalid("Evaluation delay seconds should be positive:\n%s", alert);
        }
    }

    private static void ensureExpressionNoDataPolicyValid(ResolvedEmptyPolicy resolvedEmptyPolicy, NoPointsPolicy noPointsPolicy) {
        if (resolvedEmptyPolicy == ResolvedEmptyPolicy.UNRECOGNIZED) {
            throw invalid("Unrecognized resolvedEmptyPolicy: %s", resolvedEmptyPolicy);
        }
        if (noPointsPolicy == NoPointsPolicy.UNRECOGNIZED) {
            throw invalid("Unrecognized noPointsPolicy: %s", resolvedEmptyPolicy);
        }
    }

    private static void ensureThresholdNoDataPolicyValid(ResolvedEmptyPolicy resolvedEmptyPolicy, NoPointsPolicy noPointsPolicy) {
        if (resolvedEmptyPolicy == ResolvedEmptyPolicy.UNRECOGNIZED) {
            throw invalid("Unrecognized resolvedEmptyPolicy: " + resolvedEmptyPolicy);
        }
        if (noPointsPolicy == NoPointsPolicy.UNRECOGNIZED) {
            throw invalid("Unrecognized noPointsPolicy: " + resolvedEmptyPolicy);
        }
        if (resolvedEmptyPolicy == ResolvedEmptyPolicy.RESOLVED_EMPTY_MANUAL) {
            throw invalid("Bad resolvedEmptyPolicy for threshold alert: " + resolvedEmptyPolicy);
        }
    }

    private static void ensureGroupLabelsValid(List<String> groupByLabelsList) {
        for (String groupByLabel : groupByLabelsList) {
            if (groupByLabel.isBlank()) {
                throw invalid("Group by label cannot be blank, got: " + groupByLabelsList);
            }
            // TODO: check against selector instead of key. Cannot do this now, because not all keys are strictly valid
            // E.g. Group by labels: ["host=*"]
        }
    }

    private static void ensureThresholdAlertValid(TThreshold alert, long periodMillis) {
        var newSelectors = alert.getNewSelectors();
        enusureNewSelectorsValid(newSelectors);

        if (alert.getComparison() == ECompare.UNRECOGNIZED) {
            throw invalid("Unrecognized comparison for alert %s", alert);
        }

        if (alert.getThresholdType() == EThresholdType.UNRECOGNIZED) {
            throw invalid("Unrecognized threshold type for alert %s", alert);
        }

        for (TPredicateRule rule : alert.getPredicateRulesList()) {
            ensurePredicateRuleValid(rule);
        }

        ensureTransformationsValid(alert, periodMillis);

        // TODO: until label validation rule disabled use encode/decode check (gordiychuk@)
        try {
            Selectors selectors = LabelSelectorConverter.protoToSelectors(newSelectors);
            String formatted = Selectors.format(selectors);
            if (!selectors.equals(Selectors.parse(formatted))) {
                throw invalid("Not valid selectors: %s", selectors);
            }
        } catch (Throwable e) {
            throw new ValidationException("Not valid selectors: " + newSelectors, e);
        }
    }

    private static void ensurePredicateRuleValid(TPredicateRule rule) {
        if (rule.getComparison() == ECompare.UNRECOGNIZED) {
            throw invalid("Unrecognized comparison for rule %s", rule);
        }

        if (rule.getThresholdType() == EThresholdType.UNRECOGNIZED) {
            throw invalid("Unrecognized threshold type for rule %s", rule);
        }

        if (rule.getTargetStatus() == TPredicateRule.ETargetStatus.UNRECOGNIZED) {
            throw invalid("Unrecognized target status for rule %s", rule);
        }

        if (rule.getTargetStatus() == TPredicateRule.ETargetStatus.UNSPECIFIED) {
            throw invalid("Unspecified target status for rule %s", rule);
        }

    }

    private static void enusureNewSelectorsValid(ru.yandex.solomon.model.protobuf.Selectors selectors) {
        if (selectors.getLabelSelectorsCount() == 0 && selectors.getNameSelector().isEmpty()) {
            throw invalid("Empty selectors for for threshold alert");
        }
        for (Selector selector : selectors.getLabelSelectorsList()) {
            ensureSelectorValid(selector);
        }
    }

    private static void ensureSelectorValid(Selector selector) {
        SelectorType type = SelectorType.forNumber(selector.getMatchType().getNumber());
        if (!SelectorsValidator.isValid(type, selector.getKey(), selector.getPattern())) {
            throw invalid("Selector not valid : %s", selector);
        }
    }

    private static void ensureExpressionAlertValid(TExpression expression, long periodMillis) {
        try {
            testRunExpression(expression, periodMillis);
        } catch (ValidationException e) {
            throw e;
        } catch (Throwable e) {
            throw invalid(Throwables.getStackTraceAsString(e));
        }
    }

    // Alert with none of these functions will always be in OK state
    private static final Set<String> REQUIRED_STATUS_IF = Set.of("alarm_if", "warn_if", "no_data_if");

    private static void testRunExpression(TExpression alert, long periodMillis) {
        String code = ExpressionAlert.combineProgramWithCheck(alert.getProgram(), alert.getCheckExpression());
        List<AstStatement> statements = new SelParser(code).parseBlock();

        boolean missingControlCondition = statements.stream()
                .filter(statement -> statement instanceof AstAnonymous)
                .map(statement -> (AstAnonymous)statement)
                .filter(anon -> anon.getExpr() instanceof AstCall)
                .map(anon -> (AstCall) anon.getExpr())
                .noneMatch(call -> REQUIRED_STATUS_IF.contains(call.getFunc().getIdent()));

        if (missingControlCondition) {
            invalid("Alert should have a valid check expression or use " +
                    "alarm_if, warn_if or no_data_if functions");
        }

        Program program = Program.fromSource(ProgramCompiler.ALERTING_SEL_VERSION, code)
                .withDeprOpts(DeprOpts.ALERTING)
                .compile();

        program.prepare(Interval.before(Instant.now(), Duration.ofMillis(periodMillis)));
    }

    private static void ensureTransformationsValid(TThreshold alert, long periodMillis) {
        try {
            testTransformationsCompile(alert, periodMillis);
        } catch (ValidationException e) {
            throw e;
        } catch (Throwable e) {
            throw invalid(Throwables.getStackTraceAsString(e));
        }
    }

    private static void testTransformationsCompile(TThreshold alert, long periodMillis) {
        if (alert.getTransformations().isEmpty()) {
            return;
        }

        Program program = Program.fromSource(ProgramCompiler.ALERTING_SEL_VERSION, "")
                .withExternalExpression(alert.getTransformations())
                .withDeprOpts(DeprOpts.ALERTING)
                .compile();

        PreparedProgram preparedProgram = program.prepare(Interval.before(Instant.now(), Duration.ofMillis(periodMillis)));
        if (preparedProgram.getLoadRequests().size() != 1) {
            invalid("Transformation program should have exactly one load request, but had %s", preparedProgram.getLoadRequests());
        }
    }

    public static void ensureValid(ReadAlertTemplateRequest request) {
        if ("".equals(request.getTemplateId())) {
            throw invalid("Request doesn't have template id:\n%s", request);
        }
    }

    public static void ensureValid(PublishAlertTemplateRequest request) {
        if ("".equals(request.getTemplateId())) {
            throw invalid("Request doesn't have template id:\n%s", request);
        }
        if ("".equals(request.getTemplateVersionTag())) {
            throw invalid("Request doesn't have template version tag:\n%s", request);
        }
    }

    public static void ensureValid(DeleteAlertTemplatePublicationRequest request) {
        if ("".equals(request.getTemplateId())) {
            throw invalid("Request doesn't have template id:\n%s", request);
        }
    }

    public static void ensureValid(ListAlertTemplateVersionsRequest request) {
        if ("".equals(request.getTemplateId())) {
            throw invalid("Request doesn't have template id:\n%s", request);
        }
    }

    public static void ensureValid(DeployAlertTemplateRequest request) {
        if ("".equals(request.getTemplateId())) {
            throw invalid("Request doesn't have template id:\n%s", request);
        }
        if ("".equals(request.getTemplateVersionTag())) {
            throw invalid("Request doesn't have template version tag:\n%s", request);
        }
        if (request.getTemplateDeployPolicy() == TemplateDeployPolicy.TEMPLATE_DEPLOY_POLICY_UNSPECIFIED) {
            throw invalid("Request doesn't specify deploy policy:\n%s", request);
        }
    }

    public static void ensureValid(UpdateAlertTemplateVersionRequest request) {
        if ("".equals(request.getTemplateId())) {
            throw invalid("Request doesn't have template id:\n%s", request);
        }
        if ("".equals(request.getTemplateVersionTag())) {
            throw invalid("Request doesn't have template version tag:\n%s", request);
        }
        if ("".equals(request.getProjectId())) {
            throw invalid("Request doesn't have project:\n%s", request);
        }
        switch (request.getTypeCase()) {
            case UPDATE_COUNT -> {
                if (request.getUpdateCount() < 1) {
                    throw invalid("Request update count must be more then 0:\n%s", request);
                }
            }
            case ALERT_DATA -> {
                if ("".equals(request.getAlertData().getAlertId())) {
                    throw invalid("Request doesn't have alert id:\n%s", request);
                }
            }
            case TYPE_NOT_SET -> throw invalid("Unspecified type:\n%s", request);
        }
    }

    public static void ensureValid(ListAlertTemplateRequest request) {
        if (!StringUtils.isEmpty(request.getLabelsSelector())) {
            try {
                Selectors.parse(request.getLabelsSelector());
            } catch (Throwable e) {
                throw new ValidationException("Not valid selectors: " + request.getLabelsSelector(), e);
            }
        }
    }

    public static void ensureValid(CreateAlertTemplateRequest request) {
        AlertTemplate alertTemplate = request.getAlertTemplate();
        IdValidator.ensureValid(alertTemplate.getId(), "alert template", true);

        if ("".equals(alertTemplate.getTemplateVersionTag())) {
            throw invalid("Request doesn't have template version tag");
        }
        if ("".equals(alertTemplate.getServiceProviderId())) {
            throw invalid("Request doesn't have service provider");
        }
        if (!alertTemplate.getId().startsWith(alertTemplate.getServiceProviderId())) {
            throw invalid("Template id should starts with service provider id");
        }
        if (StringUtils.isBlank(alertTemplate.getName())) {
            throw invalid("Alert template name is blank");
        }
        if (alertTemplate.getAlertTemplateStatus() != AlertTemplateStatus.ALERT_TEMPLATE_STATUS_UNSPECIFIED &&
                alertTemplate.getAlertTemplateStatus() != AlertTemplateStatus.UNRECOGNIZED) {
            throw invalid("Alert status can't be specified");
        }

        ensureGroupLabelsValid(alertTemplate.getGroupByLabelsList());
        if (alertTemplate.getPeriodMillis() <= 0) {
            throw invalid("Period should be greater then 0, for alert");
        }

        if (alertTemplate.getDelaySeconds() < 0) {
            throw invalid("Evaluation delay seconds should be positive");
        }
        if (alertTemplate.getResolvedEmptyPolicy() == ResolvedEmptyPolicy.UNRECOGNIZED) {
            throw invalid("Unrecognized resolvedEmptyPolicy: " + alertTemplate.getResolvedEmptyPolicy());
        }
        if (alertTemplate.getNoPointsPolicy() == NoPointsPolicy.UNRECOGNIZED) {
            throw invalid("Unrecognized noPointsPolicy: " + alertTemplate.getNoPointsPolicy());
        }
        var list = new ArrayList<>(alertTemplate.getAlertTemplateParametersList());
        list.addAll(alertTemplate.getAlertTemplateThresholdsList());
        validateParameters(list);
        validateThresholdTypes(alertTemplate.getAlertTemplateThresholdsList());

        switch (alertTemplate.getTypeCase()) {
            case THRESHOLD:
                ensureThresholdAlertTemplateValid(alertTemplate.getThreshold());
                break;
            case EXPRESSION:
                ensureExpressionAlertTemplateValid(alertTemplate.getExpression());
                break;
            default:
                throw invalid("Unsupported alert template type:\n%s", alertTemplate);
        }
    }

    private static void validateThresholdTypes(List<AlertTemplateParameter> thresholdList) {
        for (AlertTemplateParameter param : thresholdList) {
            switch (param.getParameterCase()) {
                case DOUBLE_PARAMETER, INTEGER_PARAMETER -> {
                }
                default -> throw invalid("Can't use \"%s\" type of threshold", param.getParameterCase());
            }
        }
    }

    private static void validateParameters(List<AlertTemplateParameter> alertTemplateParametersList) {
        Set<String> names = new HashSet<>();
        for (AlertTemplateParameter param : alertTemplateParametersList) {
            String name;
            switch (param.getParameterCase()) {
                case DOUBLE_PARAMETER -> {
                    name = param.getDoubleParameter().getName();
                    validateText(param.getDoubleParameter().getName(), "name");
                    validateText(param.getDoubleParameter().getTitle(), "title");
                }
                case INTEGER_PARAMETER -> {
                    name = param.getIntegerParameter().getName();
                    validateText(param.getIntegerParameter().getName(), "name");
                    validateText(param.getIntegerParameter().getTitle(), "title");
                }
                case TEXT_PARAMETER -> {
                    name = param.getTextParameter().getName();
                    validateText(param.getTextParameter().getName(), "name");
                    validateText(param.getTextParameter().getTitle(), "title");
                }
                case TEXT_LIST_PARAMETER -> {
                    name = param.getTextListParameter().getName();
                    validateText(param.getTextListParameter().getName(), "name");
                    validateText(param.getTextListParameter().getTitle(), "title");
                }
                case LABEL_LIST_PARAMETER -> {
                    name = param.getLabelListParameter().getName();
                    validateText(param.getLabelListParameter().getName(), "name");
                    validateText(param.getLabelListParameter().getTitle(), "title");
                }
                default -> throw invalid("Invalid parameter type", param.getParameterCase());
            }
            if (names.contains(name)) {
                throw invalid("Duplicated parameter %s", name);
            }
            names.add(name);
        }
    }

    private static void validateText(String text, String name) {
        if (StringUtils.isEmpty(text)) {
            throw invalid("Parameter %s must be specified", name);
        }
    }

    private static void ensureExpressionAlertTemplateValid(Expression expression) {
    }

    private static void ensureThresholdAlertTemplateValid(Threshold threshold) {
        for (PredicateRule rule : threshold.getPredicateRulesList()) {
            if (rule.getComparison() == ECompare.UNRECOGNIZED) {
                throw invalid("Unrecognized comparison for rule %s", rule);
            }

            if (rule.getThresholdType() == EThresholdType.UNRECOGNIZED) {
                throw invalid("Unrecognized threshold type for rule %s", rule);
            }

            if (rule.getTargetStatus() == TPredicateRule.ETargetStatus.UNRECOGNIZED) {
                throw invalid("Unrecognized target status for rule %s", rule);
            }

            if (rule.getTargetStatus() == TPredicateRule.ETargetStatus.UNSPECIFIED) {
                throw invalid("Unspecified target status for rule %s", rule);
            }
        }
    }

    private static ValidationException invalid(String message, Object... args) {
        throw new ValidationException(String.format(message, args));
    }

    public static void ensureValid(ListAlertLabelsRequest request) {
        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
    }

    public static void ensureValid(CreateAlertsFromTemplateRequest request) {
        if ("".equals(request.getProjectId())) {
            throw invalid("Project not specified for request:\n%s", request);
        }
        if (StringUtils.isEmpty(request.getServiceProviderId())) {
            throw invalid("Service provider id should be specified:\n%s", request);
        }
        if (!request.getTemplateIdsList().isEmpty()) {
            if (request.getChannelsMap().isEmpty() && request.getTemplateConfigsList().isEmpty()) {
                throw invalid("Channels should be specified:\n%s", request);
            }
        }
        for (var channel : request.getChannelsMap().entrySet()) {
            if (StringUtils.isEmpty(channel.getKey())) {
                throw invalid("Channel id should be specified:\n%s", request);
            }
        }
        if (request.getResourcesList().isEmpty()) {
            throw invalid("Resources should be specified:\n%s", request);
        }
        for (var resource : request.getResourcesList()) {
            if (resource.getResourceParametersMap().isEmpty()) {
                throw invalid("Resource id should be specified:\n%s", request);
            }
        }
    }
}
