package ru.yandex.solomon.alert.api.validators;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.mute.Mute;
import ru.yandex.solomon.alert.protobuf.mute.SelectorsType;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsFormat;

/**
 * @author Ivan Tsybulin
 */
public final class MuteValidator {
    public static void ensureValid(CreateMuteRequest request) {
        if (!request.hasMute()) {
            throw invalid("request doesn't have mute to create: %s", request);
        }

        ensureValidMute(request.getMute());
    }

    public static void ensureValid(UpdateMuteRequest request) {
        if (!request.hasMute()) {
            throw invalid("request doesn't have mute to update: %s", request);
        }

        ensureValidMute(request.getMute());
    }

    public static void ensureValid(ReadMuteRequest request) {
        if (request.getId().isEmpty()) {
            throw invalid("missing mute id in read request: %s", request);
        }

        if (request.getProjectId().isEmpty()) {
            throw invalid("missing project id in read mute request: %s", request);
        }
    }

    public static void ensureValid(DeleteMuteRequest request) {
        if (request.getId().isEmpty()) {
            throw invalid("missing mute id in delete request: %s", request);
        }

        if (request.getProjectId().isEmpty()) {
            throw invalid("missing project id in delete mute request: %s", request);
        }
    }

    public static void ensureValid(ListMutesRequest request) {
        if (request.getProjectId().isEmpty()) {
            throw invalid("missing project id in list mutes request: %s", request);
        }
    }

    private static void ensureValidMute(Mute mute) {
        if (mute.getProjectId().isEmpty()) {
            throw invalid("project is not specified for mute: %s", mute);
        }

        IdValidator.ensureValid(mute.getId(), "mute", true);

        if (StringUtils.isBlank(mute.getName())) {
            throw invalid("mute name is blank");
        }

        if (mute.getToMillis() <= mute.getFromMillis()) {
            throw invalid("expected from_millis < to_millis for mute: %s", mute);
        }

        if (mute.getTtlBase() != 0 && mute.getTtlBase() < mute.getToMillis()) {
            throw invalid("expected ttl_base >= to_millis for mute: %s", mute);
        }

        switch (mute.getTypeCase()) {
            case SELECTORS -> ensureValidSelectorsMute(mute.getSelectors());
            default -> throw invalid("unsupported mute type: %s", mute.getTypeCase());
        }
    }

    private static void ensureValidSelectorsMute(SelectorsType selectorsMute) {
        var alertSelector = selectorsMute.getAlertSelector();
        var labelsSelector = selectorsMute.getLabelSelectors();

        try {
            Selector selector = LabelSelectorConverter.protoToSelector(alertSelector);
            if (!selector.getKey().equals("alert")) {
                throw invalid("expected key 'alert' in alert_selector, got: %s", selector);
            }
            String formatted = SelectorsFormat.format(selector);
            if (!selector.equals(SelectorsFormat.parseSelector(formatted))) {
                throw invalid("invalid selector in mute: %s", selector);
            }
        } catch (Throwable e) {
            throw Status.INVALID_ARGUMENT.withDescription("not valid alert selector: " + alertSelector).withCause(e).asRuntimeException();
        }

        try {
            Selectors selectors = LabelSelectorConverter.protoToSelectors(labelsSelector);
            String formatted = Selectors.format(selectors);
            if (!selectors.equals(Selectors.parse(formatted))) {
                throw invalid("invalid selectors in mute: %s", selectors);
            }
        } catch (Throwable e) {
            throw Status.INVALID_ARGUMENT.withDescription("not valid label selectors: " + labelsSelector).withCause(e).asRuntimeException();
        }
    }

    public static void ensureValid(ReadMuteStatsRequest request) {
        if (request.getProjectId().isEmpty()) {
            throw invalid("missing project id in read mute stats request: %s", request);
        }
    }

    private static StatusRuntimeException invalid(String message, Object... args) {
        return Status.INVALID_ARGUMENT.withDescription(String.format(message, args)).asRuntimeException();
    }
}
