package ru.yandex.solomon.alert.api.validators;

import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.abc.AbcClient;
import ru.yandex.abc.dto.AbcServiceDuty;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.notification.channel.telegram.ChatIdStorage;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TListEscalationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.notification.PhoneType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudPushType;
import ru.yandex.solomon.alert.protobuf.notification.TCloudSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TDatalensEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TEmailType;
import ru.yandex.solomon.alert.protobuf.notification.TJugglerType;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.alert.protobuf.notification.TSmsType;
import ru.yandex.solomon.alert.protobuf.notification.TTelegramType;
import ru.yandex.solomon.alert.protobuf.notification.TWebhookType;
import ru.yandex.solomon.alert.protobuf.notification.TYaChatsType;
import ru.yandex.staff.StaffClient;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public final class NotificationValidator {
    @Nullable
    private final StaffClient staffClient;
    @Nullable
    private final ChatIdStorage chatIdStorage;
    @Nullable
    private final AbcClient abcClient;

    public NotificationValidator(
            @Nullable StaffClient maybeStaffClient,
            @Nullable ChatIdStorage maybeChatIdStorage,
            @Nullable AbcClient abcClient)
    {
        staffClient = maybeStaffClient;
        chatIdStorage = maybeChatIdStorage;
        this.abcClient = abcClient;
    }

    public CompletableFuture<TCreateNotificationRequest> ensureValid(TCreateNotificationRequest request) {
        if (!request.hasNotification()) {
            return invalid("Request doesn't have notification to create: %s", request);
        }

        return ensureValidNotification(request.getNotification())
                .thenApply(aVoid -> request);
    }

    public CompletableFuture<TUpdateNotificationRequest> ensureValid(TUpdateNotificationRequest request) {
        if (!request.hasNotification()) {
            invalid("Request doesn't have notification to update: %s", request);
        }

        return ensureValidNotification(request.getNotification())
                .thenApply(aVoid -> request);
    }

    public CompletableFuture<TReadNotificationRequest> ensureValid(TReadNotificationRequest request) {
        if ("".equals(request.getNotificationId())) {
            return invalid("Request doesn't have notification id to read: %s", request);
        }

        if ("".equals(request.getProjectId())) {
            return invalid("Request doesn't have project: %s", request);
        }

        return completedFuture(request);
    }

    public CompletableFuture<TDeleteNotificationRequest> ensureValid(TDeleteNotificationRequest request) {
        if ("".equals(request.getNotificationId())) {
            return invalid("Request doesn't have notification id to read: %s", request);
        }

        if ("".equals(request.getProjectId())) {
            return invalid("Request doesn't have project: %s", request);
        }

        return completedFuture(request);
    }

    public CompletableFuture<TListNotificationsRequest> ensureValid(TListNotificationsRequest request) {
        if ("".equals(request.getProjectId())) {
            return invalid("Project not specified: %s", request);
        }

        if (request.getPageSize() < 0) {
            return invalid("Page size can't be negative: %s", request);
        }

        return completedFuture(request);
    }

    public CompletableFuture<TResolveNotificationDetailsRequest> ensureValid(TResolveNotificationDetailsRequest request) {
        if ("".equals(request.getProjectId())) {
            return invalid("Request doesn't have project: %s", request);
        }

        if (request.getNotificationIdsList().stream().anyMatch(""::equals)) {
            return invalid("Request has empty notification id: %s", request);
        }

        return completedFuture(request);
    }

    private CompletableFuture<Void> ensureValidNotification(TNotification notification) {
        if ("".equals(notification.getProjectId())) {
            return invalid("No specified project for notification: %s", notification);
        }

        IdValidator.ensureValid(notification.getId(), "notification", true);

        if (StringUtils.isBlank(notification.getName())) {
            return invalid("Channel name is blank");
        }

        return switch (notification.getTypeCase()) {
            case WEBKOOK -> ensureValidWebhookType(notification.getWebkook());
            case PHONE -> ensureValidPhoneType(notification.getPhone());
            case EMAIL -> ensureValidEmailType(notification.getEmail());
            case JUGGLER -> ensureValidJugglerType(notification.getJuggler());
            case SMS -> ensureValidSmsType(notification.getSms());
            case TELEGRAM -> ensureValidTelegramType(notification.getTelegram());
            case CLOUDEMAIL -> ensureValidCloudEmailType(notification.getCloudEmail());
            case CLOUDSMS -> ensureValidCloudSmsType(notification.getCloudSms());
            case YACHATS -> ensureValidYaChatsType(notification.getYaChats());
            case DATALENSEMAIL -> ensureValidDatalensEmailType(notification.getDatalensEmail());
            case CLOUDPUSH -> ensureValidCloudPushType(notification.getCloudPush());
            default -> invalid("Unsupported notification type: %s", notification);
        };
    }

    private static CompletableFuture<Void> ensureValidWebhookType(TWebhookType target) {
        if ("".equals(target.getUrl())) {
            return invalid("No specified required url field for webhook notification: %s", target);
        }
        return completedFuture(null);
    }

    private CompletableFuture<Void> ensureValidPhoneType(PhoneType target) {
        switch (target.getReceiverCase()) {
            case DUTY -> {
                if ("".equals(target.getDuty().getDutySlug())) {
                    return invalid("No specified required duty slug field for phone notification: %s", target);
                }
                if ("".equals(target.getDuty().getAbcService())) {
                    return invalid("No specified required abc service field for phone notification: %s", target);
                }
                if (abcClient == null) {
                    return invalid("Phone notifications are not available");
                }
                return abcClient.getDutyVersion2(target.getDuty().getDutySlug())
                        .thenCompose(abcServiceDuty2 -> {
                            if (abcServiceDuty2.isPresent() && abcServiceDuty2.get().service().getSlug().equals(target.getDuty().getAbcService())) {
                                return CompletableFuture.completedFuture(null);
                            }

                            return abcClient.getServiceDuty(target.getDuty().getAbcService())
                                    .exceptionally(throwable -> {
                                        if (throwable != null) {
                                            var cause = CompletableFutures.unwrapCompletionException(throwable);
                                            throw validationException("Abc service %s is invalid, abc returned exception %s",
                                                    target.getDuty().getAbcService(), cause.getMessage());
                                        }
                                        return null;
                                    })
                                    .thenApply(duties -> {
                                        for (AbcServiceDuty duty : duties) {
                                            if (target.getDuty().getDutySlug().equals(duty.slug())) {
                                                return null;
                                            }
                                        }
                                        throw validationException("Abc service %s doesn't have duty %s",
                                                target.getDuty().getAbcService(), target.getDuty().getDutySlug());
                                    });
                        });
            }
            case LOGIN -> {
                if ("".equals(target.getLogin())) {
                    return invalid("No specified required login field for phone notification: %s", target);
                }
                if (staffClient == null) {
                    return invalid("Phone notifications are not available");
                }
                return staffClient.getUserInfo(target.getLogin())
                        .handle((userInfo, throwable) -> {
                            if (throwable != null) {
                                var cause = CompletableFutures.unwrapCompletionException(throwable);
                                throw validationException("User %s is invalid, staff returned exception %s",
                                        target.getLogin(), cause.getMessage());
                            }
                            return null;
                        });
            }
        }
        invalid("Unsupported receiver type: %s", target.getReceiverCase());
        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidEmailType(TEmailType target) {
        if (target.getRecipientsCount() == 0) {
            return invalid("Recipients email lists can not be empty: %s", target);
        }

        for (String recipient : target.getRecipientsList()) {
            try {
                InternetAddress address = new InternetAddress(recipient);
                address.validate();
            } catch (AddressException address) {
                return invalid("Address " + recipient + " invalid: " + address.getMessage());
            }
        }

        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidCloudEmailType(TCloudEmailType target) {
        if (target.getRecipientsCount() == 0) {
            return invalid("Recipients lists can not be empty: %s", target);
        }
        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidCloudSmsType(TCloudSmsType target) {
        if (target.getRecipientsCount() == 0) {
            return invalid("Recipients lists can not be empty: %s", target);
        }
        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidCloudPushType(TCloudPushType target) {
        if (target.getRecipientsCount() == 0) {
            return invalid("Recipients lists can not be empty: %s", target);
        }
        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidJugglerType(TJugglerType target) {
        // no ops
        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidSmsType(TSmsType target) {
        // No validation options
        return completedFuture(null);
    }

    private CompletableFuture<Void> ensureValidTelegramType(TTelegramType target) {
        if (StringUtils.isBlank(target.getLogin()) && StringUtils.isBlank(target.getGroupTitle())) {
            return invalid("login or groupTitle must be set: %s", target);
        }

        if (StringUtils.isNotBlank(target.getLogin())) {
            if (staffClient == null || chatIdStorage == null) {
                return invalid("Telegram notifications are not available");
            }
            return staffClient.getUserInfo(target.getLogin())
                    .handle((userInfo, throwable) -> {
                        if (throwable != null) {
                            var cause = CompletableFutures.unwrapCompletionException(throwable);
                            throw validationException("User %s is invalid, staff returned exception %s",
                                    target.getLogin(), cause.getMessage());
                        }
                        return userInfo;
                    })
                    .thenApply(userInfo -> {
                        List<String> telegramLogins = userInfo.getTelegramLogins();
                        if (telegramLogins.isEmpty()) {
                            throw validationException("User %s doesn't have telegram login on staff", target.getLogin());
                        }
                        String telegramLogin = telegramLogins.get(0);
                        if (chatIdStorage.getChatIdByTelegramLogin(telegramLogin) == 0) {
                            throw validationException("Telegram login %s (staff login %s) is not known, please say /start to the bot first",
                                    telegramLogin, target.getLogin());
                        }
                        return null;
                    });
        }

        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidYaChatsType(TYaChatsType target) {
        if (StringUtils.isBlank(target.getLogin()) && StringUtils.isBlank(target.getGroupId())) {
            return invalid("login or groupId must be set: %s", target);
        }
        return completedFuture(null);
    }

    private static CompletableFuture<Void> ensureValidDatalensEmailType(TDatalensEmailType datalensEmail) {
        if (datalensEmail.getRecipientsCount() == 0) {
            return invalid("Recipients lists can not be empty: %s", datalensEmail);
        }
        return completedFuture(null);
    }

    private static ValidationException validationException(String message, Object... args) {
        return new ValidationException(String.format(message, args));
    }

    private static <T> CompletableFuture<T> invalid(String message, Object... args) {
        return failedFuture(validationException(message, args));
    }

    public CompletableFuture<Void> ensureValid(TListEscalationsRequest request) {
        if ("".equals(request.getProjectId())) {
            return invalid("No specified project for request: %s", request);
        }
        return completedFuture(null);
    }
}
