package ru.yandex.solomon.alert.charts;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.Instant;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.charts.exceptions.ChartsEmptyResultException;
import ru.yandex.solomon.alert.charts.exceptions.ChartsFailedToFetchException;
import ru.yandex.solomon.alert.charts.exceptions.ChartsServerException;
import ru.yandex.solomon.alert.charts.exceptions.ChartsTooManyRequestsException;
import ru.yandex.solomon.alert.charts.exceptions.ChartsUnknownException;
import ru.yandex.solomon.alert.notification.channel.AlertApiKey;
import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author alexlovkov
 **/
public class ChartsClientImpl implements ChartsClient {
    private static final Logger logger = LoggerFactory.getLogger(ChartsClientImpl.class);
    private static final String URL = "https://charts.yandex-team.ru";
    private static final String userAgent = "solomon-charts-client";
    private static final ObjectMapper mapper = new ObjectMapper();

    private final HttpClient httpClient;
    private final String token;
    private final String env;
    private final Duration timeout;
    private final Metrics metrics;
    private final String widthImage;
    private final String heightImage;

    public ChartsClientImpl(Executor executor, String token, String env, Duration timeout) {
        this(executor, token, env, timeout, MetricRegistry.root());
    }

    public ChartsClientImpl(
        Executor executor,
        String token,
        String env,
        Duration timeout,
        MetricRegistry reg)
    {
        this.httpClient = HttpClient.newBuilder()
                .connectTimeout(timeout)
                .executor(executor)
                .followRedirects(HttpClient.Redirect.NEVER)
                .version(HttpClient.Version.HTTP_1_1)
                .build();
        this.token = token;
        this.env = env;
        this.metrics = new Metrics(reg.subRegistry("endpoint", "chartsScreenshot"));
        this.timeout = timeout;
        this.widthImage = "1280";
        this.heightImage = "720";
    }

    @Override
    public CompletableFuture<byte[]> getScreenshot(AlertApiKey alertApiKey, Instant time) {
        long startTime = System.nanoTime();

        URI url;

        try {
            // https://datalens.yandex-team.ru/editor/qv8q1s86j6r4o
            var uriBuilder = new UriBuilder(URL + "/api/scr/v1/screenshots/preview/qv8q1s86j6r4o");

            if (!Strings.isNullOrEmpty(alertApiKey.subAlertId())) {
                uriBuilder.addParam("subAlertId", alertApiKey.subAlertId());
            }

            uriBuilder
                    .addParam("env", env)
                    .addParam("projectId", alertApiKey.projectId())
                    .addParam("alertId", alertApiKey.alertId())
                    .addParam("time", time.toString())
                    .addParam("_no_header", "1")
                    .addParam("_no_controls", "1")
                    .addParam("__scr_width", widthImage)
                    .addParam("__scr_height", heightImage)
                    .addParam("_embedded", "1")
                    .addParam("__json_error", "1");

            url = uriBuilder.build();
        } catch (IllegalArgumentException e) {
            logger.error("Bad url in getScreenshot: {}", alertApiKey, e);
            return CompletableFuture.failedFuture(e);
        }

        var request = HttpRequest.newBuilder(url)
                .header("Authorization", "OAuth " + token)
                .header("Content-Type", "application/x-www-form-urlencoded")
                .header("X-Request-Id", UUID.randomUUID().toString())
                .header("User-Agent", userAgent)
                .timeout(timeout)
                .GET()
                .build();

        metrics.incStarted();
        return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray())
            .thenApply(httpResponse -> {
                metrics.incStatus(httpResponse.statusCode());

                if (httpResponse.statusCode() != 200) {
                    String response = new String(httpResponse.body(), StandardCharsets.UTF_8);
                    logger.warn("/alert-explanation-chart for {} received not 200 code: {}, {}",
                            alertApiKey, httpResponse.statusCode(), response);
                    throw tryParseResponseToException(httpResponse.statusCode(), response);
                }

                if (httpResponse.body().length == 0) {
                    throw new ChartsEmptyResultException("Empty response from charts", "", "");
                }

                return httpResponse.body();
            }).whenComplete((r, t) -> {
                long elapsedTime = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
                metrics.recordElapsedTimeMs(elapsedTime);
                metrics.incCompleted();
            });
    }

    private RuntimeException tryParseResponseToException(int statusCode, String body) {
        if (HttpStatus.is5xx(statusCode)) {
            return new ChartsServerException(body);
        }
        if (statusCode == HttpStatus.SC_429_TOO_MANY_REQUESTS) {
            return new ChartsTooManyRequestsException(body);
        }
        if (HttpStatus.is4xx(statusCode)) {
            try {
                // try to extract solomon error from charts response
                var chartsResponse = mapper.readValue(body, ChartsErrorResponseDto.class);
                if (chartsResponse.error == null || chartsResponse.error.data == null || chartsResponse.error.data.error == null) {
                    return new ChartsUnknownException(body);
                }
                String code = Nullables.orEmpty(chartsResponse.error.data.error.code);
                var debug = chartsResponse.error.data.error.debug;
                String requestId = debug != null ? Nullables.orEmpty(debug.requestId) : "";
                String traceId = debug != null ? Nullables.orEmpty(debug.traceId) : "";
                if (code.equals("ERR.CK.NO_DATA")) {
                    return new ChartsEmptyResultException("Empty plot from charts", requestId, traceId);
                }
                if (code.equals("ERR.CHARTS.DATA_FETCHING_ERROR")) {
                    return new ChartsFailedToFetchException("Failed to fetch solomon data", requestId, traceId);
                }
            } catch (JsonProcessingException e) {
                // fallthrough
            } catch (IOException e) {
                return new UncheckedIOException(e);
            }
        }
        return new ChartsUnknownException(body);
    }

    private static final class Metrics {

        private final MetricRegistry reg;
        private final Histogram histogramElapsedTimeMs;
        private final Rate callStarted;
        private final Rate callCompleted;

        Metrics(MetricRegistry reg) {
            this.reg = reg;

            this.histogramElapsedTimeMs =
                reg.histogramRate("charts.client.call.elapsedTimeMs", Histograms.exponential(13, 2, 16));
            this.callStarted = reg.rate("charts.client.call.started");
            this.callCompleted = reg.rate("charts.client.call.completed");
            reg.lazyGaugeInt64("charts.client.call.inFlight", () -> callStarted.get() - callCompleted.get());
        }

        void incStatus(int code) {
            reg.rate("charts.client.call", Labels.of("status", String.valueOf(code))).inc();
        }

        void recordElapsedTimeMs(long elapsedTime) {
            histogramElapsedTimeMs.record(elapsedTime);
        }

        void incStarted() {
            callStarted.inc();
        }

        void incCompleted() {
            callCompleted.inc();
        }
    }
}
