package ru.yandex.solomon.alert.cluster;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.Status;

import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.CreateMuteResponse;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteResponse;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsResponse;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsResponse;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TCreateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListEscalationsRequest;
import ru.yandex.solomon.alert.protobuf.TListEscalationsResponse;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsResponse;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsResponse;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionResponse;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteResponse;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.INVALID_REQUEST;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.NOT_FOUND;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.OK;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class AlertingEmptyShard implements AlertingClient {
    public static AlertingEmptyShard INSTANCE = new AlertingEmptyShard();

    private AlertingEmptyShard() {
    }

    @Override
    public CompletableFuture<TCreateAlertResponse> createAlert(TCreateAlertRequest request) {
        return CompletableFuture.failedFuture(Status.NOT_FOUND
                .withDescription("unknown shard: " + request.getAlert().getProjectId())
                .asRuntimeException());
    }

    @Override
    public CompletableFuture<TReadAlertResponse> readAlert(TReadAlertRequest request) {
        return CompletableFuture.completedFuture(TReadAlertResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getAlertId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TReadAlertInterpolatedResponse> readAlert(TReadAlertInterpolatedRequest request) {
        return CompletableFuture.completedFuture(TReadAlertInterpolatedResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getAlertId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TReadSubAlertResponse> readSubAlert(TReadSubAlertRequest request) {
        return CompletableFuture.completedFuture(TReadSubAlertResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getParentId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TUpdateAlertResponse> updateAlert(TUpdateAlertRequest request) {
        return CompletableFuture.completedFuture(TUpdateAlertResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getAlert().getId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TDeleteAlertResponse> deleteAlert(TDeleteAlertRequest request) {
        return CompletableFuture.completedFuture(TDeleteAlertResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getAlertId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TDeletionNotificationResponse> notifyOnDeletionProject(TDeletionNotificationRequest request) {
        return CompletableFuture.completedFuture(TDeletionNotificationResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .build());
    }

    @Override
    public CompletableFuture<TListAlertResponse> listAlerts(TListAlertRequest request) {
        return CompletableFuture.completedFuture(TListAlertResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .build());
    }

    @Override
    public CompletableFuture<TListSubAlertResponse> listSubAlerts(TListSubAlertRequest request) {
        return CompletableFuture.completedFuture(TListSubAlertResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getParentId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TReadEvaluationStateResponse> readEvaluationState(TReadEvaluationStateRequest request) {
        String id = request.getParentId().isEmpty() ? request.getAlertId() : request.getParentId();
        return CompletableFuture.completedFuture(TReadEvaluationStateResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + id + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TReadEvaluationStatsResponse> readEvaluationStats(TReadEvaluationStatsRequest request) {
        return CompletableFuture.completedFuture(TReadEvaluationStatsResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .build());
    }

    @Override
    public CompletableFuture<TExplainEvaluationResponse> explainEvaluation(TExplainEvaluationRequest request) {
        return completedFuture(TExplainEvaluationResponse.newBuilder()
                .setRequestStatus(INVALID_REQUEST)
                .setStatusMessage("Unsupported operation: " + request)
                .build());
    }

    @Override
    public CompletableFuture<TSimulateEvaluationResponse> simulateEvaluation(TSimulateEvaluationRequest request) {
        return completedFuture(TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(INVALID_REQUEST)
                .setStatusMessage("Unsupported operation: " + request)
                .build());
    }

    @Override
    public CompletableFuture<TReadNotificationStateResponse> readNotificationState(TReadNotificationStateRequest request) {
        return completedFuture(TReadNotificationStateResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .build());
    }

    @Override
    public CompletableFuture<TReadNotificationStatsResponse> readNotificationStats(TReadNotificationStatsRequest request) {
        return CompletableFuture.completedFuture(TReadNotificationStatsResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Alert with id " + request.getAlertId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TReadProjectStatsResponse> readProjectStats(TReadProjectStatsRequest request) {
        return completedFuture(TReadProjectStatsResponse.newBuilder()
                .setRequestStatus(OK)
                .build());
    }

    @Override
    public CompletableFuture<UpdateAlertTemplateVersionResponse> updateAlertTemplateVersion(UpdateAlertTemplateVersionRequest request) {
        return failedFuture(Status.NOT_FOUND.withDescription("Alert with id " + request.getAlertData().getAlertId() + " does not exist").asRuntimeException());
    }

    @Override
    public CompletableFuture<TCreateNotificationResponse> createNotification(TCreateNotificationRequest request) {
        return CompletableFuture.failedFuture(Status.NOT_FOUND
                .withDescription("unknown shard: " + request.getNotification().getProjectId())
                .asRuntimeException());
    }

    @Override
    public CompletableFuture<TReadNotificationResponse> readNotification(TReadNotificationRequest request) {
        return completedFuture(TReadNotificationResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Notification with id " + request.getNotificationId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TUpdateNotificationResponse> updateNotification(TUpdateNotificationRequest request) {
        return completedFuture(TUpdateNotificationResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Notification with id " + request.getNotification().getId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<TDeleteNotificationResponse> deleteNotification(TDeleteNotificationRequest request) {
        return completedFuture(TDeleteNotificationResponse.newBuilder()
                .setRequestStatus(NOT_FOUND)
                .setStatusMessage("Notification with id " + request.getNotificationId() + " does not exist")
                .build());
    }

    @Override
    public CompletableFuture<ListAlertLabelsResponse> listAlertLabels(ListAlertLabelsRequest request) {
        return completedFuture(ListAlertLabelsResponse.newBuilder()
                .setRequestStatusCode(ERequestStatusCode.OK)
                .build());
    }

    @Override
    public CompletableFuture<TListNotificationsResponse> listNotification(TListNotificationsRequest request) {
        return completedFuture(TListNotificationsResponse.newBuilder()
                .setRequestStatus(OK)
                .build());
    }

    @Override
    public CompletableFuture<TResolveNotificationDetailsResponse> resolveNotificationDetails(TResolveNotificationDetailsRequest request) {
        return completedFuture(TResolveNotificationDetailsResponse.newBuilder()
                .setRequestStatus(OK)
                .build());
    }

    @Override
    public CompletableFuture<TListEscalationsResponse> listEscalations(TListEscalationsRequest request) {
        return completedFuture(TListEscalationsResponse.newBuilder()
                .setRequestStatus(OK)
                .build());
    }

    @Override
    public CompletableFuture<CreateMuteResponse> createMute(CreateMuteRequest request, long deadlineMillis) {
        return failedFuture(new IllegalStateException("This method should not be called"));
    }

    @Override
    public CompletableFuture<ReadMuteResponse> readMute(ReadMuteRequest request, long deadlineMillis) {
        return failedFuture(Status.NOT_FOUND.withDescription("mute with id " + request.getId() + " does not exist").asRuntimeException());
    }

    @Override
    public CompletableFuture<UpdateMuteResponse> updateMute(UpdateMuteRequest request, long deadlineMillis) {
        return failedFuture(Status.NOT_FOUND.withDescription("mute with id " + request.getMute().getId() + " does not exist").asRuntimeException());
    }

    @Override
    public CompletableFuture<DeleteMuteResponse> deleteMute(DeleteMuteRequest request, long deadlineMillis) {
        return failedFuture(Status.NOT_FOUND.withDescription("mute with id " + request.getId() + " does not exist").asRuntimeException());
    }

    @Override
    public CompletableFuture<ListMutesResponse> listMutes(ListMutesRequest request, long deadlineMillis) {
        return completedFuture(ListMutesResponse.getDefaultInstance());
    }

    @Override
    public CompletableFuture<ReadMuteStatsResponse> readMuteStats(ReadMuteStatsRequest request, long deadlineMillis) {
        return completedFuture(ReadMuteStatsResponse.getDefaultInstance());
    }

    @Override
    public CompletableFuture<CreateAlertsFromTemplateResponse> createAlerts(CreateAlertsFromTemplateRequest request) {
        return completedFuture(CreateAlertsFromTemplateResponse.newBuilder()
                .setRequestStatusCode(INVALID_REQUEST)
                .setStatusMessage("Unsupported operation: " + request)
                .build());
    }

    @Override
    public void close() {
    }
}
