package ru.yandex.solomon.alert.cluster;

import java.time.Clock;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.net.HostAndPort;
import io.grpc.Status;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.cluster.balancer.client.AlertingBalancerClient;
import ru.yandex.solomon.alert.cluster.broker.evaluation.EvaluationAssignmentService;
import ru.yandex.solomon.alert.protobuf.THeartbeatRequest;
import ru.yandex.solomon.alert.protobuf.TNodeSummary;
import ru.yandex.solomon.alert.protobuf.TShardSummary;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.selfmon.ng.ProcSelfMon;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertingHeartbeatProcess {
    private static final Logger logger = LoggerFactory.getLogger(AlertingHeartbeatProcess.class);
    private static final long HEARTBEAT_INTERVAL_MILLIS = TimeUnit.SECONDS.toMillis(5L);

    private final Clock clock;
    private final AlertingLocalShards shards;
    private final EvaluationAssignmentService evaluationAssignmentService;
    private final DistributedLock balancer;
    private final AlertingBalancerClient client;
    private volatile long latestVisibleSeqNo;
    private volatile long expiredAt;
    private final long startedAt;
    private final PingActorRunner actor;

    public AlertingHeartbeatProcess(
        Clock clock,
        AlertingLocalShards shards,
        EvaluationAssignmentService evaluationAssignmentService,
        DistributedLock balancer,
        AlertingBalancerClient client,
        ScheduledExecutorService timer,
        ExecutorService executor)
    {
        this.clock = clock;
        this.shards = shards;
        this.evaluationAssignmentService = evaluationAssignmentService;
        this.balancer = balancer;
        this.client = client;
        this.startedAt = clock.millis();
        this.actor = PingActorRunner.newBuilder()
                .onPing(this::actHeartbeat)
                .operation("heartbeat_to_leader")
                .timer(timer)
                .executor(executor)
                .pingInterval(Duration.ofMillis(HEARTBEAT_INTERVAL_MILLIS))
                .backoffDelay(Duration.ofSeconds(1))
                .backoffMaxDelay(Duration.ofMinutes(1L))
                .build();
    }

    public void start() {
        actor.forcePing();
    }

    public void stop() {
        actor.close();
    }

    public boolean isConnected() {
        return clock.millis() < expiredAt;
    }

    private THeartbeatRequest prepareHeartbeatRequest() {
        return THeartbeatRequest.newBuilder()
            .setNode(HostUtils.getFqdn())
            .setNodeSummary(prepareNodeSummary())
            .addAllShardSummary(prepareShardSummary())
            .setExpiredAt(clock.millis() + HEARTBEAT_INTERVAL_MILLIS)
            .build();
    }

    private TNodeSummary prepareNodeSummary() {
        try {
            return TNodeSummary.newBuilder()
                .setMemoryBytes(ProcSelfMon.getRssBytes())
                .setUtimeMillis(ProcSelfMon.getUtimeMs())
                .setUpTimeMillis(clock.millis() - startedAt)
                .build();
        } catch (Throwable e) {
            // ProcSelfMon can not work on ci https://paste.yandex-team.ru/631268
            logger.error("prepare node summary failed: ", e);
            return TNodeSummary.getDefaultInstance();
        }
    }

    private List<TShardSummary> prepareShardSummary() {
        return shards.stream()
            .map(shard -> TShardSummary.newBuilder()
                .setShardId(shard.getProjectId())
                .setUpTimeMillis(clock.millis() - shard.createdAt)
                .setReady(shard.isReady())
                .setAlertsCount(shard.alertsCount())
                .build())
            .collect(Collectors.toList());
    }

    private CompletableFuture<?> actHeartbeat(int attempt) {
        return balancer.getLockDetail(latestVisibleSeqNo)
            .thenCompose(optOwner -> {
                if (optOwner.isEmpty()) {
                    return CompletableFuture.failedFuture(new StatusRuntimeExceptionNoStackTrace(Status.NOT_FOUND.withDescription("unknown leader")));
                }

                String owner = getAddress(optOwner.get());
                logger.debug("Send heartbeat to {}", owner);
                return client.heartbeat(owner, prepareHeartbeatRequest())
                    .thenApply(response -> {
                        this.latestVisibleSeqNo = response.getLeaderSeqNo();
                        this.expiredAt = response.getExpiredAt();
                        shards.isAssignmentActual(new AssignmentSeqNo(response.getLeaderSeqNo(), response.getGlobalProjectSeqNo()));
                        evaluationAssignmentService.update(response);
                        return null;
                    });
            });
    }

    private String getAddress(LockDetail details) {
        return HostAndPort.fromString(details.owner()).getHost();
    }
}
