package ru.yandex.solomon.alert.cluster;

import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.api.ErrorClassificatory;
import ru.yandex.solomon.alert.api.validators.MuteValidator;
import ru.yandex.solomon.alert.api.validators.NotificationValidator;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.client.MuteApi;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.cluster.balancer.AlertingBalancer;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.cluster.discovery.AlertingTransports;
import ru.yandex.solomon.alert.grpc.GrpcAlertService;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.CreateMuteResponse;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteResponse;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsResponse;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsResponse;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TCreateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListEscalationsRequest;
import ru.yandex.solomon.alert.protobuf.TListEscalationsResponse;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsResponse;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsResponse;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionResponse;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteResponse;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.alert.api.ErrorClassificatory.classifyError;
import static ru.yandex.solomon.alert.api.validators.AlertValidator.ensureValid;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class AlertingShardProxy implements AlertApi, NotificationApi, MuteApi {
    private static final Logger logger = LoggerFactory.getLogger(GrpcAlertService.class);

    private final AlertingLocalShards localShards;
    private final AlertingBalancer balancer;
    private final ClusterDiscovery<AlertingTransports> discovery;
    private final NotificationValidator notificationValidator;

    public AlertingShardProxy(
            AlertingLocalShards localShards,
            AlertingBalancer balancer,
            ClusterDiscovery<AlertingTransports> discovery,
            NotificationValidator notificationValidator)
    {
        this.localShards = localShards;
        this.balancer = balancer;
        this.discovery = discovery;
        this.notificationValidator = notificationValidator;
    }

    @Override
    public CompletableFuture<TCreateAlertResponse> createAlert(TCreateAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getAlert().getProjectId();
                return callWithCreateShard(projectId, AlertingClient::createAlert, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for createAlert {}", code, request, e);

                return TCreateAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadAlertResponse> readAlert(TReadAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readAlert, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readAlert {}", code, request, e);

                return TReadAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadAlertInterpolatedResponse> readAlert(TReadAlertInterpolatedRequest request) {
        return CompletableFuture.completedFuture(request)
                .thenCompose(req -> {
                    ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithoutCreateShard(projectId, AlertingClient::readAlert, req);
                })
                .exceptionally(e -> {
                    ERequestStatusCode code = classifyError(e);
                    logger.error("{} - status for readAlert {}", code, request, e);

                    return TReadAlertInterpolatedResponse.newBuilder()
                            .setRequestStatus(code)
                            .setStatusMessage(e.getMessage())
                            .build();
                });
    }

    @Override
    public CompletableFuture<TReadSubAlertResponse> readSubAlert(TReadSubAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readSubAlert, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readAlert {}", code, request, e);

                return TReadSubAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TUpdateAlertResponse> updateAlert(TUpdateAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getAlert().getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::updateAlert, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for updateAlert {}", code, request, e);

                return TUpdateAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TDeleteAlertResponse> deleteAlert(TDeleteAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::deleteAlert, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for deleteAlert {}", code, request, e);

                return TDeleteAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TDeletionNotificationResponse> notifyOnDeletionProject(TDeletionNotificationRequest request) {
        return CompletableFuture.completedFuture(TDeletionNotificationResponse.getDefaultInstance());
    }

    @Override
    public CompletableFuture<TListAlertResponse> listAlerts(TListAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::listAlerts, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for listAlerts {}", code, request, e);

                return TListAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TListSubAlertResponse> listSubAlerts(TListSubAlertRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::listSubAlerts, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for listSubAlerts {}", code, request, e);

                return TListSubAlertResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadEvaluationStateResponse> readEvaluationState(TReadEvaluationStateRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readEvaluationState, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readEvaluationState {}", code, request, e);

                return TReadEvaluationStateResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadEvaluationStatsResponse> readEvaluationStats(TReadEvaluationStatsRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readEvaluationStats, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readEvaluationState {}", code, request, e);

                return TReadEvaluationStatsResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TExplainEvaluationResponse> explainEvaluation(TExplainEvaluationRequest request) {
        return completedFuture(TExplainEvaluationResponse.newBuilder()
            .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
            .setStatusMessage("Unsupported operation: " + request)
            .build());
    }

    @Override
    public CompletableFuture<TSimulateEvaluationResponse> simulateEvaluation(TSimulateEvaluationRequest request) {
        return completedFuture(TSimulateEvaluationResponse.newBuilder()
            .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
            .setStatusMessage("Unsupported operation: " + request)
            .build());
    }

    @Override
    public CompletableFuture<TReadNotificationStateResponse> readNotificationState(TReadNotificationStateRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readNotificationState, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readNotificationState {}", code, request, e);

                return TReadNotificationStateResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadNotificationStatsResponse> readNotificationStats(TReadNotificationStatsRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readNotificationStats, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readNotificationState {}", code, request, e);

                return TReadNotificationStatsResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadProjectStatsResponse> readProjectStats(TReadProjectStatsRequest request) {
        return CompletableFuture.completedFuture(request)
            .thenCompose(req -> {
                ensureValid(req);
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readProjectStats, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readProjectStats {}", code, request, e);

                return TReadProjectStatsResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<UpdateAlertTemplateVersionResponse> updateAlertTemplateVersion(UpdateAlertTemplateVersionRequest request) {
        return CompletableFuture.completedFuture(request)
                .thenCompose(req -> {
                    ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithoutCreateShard(projectId, AlertingClient::updateAlertTemplateVersion, req);
                })
                .exceptionally(e -> {
                    ERequestStatusCode code = classifyError(e);
                    logger.error("{} - status for updateAlertTemplateVersion {}", code, request, e);

                    return UpdateAlertTemplateVersionResponse.newBuilder()
                            .setRequestStatusCode(code)
                            .setStatusMessage(e.getMessage())
                            .build();
                });
    }

    @Override
    public CompletableFuture<CreateAlertsFromTemplateResponse> createAlerts(CreateAlertsFromTemplateRequest request) {
        return CompletableFuture.completedFuture(request)
                .thenCompose(req -> {
                    ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithCreateShard(projectId, AlertingClient::createAlerts, req);
                })
                .exceptionally(e -> {
                    ERequestStatusCode code = classifyError(e);
                    logger.error("{} - status for createAlerts {}", code, request, e);

                    return CreateAlertsFromTemplateResponse.newBuilder()
                            .setRequestStatusCode(code)
                            .setStatusMessage(e.getMessage())
                            .build();
                });
    }

    @Override
    public CompletableFuture<ListAlertLabelsResponse> listAlertLabels(ListAlertLabelsRequest request) {
        return CompletableFuture.completedFuture(request)
                .thenCompose(req -> {
                    ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithCreateShard(projectId, AlertingClient::listAlertLabels, req);
                })
                .exceptionally(e -> {
                    ERequestStatusCode code = classifyError(e);
                    logger.error("{} - status for listAlertLabels {}", code, request, e);

                    return ListAlertLabelsResponse.newBuilder()
                            .setRequestStatusCode(code)
                            .setStatusMessage(e.getMessage())
                            .build();
                });
    }

    @Override
    public CompletableFuture<TCreateNotificationResponse> createNotification(TCreateNotificationRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
            .thenCompose(req -> {
                var projectId = req.getNotification().getProjectId();
                return callWithCreateShard(projectId, AlertingClient::createNotification, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for createNotification {}", code, request, e);

                return TCreateNotificationResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TReadNotificationResponse> readNotification(TReadNotificationRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
            .thenCompose(req -> {
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::readNotification, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for readNotification {}", code, request, e);

                return TReadNotificationResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TUpdateNotificationResponse> updateNotification(TUpdateNotificationRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
            .thenCompose(req -> {
                var projectId = req.getNotification().getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::updateNotification, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for updateNotification {}", code, request, e);

                return TUpdateNotificationResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TDeleteNotificationResponse> deleteNotification(TDeleteNotificationRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
            .thenCompose(req -> {
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::deleteNotification, req);
            }).exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for deleteNotification {}", code, request, e);

                return TDeleteNotificationResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TListNotificationsResponse> listNotification(TListNotificationsRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
            .thenCompose(req -> {
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::listNotification, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for listNotifications {}", code, request, e);

                return TListNotificationsResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TResolveNotificationDetailsResponse> resolveNotificationDetails(TResolveNotificationDetailsRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
            .thenCompose(req -> {
                var projectId = req.getProjectId();
                return callWithoutCreateShard(projectId, AlertingClient::resolveNotificationDetails, req);
            })
            .exceptionally(e -> {
                ERequestStatusCode code = classifyError(e);
                logger.error("{} - status for listNotifications {}", code, request, e);

                return TResolveNotificationDetailsResponse.newBuilder()
                    .setRequestStatus(code)
                    .setStatusMessage(e.getMessage())
                    .build();
            });
    }

    @Override
    public CompletableFuture<TListEscalationsResponse> listEscalations(TListEscalationsRequest request) {
        return CompletableFutures.safeCall(() -> notificationValidator.ensureValid(request))
                .thenCompose(unused -> {
                    var projectId = request.getProjectId();
                    return callWithoutCreateShard(projectId, AlertingClient::listEscalations, request);
                })
                .exceptionally(e -> {
                    ERequestStatusCode code = classifyError(e);
                    logger.error("{} - status for listEscalations {}", code, request, e);

                    return TListEscalationsResponse.newBuilder()
                            .setRequestStatus(code)
                            .setStatusMessage(e.getMessage())
                            .build();
                });
    }

    @Override
    public CompletableFuture<CreateMuteResponse> createMute(CreateMuteRequest request, long deadline) {
        return completedFuture(request)
                .thenCompose(req -> {
                    MuteValidator.ensureValid(req);
                    var projectId = req.getMute().getProjectId();
                    return callWithCreateShardWithDeadline(projectId, AlertingClient::createMute, req, deadline);
                })
                .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
    }

    @Override
    public CompletableFuture<ReadMuteResponse> readMute(ReadMuteRequest request, long deadline) {
        return completedFuture(request)
                .thenCompose(req -> {
                    MuteValidator.ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithoutCreateShardWithDeadline(projectId, AlertingClient::readMute, req, deadline);
                })
                .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
    }

    @Override
    public CompletableFuture<UpdateMuteResponse> updateMute(UpdateMuteRequest request, long deadline) {
        return completedFuture(request)
                .thenCompose(req -> {
                    MuteValidator.ensureValid(req);
                    var projectId = req.getMute().getProjectId();
                    return callWithoutCreateShardWithDeadline(projectId, AlertingClient::updateMute, req, deadline);
                })
                .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
    }

    @Override
    public CompletableFuture<DeleteMuteResponse> deleteMute(DeleteMuteRequest request, long deadline) {
        return completedFuture(request)
                .thenCompose(req -> {
                    MuteValidator.ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithoutCreateShardWithDeadline(projectId, AlertingClient::deleteMute, req, deadline);
                })
                .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
    }

    @Override
    public CompletableFuture<ListMutesResponse> listMutes(ListMutesRequest request, long deadline) {
        return completedFuture(request)
                .thenCompose(req -> {
                    MuteValidator.ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithoutCreateShardWithDeadline(projectId, AlertingClient::listMutes, req, deadline);
                })
                .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
    }

    @Override
    public CompletableFuture<ReadMuteStatsResponse> readMuteStats(ReadMuteStatsRequest request, long deadline) {
        return completedFuture(request)
                .thenCompose(req -> {
                    MuteValidator.ensureValid(req);
                    var projectId = req.getProjectId();
                    return callWithoutCreateShardWithDeadline(projectId, AlertingClient::readMuteStats, req, deadline);
                })
                .exceptionally(ErrorClassificatory::throwExceptionAsGrpc);
    }

    private <ReqT, ResT> CompletableFuture<ResT> callWithoutCreateShard(
        String shardId,
        BiFunction<AlertingClient, ReqT, CompletableFuture<ResT>> fn,
        ReqT reqT)
    {
        var shard = localShards.getShardById(shardId);
        if (shard != null) {
            return fn.apply(shard, reqT);
        }

        return balancer.getAssignment(shardId)
            .thenCompose(node -> {
                if (Strings.isNullOrEmpty(node)) {
                    return fn.apply(AlertingEmptyShard.INSTANCE, reqT);
                }

                var transport = discovery.getTransportByNode(node).publicApi;
                var remote = new AlertingRemoteShard(transport);
                return fn.apply(remote, reqT);
            });
    }

    private <ReqT, ResT> CompletableFuture<ResT> callWithCreateShard(
            String shardId,
            BiFunction<AlertingClient, ReqT, CompletableFuture<ResT>> fn,
            ReqT reqT)
    {
        var shard = localShards.getShardById(shardId);
        if (shard != null) {
            return fn.apply(shard, reqT);
        }

        return balancer.getOrCreateAssignment(shardId)
                .thenCompose(node -> {
                    var transport = discovery.getTransportByNode(node).publicApi;
                    var remote = new AlertingRemoteShard(transport);
                    return fn.apply(remote, reqT);
                });
    }

    private interface RpcWithDeadline<ReqT, RespT> {
        CompletableFuture<RespT> call(AlertingClient client, ReqT request, long deadline);
    }

    private <ReqT, ResT> CompletableFuture<ResT> callWithoutCreateShardWithDeadline(
            String shardId,
            RpcWithDeadline<ReqT, ResT> fn,
            ReqT reqT,
            long deadline)
    {
        var shard = localShards.getShardById(shardId);
        if (shard != null) {
            return fn.call(shard, reqT, deadline);
        }

        return balancer.getAssignment(shardId)
                .thenCompose(node -> {
                    if (Strings.isNullOrEmpty(node)) {
                        return fn.call(AlertingEmptyShard.INSTANCE, reqT, deadline);
                    }

                    var transport = discovery.getTransportByNode(node).publicApi;
                    var remote = new AlertingRemoteShard(transport);
                    return fn.call(remote, reqT, deadline);
                });
    }

    private <ReqT, ResT> CompletableFuture<ResT> callWithCreateShardWithDeadline(
            String shardId,
            RpcWithDeadline<ReqT, ResT> fn,
            ReqT reqT,
            long deadline)
    {
        var shard = localShards.getShardById(shardId);
        if (shard != null) {
            return fn.call(shard, reqT, deadline);
        }

        return balancer.getOrCreateAssignment(shardId)
                .thenCompose(node -> {
                    var transport = discovery.getTransportByNode(node).publicApi;
                    var remote = new AlertingRemoteShard(transport);
                    return fn.call(remote, reqT, deadline);
                });
    }
}
