package ru.yandex.solomon.alert.cluster.balancer;

import java.time.Clock;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.WillNotClose;

import io.grpc.Status;

import ru.yandex.solomon.alert.cluster.balancer.remote.ProtoConverter;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;
import ru.yandex.solomon.alert.protobuf.THeartbeatRequest;
import ru.yandex.solomon.alert.protobuf.THeartbeatResponse;
import ru.yandex.solomon.alert.protobuf.TProjectAssignmentRequest;
import ru.yandex.solomon.alert.protobuf.TProjectAssignmentResponse;
import ru.yandex.solomon.balancer.Balancer;
import ru.yandex.solomon.balancer.remote.RemoteCluster;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertingBalancerLeader implements AlertingBalancer {
    private final Clock clock;
    private final RemoteCluster cluster;
    @WillNotClose
    public final Balancer balancer;

    public AlertingBalancerLeader(Clock clock, RemoteCluster cluster, @WillNotClose Balancer balancer) {
        this.clock = clock;
        this.cluster = cluster;
        this.balancer = balancer;
    }

    @Override
    public CompletableFuture<THeartbeatResponse> receiveHeartbeat(THeartbeatRequest request) {
        var remoteState = ProtoConverter.convert(request, clock.millis());
        cluster.receiveHeartbeat(remoteState);
        var latestSeqNo = balancer.getLatestSeqNo();
        var nodes = balancer.getNodes();
        var opts = balancer.getOptions();
        return CompletableFuture.completedFuture(THeartbeatResponse.newBuilder()
                .setLeaderSeqNo(latestSeqNo.getLeaderSeqNo())
                .setGlobalProjectSeqNo(latestSeqNo.getAssignSeqNo())
                .addAllMembers(nodes.values()
                        .stream()
                        .map(ProtoConverter::convert)
                        .collect(Collectors.toList()))
                .setExpiredAt(clock.millis() + opts.getHeartbeatExpirationMillis())
                .build());
    }

    @Override
    public CompletableFuture<TProjectAssignmentResponse> getAssignmentSnapshot(TProjectAssignmentRequest request) {
        return balancer.supplyAsync(() -> {
            var latestSeqNo = balancer.getLatestSeqNo();
            var response = TProjectAssignmentResponse.newBuilder()
                    .setSeqNo(TAssignmentSeqNo.newBuilder()
                            .setLeaderSeqNo(latestSeqNo.getLeaderSeqNo())
                            .setProjectSeqNo(latestSeqNo.getAssignSeqNo())
                            .build());

            var shards = balancer.getShards();
            for (var shard : shards.values()) {
                if (shard.getNode() == null || shard.getAssignmentSeqNo() == null) {
                    continue;
                }

                response.addAssignEntries(TProjectAssignmentResponse.Entry.newBuilder()
                        .setProjectId(shard.getShardId())
                        .setAddress(shard.getNode().getAddress())
                        .setSeqNo(TAssignmentSeqNo.newBuilder()
                                .setLeaderSeqNo(shard.getAssignmentSeqNo().getLeaderSeqNo())
                                .setProjectSeqNo(shard.getAssignmentSeqNo().getAssignSeqNo())
                                .build())
                        .build());
            }
            return response.build();
        }).thenApply(response -> {
            if (response.getAssignEntriesCount() == 0) {
                throw Status.ABORTED
                        .withDescription("not initialized yet")
                        .asRuntimeException();
            }

            return response;
        });
    }

    @Override
    public CompletableFuture<String> getOrCreateAssignment(String shardId) {
        return balancer.getOrCreateAssignment(shardId);
    }

    @Override
    public CompletableFuture<String> getAssignment(String shardId) {
        return balancer.getAssignment(shardId);
    }
}
