package ru.yandex.solomon.alert.cluster.balancer;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import com.google.common.collect.Sets;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.Status;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.alert.dao.ShardsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.async.InFlightLimiter;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class ShardsHolderImpl implements ShardsHolder, AutoCloseable {
    private final ProjectsHolder projectsHolder;
    private final DistributedLock lock;
    private final ShardsDao shardsDao;
    private final EntitiesDao<Alert> alerts;
    private final EntitiesDao<Notification> notifications;
    private final EntitiesDao<Mute> mutes;
    private final AlertStatesDao states;
    private final ConcurrentMap<String, String> shardIds = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, CompletableFuture<Void>> deletingShards = new ConcurrentHashMap<>();
    private final InFlightLimiter deleteLimiter = new InFlightLimiter(10);
    private volatile boolean closed;

    public ShardsHolderImpl(
            ProjectsHolder projectsHolder,
            ShardsDao shardsDao,
            EntitiesDao<Alert> alerts,
            EntitiesDao<Notification> notifications,
            EntitiesDao<Mute> mutes,
            AlertStatesDao states,
            DistributedLock lock)
    {
        this.projectsHolder = projectsHolder;
        this.lock = lock;
        this.shardsDao = shardsDao;
        this.alerts = alerts;
        this.notifications = notifications;
        this.mutes = mutes;
        this.states = states;
    }

    @Override
    public CompletableFuture<Void> reload() {
        return projectsHolder.reload()
                .thenCompose(ignore -> shardsDao.createSchemaForTests())
                .thenCompose(ignore -> shardsDao.findAll())
                .thenAccept(ids -> {
                    for (var id : ids) {
                        shardIds.put(id, id);
                    }
                })
                // double check to avoid loose shards until migrate to new schema
                .thenCompose(ignore -> alerts.findProjects())
                .thenCompose(this::syncShardsIds)
                .thenCompose(ignore -> notifications.findProjects())
                .thenCompose(this::syncShardsIds)
                .thenCompose(ignore -> mutes.findProjects())
                .thenAccept(this::syncShardsIds);
    }

    private CompletableFuture<Void> syncShardsIds(Set<String> additional) {
        var it = additional.iterator();
        AsyncActorBody body = () -> {
            while (it.hasNext()) {
                var shardId = it.next();
                if (shardIds.containsKey(shardId)) {
                    continue;
                }

                return add(shardId, false);
            }

            return completedFuture(AsyncActorBody.DONE_MARKER);
        };

        var runner = new AsyncActorRunner(body, MoreExecutors.directExecutor(), 100);
        return runner.start();
    }

    @Override
    public Set<String> getShards() {
        var shards = Sets.difference(shardIds.keySet(), deletingShards.keySet());
        var projects = projectsHolder.getProjects();
        return Sets.intersection(shards, projects);
    }

    @Override
    public CompletableFuture<Void> add(String shardId) {
        return add(shardId, true);
    }

    private CompletableFuture<Void> add(String shardId, boolean shouldExist) {
        return projectsHolder.hasProject(shardId)
                .thenCompose(exists -> {
                    if (!exists && shouldExist) {
                        throw Status.NOT_FOUND.withDescription("unknown shard: " + shardId).asRuntimeException();
                    }

                    var deleting = deletingShards.get(shardId);
                    if (deleting != null) {
                        return deleting.thenCompose(ignore -> add(shardId));
                    }

                    if (shardIds.containsKey(shardId)) {
                        return completedFuture(null);
                    }

                    return shardsDao.insert(shardId).thenAccept(ignore -> shardIds.put(shardId, shardId));
                });
    }

    @Override
    public CompletableFuture<Void> delete(String shardId) {
        if (!shardIds.containsKey(shardId)) {
            return completedFuture(null);
        }

        var deleteFuture = new CompletableFuture<Void>();
        var doneFuture = deleteFuture.whenComplete((ignore, e) -> {
            if (e == null) {
                shardIds.remove(shardId);
            }
            deletingShards.remove(shardId);
        });

        var prevFuture = deletingShards.putIfAbsent(shardId, doneFuture);
        if (prevFuture != null) {
            return prevFuture;
        }

        deleteLimiter.run(() -> {
            if (!lock.isLockedByMe()) {
                deleteFuture.completeExceptionally(new IllegalStateException("Not leader anymore"));
                return deleteFuture;
            }

            if (closed) {
                deleteFuture.completeExceptionally(new IllegalStateException("Already closed"));
                return deleteFuture;
            }

            var deleteAlerts = alerts.deleteProject(shardId);
            var deleteChannels = notifications.deleteProject(shardId);
            var deleteMutes = mutes.deleteProject(shardId);
            var deleteStates = states.deleteProject(shardId);
            var future = CompletableFuture.allOf(deleteAlerts, deleteChannels, deleteMutes, deleteStates)
                    .thenCompose(ignore -> shardsDao.delete(shardId));
            CompletableFutures.whenComplete(future, deleteFuture);
            return deleteFuture;
        });

        return doneFuture;
    }

    @Override
    public void close() {
        closed = true;
    }
}
