package ru.yandex.solomon.alert.cluster.balancer.www;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.springframework.context.annotation.Bean;
import org.springframework.http.MediaType;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.cluster.broker.AlertingProjectShard;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.html.AHref;
import ru.yandex.solomon.staffOnly.html.HtmlWriter;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;
import ru.yandex.solomon.staffOnly.manager.WritableToHtml;
import ru.yandex.solomon.staffOnly.manager.find.NamedObjectId;
import ru.yandex.solomon.staffOnly.manager.table.Column;
import ru.yandex.solomon.staffOnly.manager.table.Table;
import ru.yandex.solomon.staffOnly.manager.table.TableRecord;
import ru.yandex.solomon.util.time.DurationUtils;

import static ru.yandex.solomon.staffOnly.manager.ManagerController.namedObjectLink;

/**
 * @author Sergey Polovko
 */
@RestController
public class AlertingLocalShardsWww {
    private final HttpAuthenticator authenticator;
    private final InternalAuthorizer authorizer;
    private final ManagerWriterContext context;
    private final AlertingLocalShards shards;

    public AlertingLocalShardsWww(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            ManagerWriterContext context,
            AlertingLocalShards shards)
    {
        this.authenticator = authenticator;
        this.authorizer = authorizer;
        this.context = context;
        this.shards = shards;
    }

    @Bean
    public RootLink alertingLocalShardLinks() {
        return new RootLink("/local-shards", "Alerting local shards");
    }

    @RequestMapping(value = {"/alerting-local-shards", "/local-shards"}, produces = MediaType.TEXT_HTML_VALUE)
    public CompletableFuture<String> alertingLocalShards(
        @RequestParam(value = "sortBy", defaultValue = "1") int sortBy,
        ServerHttpRequest request)
    {
        return authenticator.authenticate(request)
            .thenCompose(authorizer::authorize)
            .thenApply(account -> alertingLocalShardsImpl(sortBy));
    }

    private String alertingLocalShardsImpl(int sortBy) {
        List<Record> records = this.shards.stream()
            .map(Record::of)
            .sorted(Comparator.comparing(r -> r.shardId))
            .collect(Collectors.toList());

        List<Column<Record>> columns = makeColumns();
        return new Table<>("Alerting local shards", context, columns, records, sortBy).genString();
    }

    private List<Column<Record>> makeColumns() {
        return List.of(
            Column.of(
                "ShardId",
                r -> {
                    String shardId = r.shardId;
                    return new AHref(namedObjectLink(new NamedObjectId(AlertingProjectShard.class, shardId)), shardId);
                },
                Comparator.comparing(o -> o.shardId)),

            Column.of("Ready", r -> r.ready, Comparator.comparing(o2 -> o2.ready)),
            Column.of("Uptime",
                r -> DurationUtils.formatDurationMillis(r.uptimeMillis),
                Comparator.comparingLong(o2 -> o2.uptimeMillis)
            ),
            Column.of("Alerts",
                r -> DataSize.shortString(r.alerts),
                Comparator.comparingLong(o2 -> o2.alerts)
            ),
            Column.of("Assignment", r -> r.assignment.toString(), Comparator.comparing(r -> r.assignment)),
            Column.of("Kick", r -> {
                return (WritableToHtml) mw -> {
                    var hw = mw.getHtmlWriter();
                    hw.aHref("/balancer/kickShard?shardId=" + r.shardId, () -> {
                        hw.tag("span", HtmlWriter.Attr.cssClass("glyphicon glyphicon-transfer"));
                    });
                };
            }, Comparator.comparing(r -> r.shardId))
        );
    }

    public static class Record implements TableRecord {
        String shardId;
        AssignmentSeqNo assignment;
        boolean ready;
        long uptimeMillis;
        long alerts;

        public static Record of(AlertingProjectShard shard) {
            var r = new Record();
            r.shardId = shard.getProjectId();
            r.assignment = shard.getAssignment().getSeqNo();
            r.ready = shard.isReady();
            r.uptimeMillis = System.currentTimeMillis() - shard.createdAt;
            r.alerts = shard.alertsCount();
            return r;
        }
    }
}
