package ru.yandex.solomon.alert.cluster.broker;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.MoreObjects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.client.AlertingClient;
import ru.yandex.solomon.alert.cluster.broker.alert.ProjectAlertService;
import ru.yandex.solomon.alert.cluster.broker.alert.SnapshotProcess;
import ru.yandex.solomon.alert.cluster.broker.mute.ProjectMuteService;
import ru.yandex.solomon.alert.cluster.broker.notification.ProjectNotificationService;
import ru.yandex.solomon.alert.cluster.project.ProjectAssignment;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.CreateMuteResponse;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteResponse;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsResponse;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsResponse;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TCreateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListEscalationsRequest;
import ru.yandex.solomon.alert.protobuf.TListEscalationsResponse;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsResponse;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsResponse;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionResponse;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteResponse;
import ru.yandex.solomon.balancer.AssignmentSeqNo;

import static java.util.concurrent.CompletableFuture.runAsync;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class AlertingProjectShard implements AlertingClient {
    private static final Logger logger = LoggerFactory.getLogger(AlertingProjectShard.class);

    private final ProjectAssignment assignment;

    private final ProjectNotificationService notificationService;
    private final ProjectAlertService alertService;
    private final ProjectMuteService muteService;
    private final SnapshotProcess snapshotProcess;

    private final ShardMetrics metrics;

    public final long createdAt;

    public AlertingProjectShard(
            ProjectAssignment assignment,
            ProjectNotificationService notificationService,
            ProjectAlertService alertService,
            ProjectMuteService muteService,
            SnapshotProcess snapshotProcess)
    {
        this.assignment = assignment;

        this.notificationService = notificationService;
        this.alertService = alertService;
        this.muteService = muteService;
        this.snapshotProcess = snapshotProcess;

        this.metrics = new ShardMetrics(
            Labels.of("projectId", assignment.getProjectId()),
            notificationService.getMetrics(),
            muteService.getMetrics(),
            alertService.getMetrics(),
            snapshotProcess.getMetrics());

        this.createdAt =  System.currentTimeMillis();
    }

    public ProjectAssignment getAssignment() {
        return assignment;
    }

    public String getProjectId() {
        return assignment.getProjectId();
    }

    public AssignmentSeqNo getSeqNo() {
        return assignment.getSeqNo();
    }

    public boolean isReady() {
        return notificationService.isReady() && alertService.isReady() && muteService.isReady();
    }

    public long alertsCount() {
        return alertService.alertsCount();
    }

    public CompletableFuture<?> run() {
        return notificationService.run()
                .thenCompose(ignore -> muteService.run())
                .thenCompose(ignore -> alertService.run())
                .thenRun(snapshotProcess::run);
    }

    @Override
    public CompletableFuture<TCreateAlertResponse> createAlert(TCreateAlertRequest request) {
        return alertService.createAlert(request);
    }

    @Override
    public CompletableFuture<TReadAlertResponse> readAlert(TReadAlertRequest request) {
        return alertService.readAlert(request);
    }

    @Override
    public CompletableFuture<TReadAlertInterpolatedResponse> readAlert(TReadAlertInterpolatedRequest request) {
        return alertService.readAlert(request);
    }

    @Override
    public CompletableFuture<TReadSubAlertResponse> readSubAlert(TReadSubAlertRequest request) {
        return alertService.readSubAlert(request);
    }

    @Override
    public CompletableFuture<TUpdateAlertResponse> updateAlert(TUpdateAlertRequest request) {
        return alertService.updateAlert(request);
    }

    @Override
    public CompletableFuture<TDeleteAlertResponse> deleteAlert(TDeleteAlertRequest request) {
        return alertService.deleteAlert(request);
    }

    @Override
    public CompletableFuture<TDeletionNotificationResponse> notifyOnDeletionProject(TDeletionNotificationRequest request) {
        return alertService.notifyOnDeletionProject(request);
    }

    @Override
    public CompletableFuture<TListAlertResponse> listAlerts(TListAlertRequest request) {
        return alertService.listAlerts(request);
    }

    @Override
    public CompletableFuture<TListSubAlertResponse> listSubAlerts(TListSubAlertRequest request) {
        return alertService.listSubAlerts(request);
    }

    @Override
    public CompletableFuture<TReadEvaluationStateResponse> readEvaluationState(TReadEvaluationStateRequest request) {
        return alertService.readEvaluationState(request);
    }

    @Override
    public CompletableFuture<TReadEvaluationStatsResponse> readEvaluationStats(TReadEvaluationStatsRequest request) {
        return alertService.readEvaluationStats(request);
    }

    @Override
    public CompletableFuture<TExplainEvaluationResponse> explainEvaluation(TExplainEvaluationRequest request) {
        return alertService.explainEvaluation(request);
    }

    @Override
    public CompletableFuture<TSimulateEvaluationResponse> simulateEvaluation(TSimulateEvaluationRequest request) {
        return alertService.simulateEvaluation(request);
    }

    @Override
    public CompletableFuture<TReadNotificationStateResponse> readNotificationState(TReadNotificationStateRequest request) {
        return alertService.readNotificationState(request);
    }

    @Override
    public CompletableFuture<TReadNotificationStatsResponse> readNotificationStats(TReadNotificationStatsRequest request) {
        return alertService.readNotificationStats(request);
    }

    @Override
    public CompletableFuture<TReadProjectStatsResponse> readProjectStats(TReadProjectStatsRequest request) {
        return alertService.readProjectStats(request);
    }

    @Override
    public CompletableFuture<UpdateAlertTemplateVersionResponse> updateAlertTemplateVersion(UpdateAlertTemplateVersionRequest request) {
        return alertService.updateAlertTemplateVersion(request);
    }

    @Override
    public CompletableFuture<CreateAlertsFromTemplateResponse> createAlerts(CreateAlertsFromTemplateRequest request) {
        return alertService.createAlerts(request);
    }

    @Override
    public CompletableFuture<TCreateNotificationResponse> createNotification(TCreateNotificationRequest request) {
        return notificationService.createNotification(request);
    }

    @Override
    public CompletableFuture<TReadNotificationResponse> readNotification(TReadNotificationRequest request) {
        return notificationService.readNotification(request);
    }

    @Override
    public CompletableFuture<TUpdateNotificationResponse> updateNotification(TUpdateNotificationRequest request) {
        return notificationService.updateNotification(request);
    }

    @Override
    public CompletableFuture<TDeleteNotificationResponse> deleteNotification(TDeleteNotificationRequest request) {
        return notificationService.deleteNotification(request);
    }

    @Override
    public CompletableFuture<TListNotificationsResponse> listNotification(TListNotificationsRequest request) {
        return notificationService.listNotification(request);
    }

    @Override
    public CompletableFuture<TResolveNotificationDetailsResponse> resolveNotificationDetails(TResolveNotificationDetailsRequest request) {
        return notificationService.resolveNotificationDetails(request);
    }

    @Override
    public CompletableFuture<TListEscalationsResponse> listEscalations(TListEscalationsRequest request) {
        return notificationService.listEscalations(request);
    }

    @Override
    public CompletableFuture<CreateMuteResponse> createMute(CreateMuteRequest request, long deadline) {
        return muteService.createMute(request);
    }

    @Override
    public CompletableFuture<ReadMuteResponse> readMute(ReadMuteRequest request, long deadline) {
        return muteService.readMute(request);
    }

    @Override
    public CompletableFuture<UpdateMuteResponse> updateMute(UpdateMuteRequest request, long deadline) {
        return muteService.updateMute(request);
    }

    @Override
    public CompletableFuture<DeleteMuteResponse> deleteMute(DeleteMuteRequest request, long deadline) {
        return muteService.deleteMute(request);
    }

    @Override
    public CompletableFuture<ListMutesResponse> listMutes(ListMutesRequest request, long deadline) {
        return muteService.listMutes(request);
    }

    @Override
    public CompletableFuture<ReadMuteStatsResponse> readMuteStats(ReadMuteStatsRequest request, long deadline) {
        return muteService.readMuteStats(request);
    }

    @Override
    public CompletableFuture<ListAlertLabelsResponse> listAlertLabels(ListAlertLabelsRequest request) {
        return alertService.listAlertLabels(request);
    }

    public void forceShutdown() {
        alertService.close();
        notificationService.close();
        snapshotProcess.cancel();
    }

    public CompletableFuture<?> gracefulShutdown() {
        return CompletableFuture.allOf(runAsync(alertService::close), runAsync(notificationService::close))
            .thenCompose(ignore -> snapshotProcess.flush())
            .whenComplete((ignore, ignore2) -> {
                snapshotProcess.cancel();
                logger.info("{} was graceful shutdown", assignment.getProjectId());
            });
    }

    public ShardMetrics getMetrics() {
        return metrics;
    }

    public void appendMetrics(MetricConsumer consumer) {
        alertService.appendAlertMetrics(consumer);
        notificationService.appendNotificationMetrics(consumer);
    }

    @Override
    public void close() {
        gracefulShutdown().join();
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("assignment", assignment)
                .toString();
    }
}
