package ru.yandex.solomon.alert.cluster.broker.alert;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;

/**
 * @author Alexey Trushkin
 */
public class AlertPostInitializerImpl implements AlertPostInitializer {

    private final AlertTemplateDao alertTemplateDao;
    private final AlertTemplateLastVersionDao alertTemplateLastVersionDao;

    public AlertPostInitializerImpl(
            AlertTemplateDao alertTemplateDao,
            AlertTemplateLastVersionDao alertTemplateLastVersionDao)
    {
        this.alertTemplateDao = alertTemplateDao;
        this.alertTemplateLastVersionDao = alertTemplateLastVersionDao;
    }

    @Override
    public CompletableFuture<Alert> initializeCreate(Alert alert) {
        switch (alert.getAlertType()) {
            case FROM_TEMPLATE -> {
                return initializeCreateAlertFromTemplate((AlertFromTemplatePersistent) alert);
            }
            default -> {
                return CompletableFuture.completedFuture(alert);
            }
        }
    }

    @Override
    public CompletableFuture<Alert> initializeVersionUpdate(Alert alert) {
        return initializeCreate(alert);
    }

    private CompletableFuture<Alert> initializeCreateAlertFromTemplate(AlertFromTemplatePersistent alert) {
        return alertTemplateDao.findById(alert.getTemplateId(), alert.getTemplateVersionTag())
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        throw new IllegalArgumentException("Hasn't alert template with id " + alert.getTemplateId() + " and version " + alert.getTemplateVersionTag());
                    }
                    return alert.toBuilder()
                            .setServiceProviderAnnotations(alertTemplate.get().getAnnotations())
                            .build();
                });
    }

    @Override
    public CompletableFuture<List<AlertFromTemplatePersistent>> initializeTemplateAlertsFromPublishedTemplates(
            List<AlertFromTemplatePersistent> alerts,
            String serviceProviderId)
    {
        return alertTemplateLastVersionDao.find(serviceProviderId, "", 1000, "0")
                .thenCompose(page -> alertTemplateDao.findVersions(page.getItems()).thenApply(alertTemplates -> {
                    var map = alertTemplates.stream().collect(Collectors.toMap(AlertTemplate::getId, Function.identity()));
                    return alerts.stream()
                            .map(alert -> {
                                AlertTemplate template = map.get(alert.getTemplateId());
                                if (template == null) {
                                    throw new IllegalArgumentException("Template " + alert.getTemplateId() + " not published");
                                }
                                if (!resourceTypeAccepted(alert, template)) {
                                    return null;
                                }
                                return alert.toBuilder()
                                        .setTemplateVersionTag(template.getTemplateVersionTag())
                                        .setName(template.getName() + "(" + String.join(", ", alert.getLabels().values()) + ")")
                                        .setDescription(template.getDescription())
                                        .setServiceProviderAnnotations(template.getAnnotations())
                                        .setParameters(template.getParameters().stream()
                                                .map(parameter -> {
                                                    var value = alert.getLabels().get(parameter.getName());
                                                    if (value == null) {
                                                        throw new IllegalArgumentException("Alert hasn't parameter " + parameter.getName() + " from template " + alert.getTemplateId());
                                                    }
                                                    switch (parameter.getType()) {
                                                        case TEXT -> {
                                                            return new AlertParameter.TextParameterValue(value, parameter.getName());
                                                        }
                                                        case TEXT_LIST -> {
                                                            return new AlertParameter.TextListParameterValue(List.of(value), parameter.getName());
                                                        }
                                                        case LABEL_LIST -> {
                                                            return new AlertParameter.LabelListParameterValue(List.of(value), parameter.getName());
                                                        }
                                                        default -> throw new UnsupportedOperationException(parameter.getType() + " not supported");
                                                    }
                                                })
                                                .collect(Collectors.toList()))
                                        .build();
                            })
                            .filter(Objects::nonNull)
                            .collect(Collectors.toList());
                }));
    }

    private boolean resourceTypeAccepted(AlertFromTemplatePersistent alert, AlertTemplate template) {
        var resourceType = alert.getLabels().getOrDefault("resourceType", "all");
        var templateResourceType = new HashSet<>(Arrays.asList(template.getLabels().getOrDefault("resourceType", "all").split("\\|")));
        if (templateResourceType.contains("all")) {
            return true;
        }
        return templateResourceType.contains(resourceType);
    }
}
