package ru.yandex.solomon.alert.cluster.broker.alert;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.ActivityMetrics;
import ru.yandex.solomon.alert.cluster.broker.quota.AlertingProjectQuota;
import ru.yandex.solomon.alert.cluster.broker.quota.QuotaMetrics;
import ru.yandex.solomon.alert.domain.AlertType;

/**
 * @author Vladimir Gordiychuk
 */
public class AlertServiceMetrics implements MetricSupplier {
    private final Runnable actualizeAll;
    private final Runnable actualizeCountAlerts;

    private final MetricRegistry registry;
    private final GaugeInt64 countAlerts;
    private final GaugeInt64 countSubAlerts;
    private final QuotaMetrics quotaMetrics;
    private final GaugeInt64 failedAlertsFromTemplate;
    private volatile ActivityMetrics activityMetrics;
    private final ConcurrentMap<Labels, GaugeInt64> fromTemplateCounter = new ConcurrentHashMap<>();

    public AlertServiceMetrics() {
        this(() -> { /* no-op */ }, () -> { /* no-op */ });
    }

    AlertServiceMetrics(Runnable actualizeAll, Runnable actualizeCountAlerts) {
        this.actualizeAll = actualizeAll;
        this.actualizeCountAlerts = actualizeCountAlerts;
        this.registry = new MetricRegistry();
        this.activityMetrics = ActivityMetrics.empty();
        this.countAlerts = this.registry.gaugeInt64("broker.shard.alerts.count");
        this.countSubAlerts = this.registry.gaugeInt64("broker.shard.subAlerts.count");
        this.failedAlertsFromTemplate = this.registry.gaugeInt64("broker.shard.alerts.loading_error", Labels.of("alert_type", AlertType.FROM_TEMPLATE.name()));
        this.quotaMetrics = new QuotaMetrics(registry);
    }

    void setCountAlerts(long count) {
        this.countAlerts.set(count);
    }

    void setCountSubAlerts(long count) {
        this.countSubAlerts.set(count);
    }

    void setActivityMetrics(ActivityMetrics metrics) {
        this.activityMetrics = metrics;
    }

    void setFailedLoadingAlertsCount(int count) {
        failedAlertsFromTemplate.set(count);
    }

    void updateQuotaMetrics(AlertingProjectQuota quota) {
        this.quotaMetrics.update(quota);
    }

    public void combine(AlertServiceMetrics metrics) {
        countAlerts.combine(metrics.countAlerts);
        countSubAlerts.combine(metrics.countSubAlerts);
        activityMetrics.combine(metrics.activityMetrics);
        failedAlertsFromTemplate.combine(metrics.failedAlertsFromTemplate);
        metrics.fromTemplateCounter.forEach((key, value) -> {
            GaugeInt64 gaugeInt64 = fromTemplateCounter.get(key);
            if (gaugeInt64 == null) {
                GaugeInt64 gaugeInt641 = registry.gaugeInt64("broker.shard.alerts.from_template.count", key);
                gaugeInt641.set(value.get());
                fromTemplateCounter.put(key, gaugeInt641);
            } else {
                gaugeInt64.combine(value);
            }
        });
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount() + activityMetrics.estimateCount();
    }

    public boolean hasAlerts() {
        actualizeCountAlerts.run();
        return countAlerts.get() > 0;
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        actualizeAll.run();
        registry.append(tsMillis, commonLabels, consumer);
        activityMetrics.append(tsMillis, commonLabels, consumer);
    }

    void incrementAlertCount(String templateId, String createdBy, int count, String serviceProvider) {
        Labels commonLabels = Labels.of("template_id", templateId,
                "created_by_service_provider", getValue(createdBy, "byUser"),
                "template_service_provider", serviceProvider);
        fromTemplateCounter.computeIfAbsent(commonLabels, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabels)).add(count);

        Labels commonLabelsTotalTemplate = Labels.of("template_id", "total",
                "created_by_service_provider", getValue(createdBy, "byUser"),
                "template_service_provider", serviceProvider);
        fromTemplateCounter.computeIfAbsent(commonLabelsTotalTemplate, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotalTemplate)).add(count);

        Labels commonLabelsTotalSp = Labels.of("template_id", templateId,
                "created_by_service_provider", "total",
                "template_service_provider", serviceProvider);
        fromTemplateCounter.computeIfAbsent(commonLabelsTotalSp, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotalSp)).add(count);

        Labels commonLabelsTotal = Labels.of("template_id", "total",
                "created_by_service_provider", "total",
                "template_service_provider", serviceProvider);
        fromTemplateCounter.computeIfAbsent(commonLabelsTotal, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotal)).add(count);

        Labels commonLabelsTotal1 = Labels.of("template_id", templateId,
                "created_by_service_provider",  getValue(createdBy, "byUser"),
                "template_service_provider", "total");
        fromTemplateCounter.computeIfAbsent(commonLabelsTotal1, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotal1)).add(count);

        Labels commonLabelsTotal2 = Labels.of("template_id", templateId,
                "created_by_service_provider",  "total",
                "template_service_provider", "total");
        fromTemplateCounter.computeIfAbsent(commonLabelsTotal2, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotal2)).add(count);

        Labels commonLabelsTotal3 = Labels.of("template_id", "total",
                "created_by_service_provider",  getValue(createdBy, "byUser"),
                "template_service_provider", "total");
        fromTemplateCounter.computeIfAbsent(commonLabelsTotal3, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotal3)).add(count);

        Labels commonLabelsTotal4 = Labels.of("template_id", "total",
                "created_by_service_provider",  "total",
                "template_service_provider", "total");
        fromTemplateCounter.computeIfAbsent(commonLabelsTotal4, labels -> registry.gaugeInt64("broker.shard.alerts.from_template.count", commonLabelsTotal4)).add(count);

    }

    private String getValue(String serviceProvider, String defValue) {
        return StringUtils.isEmpty(serviceProvider) ? defValue : serviceProvider;
    }

    void clearFromTemplateCount() {
        fromTemplateCounter.values().forEach(gaugeInt64 -> gaugeInt64.set(0));
    }
}
