package ru.yandex.solomon.alert.cluster.broker.alert;

import java.time.Clock;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Throwables;
import com.google.common.util.concurrent.MoreExecutors;
import io.grpc.Status;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.api.converters.NotificationConverter;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.ActivityFactory;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.ActivityMetrics;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.AlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.EvaluationSummaryStatistics;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.MultiAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.NotificationSummaryStatistics;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SimpleAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SubAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SummaryStatisticsConverter;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.TemplateAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.search.ActivitySearch;
import ru.yandex.solomon.alert.cluster.broker.quota.AlertingProjectQuota;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertLabelsResponse;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TCreateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TCreateAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertResponse;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeletionNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TEvaluationStats;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TExplainEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TNotificationState;
import ru.yandex.solomon.alert.protobuf.TNotificationStats;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertInterpolatedResponse;
import ru.yandex.solomon.alert.protobuf.TReadAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadAlertResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadEvaluationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStateResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsRequest;
import ru.yandex.solomon.alert.protobuf.TReadProjectStatsResponse;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TReadSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationRequest;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateAlertResponse;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionResponse;
import ru.yandex.solomon.alert.rule.AlertProcessingState;
import ru.yandex.solomon.core.container.ContainerType;
import ru.yandex.solomon.idempotency.IdempotentOperation;
import ru.yandex.solomon.idempotency.IdempotentOperationExistException;
import ru.yandex.solomon.idempotency.IdempotentOperationService;
import ru.yandex.solomon.quotas.watcher.QuotaWatcher;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.collection.Nullables;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.runAsync;
import static ru.yandex.solomon.alert.cluster.broker.alert.AlertingIdempotency.CREATE_OPERATION_TYPE;
import static ru.yandex.solomon.alert.cluster.broker.alert.AlertingIdempotency.DELETE_OPERATION_TYPE;
import static ru.yandex.solomon.alert.cluster.broker.alert.AlertingIdempotency.UPDATE_OPERATION_TYPE;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncCreateAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncDeleteAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncListAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncListAlertLabels;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncListSubAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadEvaluationState;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadEvaluationStats;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadInterpolatedAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadNotificationState;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadNotificationStats;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncReadSubAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncUpdateAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyAsyncUpdateAlertTemplateVersion;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyCreateAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyCreateAlertFromOperation;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyDeleteAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyUpdateAlert;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyUpdateAlertFromOperation;
import static ru.yandex.solomon.alert.cluster.broker.alert.ProtoReply.replyUpdateAlertTemplateVersion;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.CONCURRENT_MODIFICATION;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.INVALID_REQUEST;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.NOT_FOUND;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.OK;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.SHARD_NOT_INITIALIZED;

/**
 * @author Vladimir Gordiychuk
 */
public class ProjectAlertService implements AlertApi, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ProjectAlertService.class);

    private static final String TOTAL_FOLDER = "";

    private final Clock clock;
    private final String projectId;
    private final EntitiesDao<Alert> alertDao;
    private final AlertStatesDao stateDao;
    private final ActivityFactory activityFactory;
    private final ConcurrentMap<String, AlertActivity> activityById = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, AlertActivity> failedLoadingAlerts = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, AtomicInteger> alertLabels = new ConcurrentHashMap<>();
    private final AtomicReference<State> state = new AtomicReference<>(State.IDLE);
    private final AlertFromTemplateMetrics alertFromTemplateMetrics;
    private final ScheduledExecutorService timer;
    private final NotificationConverter notificationConverter;
    private final ActivitySearch activitySearch;
    private final AlertPostInitializer postInitializer;
    private final Executor executor;
    private final IdempotentOperationService idempotentOperationService;
    private final AlertingProjectQuota quota;
    private final ProjectAlertServiceValidator projectAlertServiceValidator;
    private final AlertServiceMetrics metrics;
    private final ForkJoinPool selfMonFjp;

    private volatile Status lastInitStatus = Status.OK;
    private volatile ScheduledFuture<?> refreshFuture;
    private volatile Map<String, EvaluationSummaryStatistics> evaluationSummaryStatisticsByFolder;
    private volatile Map<String, EvaluationSummaryStatistics> mutedSummaryStatisticsByFolder;
    private volatile Map<String, NotificationSummaryStatistics> notificationSummaryStatisticsByFolder;
    private volatile Map<String, Integer> alertsCountByFolder;

    public ProjectAlertService(
            String projectId,
            Clock clock,
            EntitiesDao<Alert> alertDao,
            AlertStatesDao statesDaoV2,
            ActivityFactory activityFactory,
            ScheduledExecutorService timer,
            NotificationConverter notificationConverter,
            ActivitySearch activitySearch,
            QuotaWatcher quotaWatcher,
            ProjectAlertServiceValidator projectAlertServiceValidator,
            AlertPostInitializer postInitializer,
            Executor executor,
            MetricRegistry metricRegistry,
            IdempotentOperationService idempotentOperationService,
            ForkJoinPool selfMonFjp)
    {
        this.projectId = projectId;
        this.clock = clock;
        this.alertDao = alertDao;
        this.stateDao = statesDaoV2;
        this.activityFactory = activityFactory;
        this.timer = timer;
        this.notificationConverter = notificationConverter;
        this.activitySearch = activitySearch;
        this.postInitializer = postInitializer;
        this.executor = executor;
        this.idempotentOperationService = idempotentOperationService;
        this.quota = new AlertingProjectQuota(quotaWatcher, projectId);
        this.projectAlertServiceValidator = projectAlertServiceValidator;
        this.metrics = new AlertServiceMetrics(this::actualizeMetrics, this::actualizeCountAlerts);
        this.selfMonFjp = selfMonFjp;
        this.alertFromTemplateMetrics = new AlertFromTemplateMetrics(metricRegistry);
    }

    public void appendAlertMetrics(MetricConsumer consumer) {
        if (!isReady()) {
            return;
        }

        if (activityById.isEmpty()) {
            return;
        }

        for (AlertActivity activity : activityById.values()) {
            activity.appendAlertMetrics(consumer);
        }

        ensureSummaryStatisticCalculated();
        EvaluationSummaryStatistics evaluationSummary = evaluationSummaryStatisticsByFolder.get(TOTAL_FOLDER);
        EvaluationSummaryStatistics mutedSummary = mutedSummaryStatisticsByFolder.get(TOTAL_FOLDER);
        NotificationSummaryStatistics notificationSummary = notificationSummaryStatisticsByFolder.get(TOTAL_FOLDER);

        for (EvaluationStatus.Code code : EvaluationStatus.Code.values()) {
            consumer.onMetricBegin(MetricType.IGAUGE);
            consumer.onLabelsBegin(3);
            consumer.onLabel("sensor", "evaluation.status.summary");
            consumer.onLabel("projectId", projectId);
            consumer.onLabel("status", code.name());
            consumer.onLabelsEnd();
            consumer.onLong(0, evaluationSummary.getCount(code));
            consumer.onMetricEnd();
        }

        for (EvaluationStatus.Code code : EvaluationStatus.Code.values()) {
            consumer.onMetricBegin(MetricType.IGAUGE);
            consumer.onLabelsBegin(3);
            consumer.onLabel("sensor", "muted.status.summary");
            consumer.onLabel("projectId", projectId);
            consumer.onLabel("status", code.name());
            consumer.onLabelsEnd();
            consumer.onLong(0, mutedSummary.getCount(code));
            consumer.onMetricEnd();
        }

        for (NotificationStatus.Code code : NotificationStatus.Code.values()) {
            consumer.onMetricBegin(MetricType.IGAUGE);
            consumer.onLabelsBegin(3);
            consumer.onLabel("sensor", "notification.status.summary");
            consumer.onLabel("projectId", projectId);
            consumer.onLabel("status", code.name());
            consumer.onLabelsEnd();
            consumer.onLong(0, notificationSummary.getCount(code));
            consumer.onMetricEnd();
        }
    }

    private void ensureSummaryStatisticCalculated() {
        if (evaluationSummaryStatisticsByFolder == null || notificationSummaryStatisticsByFolder == null
                || mutedSummaryStatisticsByFolder == null) {
            refreshSummaryStatisticsInSelfMonFjp().join();
        }
    }

    private void scheduledRefreshStatistics() {
        refreshSummaryStatisticsInSelfMonFjp()
            .whenComplete((i, t) -> {
                if (state.get() != State.CLOSED) {
                    refreshFuture = timer.schedule(this::scheduledRefreshStatistics, 15, TimeUnit.SECONDS);
                }
            });
    }

    private void appendStatsForFolder(
            String folderId,
            HashMap<String, EvaluationSummaryStatistics> evaluationSummaryByFolder,
            HashMap<String, EvaluationSummaryStatistics> mutedSummaryByFolder,
            HashMap<String, NotificationSummaryStatistics> notificationSummaryByFolder,
            AlertActivity activity)
    {
        var evaluationSummary = evaluationSummaryByFolder.computeIfAbsent(folderId, ignore -> new EvaluationSummaryStatistics());
        var mutedSummary = mutedSummaryByFolder.computeIfAbsent(folderId, ignore -> new EvaluationSummaryStatistics());
        var notificationSummary = notificationSummaryByFolder.computeIfAbsent(folderId, ignore -> new NotificationSummaryStatistics());
        activity.appendEvaluationStatistics(evaluationSummary);
        activity.appendMutedStatistics(mutedSummary, Collections.emptySet(), null);
        activity.appendNotificationStatistics(notificationSummary, Collections.emptySet());
    }

    private CompletableFuture<Void> refreshSummaryStatisticsInSelfMonFjp() {
        return runAsync(this::refreshSummaryStatistics, selfMonFjp);
    }

    private void refreshSummaryStatistics() {
        if (!isReady()) {
            return;
        }

        HashMap<String, EvaluationSummaryStatistics> evaluationSummaryByFolder = new HashMap<>();
        HashMap<String, EvaluationSummaryStatistics> mutedSummaryByFolder = new HashMap<>();
        HashMap<String, NotificationSummaryStatistics> notificationSummaryByFolder = new HashMap<>();
        Map<String, Integer> countByFolder = new Object2IntOpenHashMap<>();

        evaluationSummaryByFolder.put(TOTAL_FOLDER, new EvaluationSummaryStatistics());
        notificationSummaryByFolder.put(TOTAL_FOLDER, new NotificationSummaryStatistics());

        for (AlertActivity activity : activityById.values()) {
            String folderId = activity.getAlert().getFolderId();
            if (!folderId.isEmpty()) {
                appendStatsForFolder(folderId, evaluationSummaryByFolder, mutedSummaryByFolder, notificationSummaryByFolder, activity);
            }
            appendStatsForFolder(TOTAL_FOLDER, evaluationSummaryByFolder, mutedSummaryByFolder, notificationSummaryByFolder, activity);
            countByFolder.compute(folderId, (ignore, oldValue) -> oldValue == null ? 1 : oldValue + 1);
        }

        evaluationSummaryStatisticsByFolder = evaluationSummaryByFolder;
        mutedSummaryStatisticsByFolder = mutedSummaryByFolder;
        notificationSummaryStatisticsByFolder = notificationSummaryByFolder;
        alertsCountByFolder = countByFolder;
    }

    public AlertServiceMetrics getMetrics() {
        return metrics;
    }

    private void actualizeMetrics() {
        metrics.setCountAlerts(activityById.size());
        metrics.setCountSubAlerts(activityById.values()
                .stream()
                .mapToLong(AlertActivity::countSubAlerts)
                .sum());
        metrics.setFailedLoadingAlertsCount(failedLoadingAlerts.size());

        selfMonFjp.submit(() -> {
            var nowMillis = System.currentTimeMillis();
            var activityMetrics = activityById.values()
                .parallelStream()
                .collect(ActivityMetrics.collector(nowMillis));
            metrics.setActivityMetrics(activityMetrics);
        }).join();

        metrics.updateQuotaMetrics(quota);
    }

    private void actualizeCountAlerts() {
        metrics.setCountAlerts(activityById.size());
    }

    public long alertsCount() {
        long count = 0;
        for (var activity : activityById.values()) {
            count += 1 + activity.countSubAlerts();
        }
        return count;
    }

    public boolean isReady() {
        return state.get() == State.RUNNING;
    }

    public CompletableFuture<?> run() {
        CompletableFuture<?> init = new CompletableFuture<>();
        CompletableFuture<?> result = init
                .thenCompose(ignore -> createSchema(State.IDLE))
                .thenCompose(ignore -> fetchAlerts(State.SCHEME_CREATING))
                .thenCompose(ignore -> fetchStates(State.FETCH_ALERTS))
                .thenCompose(ignore -> startAllActivities(State.FETCH_STATE))
                .thenAccept(ignore -> {
                    State current = state.get();
                    if (current == State.CLOSED || current == State.RUNNING) {
                        return;
                    }

                    if (!changeState(State.START_ACTIVITIES, State.RUNNING)) {
                        throw new IllegalStateException("Expected state " + State.START_ACTIVITIES + " but was " + state.get());
                    } else {
                        scheduledRefreshStatistics();
                    }
                })
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        lastInitStatus = Status.fromThrowable(e);
                    } else {
                        lastInitStatus = Status.OK;
                    }
                });

        init.complete(null);
        return result;
    }

    private CompletableFuture<?> createSchema(State from) {
        if (!changeState(from, State.SCHEME_CREATING)) {
            return CompletableFuture.completedFuture(null);
        }

        return alertDao.createSchema(projectId)
                .thenCompose(ignore -> stateDao.createSchema(projectId))
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed create schema for project: " + projectId, e);
                        changeState(State.SCHEME_CREATING, from);
                    } else {
                        lastInitStatus = Status.OK;
                    }
                });
    }

    private CompletableFuture<?> fetchAlerts(State from) {
        if (!changeState(from, State.FETCH_ALERTS)) {
            return CompletableFuture.completedFuture(null);
        }
        var alerts = new ArrayList<Alert>();
        return CompletableFutures.safeCall(() -> alertDao.find(projectId, alerts::add))
                .thenCompose(unused -> {
                    metrics.clearFromTemplateCount();
                    var runner = new AsyncActorRunner(new AlertActivityLoader(alerts), MoreExecutors.directExecutor(), 1);
                    return runner.start();
                })
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed fetch alerts for project: " + projectId, e);
                        changeState(State.FETCH_ALERTS, from);
                    } else {
                        lastInitStatus = Status.OK;
                    }
                });
    }

    private CompletableFuture<?> fetchStates(State from) {
        if (!changeState(from, State.FETCH_STATE)) {
            return CompletableFuture.completedFuture(null);
        }

        if (activityById.isEmpty()) {
            return completedFuture(null);
        }

        return stateDao.find(projectId, state -> {
            AlertActivity activity = activityById.get(state.getId());
            if (activity != null) {
                activity.restore(state);
            }
        }).whenComplete((ignore, e) -> {
            if (e != null) {
                logger.error("Failed fetch states for project: " + projectId, e);
                Status status = Status.fromThrowable(e);
                if (isInvalidState(lastInitStatus) && isInvalidState(status)) {
                    logger.error("State file corrupted at project {}, start without it", projectId);
                    return;
                }
                changeState(State.FETCH_STATE, from);
            } else {
                lastInitStatus = Status.OK;
            }
        });
    }

    private boolean isInvalidState(Status status) {
        switch (status.getCode()) {
            case NOT_FOUND:
            case DATA_LOSS:
                return true;
            default:
                return false;
        }
    }

    private CompletableFuture<?> startAllActivities(State from) {
        if (!changeState(from, State.START_ACTIVITIES)) {
            return CompletableFuture.completedFuture(null);
        }

        try {
            activityById.values().forEach(AlertActivity::run);
            lastInitStatus = Status.OK;
            return CompletableFuture.completedFuture(null);
        } catch (Throwable e) {
            logger.error("Failed start activities for project: " + projectId, e);
            changeState(State.START_ACTIVITIES, from);
            return CompletableFuture.failedFuture(e);
        }
    }

    @Override
    public CompletableFuture<TCreateAlertResponse> createAlert(TCreateAlertRequest request) {
        if (!isReady()) {
            return replyAsyncCreateAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        long now = clock.millis();
        Alert convertedAlert = AlertConverter.protoToAlert(request.getAlert())
                .toBuilder()
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setVersion(1)
                .build();

        return createAlert(request.getIdempotentOperationId(), convertedAlert);
    }

    private CompletableFuture<TCreateAlertResponse> createAlert(String operationId, Alert convertedAlert) {
        return idempotentOperationService.get(operationId, convertedAlert.getProjectId(), ContainerType.PROJECT, CREATE_OPERATION_TYPE)
                .thenCompose(operationOptional -> {
                    if (operationOptional.isPresent()) {
                        return CompletableFuture.completedFuture(replyCreateAlertFromOperation(operationOptional.get()));
                    }
                    if (activityById.containsKey(convertedAlert.getId())) {
                        return replyAsyncCreateAlert(INVALID_REQUEST, "Alert with id " + convertedAlert.getId() + " already exists");
                    }
                    if (!quota.tryIncAlertCount()) {
                        String message = "Alert count limit (" + quota.getAlertCountValue() + " of " + quota.getAlertCountLimit() + ") is reached";
                        return replyAsyncCreateAlert(ERequestStatusCode.RESOURCE_EXHAUSTED, message);
                    }
                    return projectAlertServiceValidator.validateCreate(convertedAlert)
                            .thenCompose(validationResult -> {
                                if (validationResult != null) {
                                    quota.decAlertCount();
                                    return replyAsyncCreateAlert(INVALID_REQUEST, validationResult);
                                }
                                return postInitializer.initializeCreate(convertedAlert)
                                        .thenCompose(alert -> {
                                            var op = AlertingIdempotency.operation(AlertConverter.alertToProto(alert), operationId, CREATE_OPERATION_TYPE);
                                            return alertDao.insert(alert, op)
                                                    .handle(this::handleInsertResult)
                                                    .thenCompose(insertOperation -> {
                                                        if (insertOperation.wasIdempotent()) {
                                                            return replyCreateFromOperation(operationId, alert);
                                                        }
                                                        var maybePrev = insertOperation.optionalAlert;
                                                        if (maybePrev.isPresent()) {
                                                            // DB and activityById out of sync

                                                            Alert prev = maybePrev.get();

                                                            if (!prev.equals(alert)) {
                                                                // make activity (better late than never) and return conflict
                                                                return upsertActivity(prev)
                                                                        .thenAccept(o -> alertChanged(prev, alert))
                                                                        .thenApply(unused -> replyCreateAlert(INVALID_REQUEST, "Alert with id " + alert.getId() + " already exists"));
                                                            }
                                                        }
                                                        // prev = null or prev = alert
                                                        return upsertActivity(alert)
                                                                .thenApply(unused -> alertCountChanged(alert, activityById.get(alert.getId()), 1))
                                                                .thenApply(unused -> replyCreateAlert(AlertConverter.alertToProto(alert)));
                                                    })
                                                    .handle((result, e) -> {
                                                        if (e != null) {
                                                            logger.error("Failed create alert {} request into project", alert.getKey(), e);
                                                            return replyCreateAlert(ERequestStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                                                        }
                                                        return result;
                                                    });
                                        });
                            });
                });
    }

    private CompletableFuture<TCreateAlertResponse> replyCreateFromOperation(String operationId, Alert alert) {
        return upsertActivity(alert)
            .thenCompose(unused -> idempotentOperationService.get(operationId, alert.getProjectId(), ContainerType.PROJECT, CREATE_OPERATION_TYPE))
            .thenApply(operationOptional -> replyCreateAlertFromOperation(operationOptional.orElseThrow(IdempotentOperationExistException::new)));
    }

    private QueryOperation handleInsertResult(Optional<Alert> alertOptional, Throwable throwable) {
        if (throwable != null) {
            var cause = Throwables.getRootCause(throwable);
            quota.decAlertCount();
            if (cause instanceof IdempotentOperationExistException) {
                return new QueryOperation(Optional.empty(), true);
            }
            Throwables.propagate(throwable);
        }
        return new QueryOperation(alertOptional, false);
    }

    private CompletableFuture<?> upsertActivity(Alert alert) {
        return activityFactory.makeActivity(alert)
                .thenApply(activity -> {
                    addActivityWithRun(alert, activity);
                    failedLoadingAlerts.remove(alert.getId());
                    return null;
                })
                .exceptionally(e -> {
                    if (alert.getAlertType() == AlertType.FROM_TEMPLATE) {
                        var activity = activityFactory.makeFailedActivity(alert, e);
                        addActivityWithRun(alert, activity);
                        logger.error("Failed update template alert {} for project: {}", alert.getId(), projectId, e);
                        failedLoadingAlerts.put(alert.getId(), activity);
                        return null;
                    }
                    throw new RuntimeException(e);
                });
    }

    private void addActivityWithRun(Alert alert, AlertActivity activity) {
        var prev = activityById.put(alert.getId(), activity);
        activity.run();
        if (prev != null) {
            prev.cancel();
        }
    }

    @Override
    public CompletableFuture<TReadAlertResponse> readAlert(TReadAlertRequest request) {
        if (!isReady()) {
            return replyAsyncReadAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        String alertId = request.getAlertId();
        AlertActivity activity = activityById.get(alertId);
        if (activity == null) {
            return replyAsyncReadAlert(NOT_FOUND, "Alert with id " + alertId + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            Alert alert = activity.getAlert();
            if (!alert.getFolderId().equals(folderId)) {
                return replyAsyncReadAlert(NOT_FOUND, "Alert with id " + alertId + " does not exit in folderId = " + folderId);
            }
        }
        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadAlert(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + alertId);
        }

        return replyAsyncReadAlert(AlertConverter.alertToProto(activity.getAlert()));
    }

    @Override
    public CompletableFuture<TReadAlertInterpolatedResponse> readAlert(TReadAlertInterpolatedRequest request) {
        if (!isReady()) {
            return replyAsyncReadInterpolatedAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        String alertId = request.getAlertId();
        AlertActivity activity = activityById.get(alertId);
        if (activity == null) {
            return replyAsyncReadInterpolatedAlert(NOT_FOUND, "Alert with id " + alertId + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            Alert alert = activity.getAlert();
            if (!alert.getFolderId().equals(folderId)) {
                return replyAsyncReadInterpolatedAlert(NOT_FOUND, "Alert with id " + alertId + " does not exit in folderId = " + folderId);
            }
        }
        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadInterpolatedAlert(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + alertId);
        }
        var alert = activity.getAlert();
        Alert interpolated;
        if (activity instanceof TemplateAlertActivity taa) {
            interpolated = taa.getDelegateAlert();
        } else {
            interpolated = alert;
        }

        return completedFuture(TReadAlertInterpolatedResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .setAlert(AlertConverter.alertToProto(alert))
                .setInterpolatedAlert(AlertConverter.alertToProto(interpolated))
                .build());
    }

    @Override
    public CompletableFuture<TReadSubAlertResponse> readSubAlert(TReadSubAlertRequest request) {
        if (!isReady()) {
            return replyAsyncReadSubAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        AlertActivity activity = activityById.get(request.getParentId());
        if (activity == null) {
            return replyAsyncReadSubAlert(NOT_FOUND, "Alert with id " + request.getParentId() + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            Alert alert = activity.getAlert();
            if (!alert.getFolderId().equals(folderId)) {
                return replyAsyncReadSubAlert(NOT_FOUND, "Alert with id " + alert.getId() + " does not exit in folderId = " + folderId);
            }
        }
        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadSubAlert(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + request.getParentId());
        }

        activity = ActivityFactory.unwrap(activity);
        if (!(activity instanceof MultiAlertActivity)) {
            return replyAsyncReadSubAlert(NOT_FOUND, "It's not a multi alert: " + request.getParentId());
        }
        MultiAlertActivity parent = (MultiAlertActivity) activity;
        SubAlertActivity child = parent.getSubActivity(request.getAlertId());
        if (child == null) {
            return replyAsyncReadSubAlert(NOT_FOUND, "Sub alert with id "
                    + request.getAlertId()
                    + " under alert "
                    + request.getParentId()
                    + " not exists");
        }

        return replyAsyncReadSubAlert(AlertConverter.subAlertToProto(child.getAlert()));
    }

    @Override
    public CompletableFuture<TUpdateAlertResponse> updateAlert(TUpdateAlertRequest request) {
        if (!isReady()) {
            return replyAsyncUpdateAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        return idempotentOperationService.get(request.getIdempotentOperationId(), request.getAlert().getProjectId(), ContainerType.PROJECT, UPDATE_OPERATION_TYPE)
                .thenCompose(operationOptional -> {
                    if (operationOptional.isPresent()) {
                        return CompletableFuture.completedFuture(replyUpdateAlertFromOperation(operationOptional.get()));
                    }
                    var id = request.getAlert().getId();
                    AlertActivity prev = activityById.get(id);
                    if (prev == null) {
                        return replyAsyncUpdateAlert(NOT_FOUND, "Alert with id " + id + " does not exist");
                    }

                    String folderId = request.getAlert().getFolderId();
                    if (!folderId.isEmpty()) {
                        if (!prev.getAlert().getFolderId().equals(folderId)) {
                            return replyAsyncUpdateAlert(NOT_FOUND, "Alert with id " + id + " does not exist in folderId = " + folderId);
                        }
                    }
                    if (!validateServiceProvider(prev, request.getAlert().getServiceProvider())) {
                        return replyAsyncUpdateAlert(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + id);
                    }
                    Alert alert = AlertConverter.protoToAlertWithPrevState(request.getAlert(), prev.getAlert());

                    if (alert.getVersion() != -1 && prev.getAlert().getVersion() != alert.getVersion()) {
                        return replyAsyncUpdateAlert(CONCURRENT_MODIFICATION, "Version mismatch for alert: " + alert.getId());
                    }

                    if (prev.getAlert().equalContent(alert)) {
                        return completedFuture(TUpdateAlertResponse.newBuilder()
                                .setRequestStatus(ERequestStatusCode.OK)
                                .setStatusMessage("Not Modified")
                                .setAlert(AlertConverter.alertToProto(prev.getAlert()))
                                .build());
                    }

                    long now = clock.millis();
                    Alert updated = alert.toBuilder()
                            .setFolderId(prev.getAlert().getFolderId())
                            .setCreatedBy(prev.getAlert().getCreatedBy())
                            .setCreatedAt(prev.getAlert().getCreatedAt())
                            .setUpdatedAt(now)
                            .setVersion(prev.getAlert().getVersion() + 1)
                            .build();
                    return projectAlertServiceValidator.validateUpdate(updated, prev, StringUtils.isEmpty(request.getAlert().getServiceProvider()))
                            .thenCompose(validationResult -> {
                                if (validationResult != null) {
                                    return CompletableFuture.completedFuture(replyUpdateAlert(INVALID_REQUEST, validationResult));
                                }
                                var op = AlertingIdempotency.operation(AlertConverter.alertToProto(updated), request.getIdempotentOperationId(), UPDATE_OPERATION_TYPE);
                                return alertDao.update(updated, op)
                                        .handle(this::handleUpdateResult)
                                        .thenCompose(updateOperation -> {
                                            if (updateOperation.wasIdempotent()) {
                                                return replyUpdateFromOperation(request, alert);
                                            }
                                            var maybePrev = updateOperation.optionalAlert;
                                            if (maybePrev.isEmpty()) {
                                                // No record in db for an existing activity.
                                                prev.cancel();
                                                return CompletableFuture.completedFuture(replyUpdateAlert(NOT_FOUND, "Alert with id " + alert.getId() + " does not exist"));
                                            }

                                            Alert prevAlert = maybePrev.get();
                                            if (prevAlert.getVersion() == updated.getVersion() - 1 || prevAlert.equals(updated)) {
                                                // regular update
                                                return upsertActivity(updated)
                                                        .thenAccept(o -> alertChanged(prevAlert, updated))
                                                        .thenApply(unused -> replyUpdateAlert(AlertConverter.alertToProto(updated)));
                                            }

                                            // db and activity are out of sync. Recreate activity from db
                                            return upsertActivity(prevAlert)
                                                    .thenAccept(o -> alertChanged(prevAlert, updated))
                                                    .thenApply(unused -> replyUpdateAlert(CONCURRENT_MODIFICATION, "Version mismatch for alert: " + alert.getId()));
                                        })
                                        .handle((result, e) -> {
                                            if (e != null) {
                                                logger.error("Failed to update alert {}", updated.getKey(), e);
                                                return replyUpdateAlert(ERequestStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                                            }
                                            return result;
                                        });
                            });
                });
    }

    private CompletableFuture<TUpdateAlertResponse> replyUpdateFromOperation(TUpdateAlertRequest request, Alert alert) {
        return upsertActivity(alert)
                .thenCompose(unused -> idempotentOperationService.get(request.getIdempotentOperationId(), alert.getProjectId(), ContainerType.PROJECT, UPDATE_OPERATION_TYPE))
                .thenApply(operationOptional -> replyUpdateAlertFromOperation(operationOptional.orElseThrow(IdempotentOperationExistException::new)));
    }

    private QueryOperation handleUpdateResult(Optional<Alert> alertOptional, Throwable throwable) {
        if (throwable != null) {
            var cause = Throwables.getRootCause(throwable);
            if (cause instanceof IdempotentOperationExistException) {
                return new QueryOperation(Optional.empty(), true);
            }
            Throwables.propagate(throwable);
        }
        return new QueryOperation(alertOptional, false);
    }

    @Override
    public CompletableFuture<TDeleteAlertResponse> deleteAlert(TDeleteAlertRequest request) {
        if (!isReady()) {
            return replyAsyncDeleteAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        return idempotentOperationService.get(request.getIdempotentOperationId(), request.getProjectId(), ContainerType.PROJECT, DELETE_OPERATION_TYPE)
                .thenCompose(operationOptional -> {
                    if (operationOptional.isEmpty()) {
                        var activity = activityById.get(request.getAlertId());
                        if (activity instanceof TemplateAlertActivity taa) {
                            return idempotentOperationService.delete(makeOperationId((AlertFromTemplatePersistent) taa.getAlert()), request.getProjectId(), ContainerType.PROJECT, CREATE_OPERATION_TYPE)
                                    .thenApply(aBoolean -> operationOptional);
                        }
                    }
                   return CompletableFuture.completedFuture(operationOptional);
                })
                .thenCompose(operationOptional -> {
                    if (operationOptional.isPresent()) {
                        return CompletableFuture.completedFuture(replyDeleteAlert());
                    }
                    String alertId = request.getAlertId();
                    if (!activityById.containsKey(alertId)) {
                        return replyAsyncDeleteAlert(NOT_FOUND, "Alert with id " + alertId + " not exits");
                    }

                    String folderId = request.getFolderId();
                    if (!folderId.isEmpty()) {
                        Alert alert = activityById.get(alertId).getAlert();
                        if (!alert.getFolderId().equals(folderId)) {
                            return replyAsyncDeleteAlert(NOT_FOUND, "Alert with id " + alertId + " does not exit in folderId = " + folderId);
                        }
                    }
                    if (!validateServiceProvider(activityById.get(alertId), request.getServiceProvider())) {
                        return replyAsyncDeleteAlert(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + alertId);
                    }
                    if (!projectAlertServiceValidator.validateDelete(request, activityById.get(alertId))) {
                        return replyAsyncDeleteAlert(INVALID_REQUEST, "Alert with id " + alertId + " can't be deleted");
                    }
                    var op = AlertingIdempotency.operation(request.getProjectId(), request.getAlertId(), request.getIdempotentOperationId(), DELETE_OPERATION_TYPE);
                    return alertDao.deleteById(projectId, alertId, op)
                            .handle((ignore, e) -> {
                                if (e != null) {
                                    var cause = Throwables.getRootCause(e);
                                    if (cause instanceof IdempotentOperationExistException) {
                                        AlertActivity activity = activityById.remove(alertId);
                                        if (activity != null) {
                                            activity.cancel();
                                        }
                                        return replyDeleteAlert();
                                    }
                                    logger.error("Failed to delete alert {} into project {}", alertId, projectId, e);
                                    return replyDeleteAlert(ERequestStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                                }

                                AlertActivity activity = activityById.remove(alertId);
                                if (activity != null) {
                                    activity.cancel();
                                }
                                failedLoadingAlerts.remove(alertId);
                                quota.decAlertCount();
                                alertCountChanged(activity.getAlert(), activity,  -1);

                                return replyDeleteAlert();
                            });
                });
    }

    @Override
    public CompletableFuture<TDeletionNotificationResponse> notifyOnDeletionProject(TDeletionNotificationRequest request) {
        throw new UnsupportedOperationException();
    }

    @Override
    public CompletableFuture<TListAlertResponse> listAlerts(TListAlertRequest request) {
        if (!isReady()) {
            return replyAsyncListAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        return CompletableFuture.completedFuture(activitySearch.listAlerts(activityById.values(), request));
    }

    @Override
    public CompletableFuture<TListSubAlertResponse> listSubAlerts(TListSubAlertRequest request) {
        if (!isReady()) {
            return replyAsyncListSubAlert(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        AlertActivity activity = activityById.get(request.getParentId());
        if (activity == null) {
            return replyAsyncListSubAlert(NOT_FOUND, "Alert with id " + state.get() + " not exists");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(folderId)) {
                return replyAsyncListSubAlert(NOT_FOUND, "Alert with id " + state.get() + " does not exist in folderId = " + folderId);
            }
        }

        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncListSubAlert(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + request.getParentId());
        }

        return CompletableFuture.completedFuture(activitySearch.listSubAlerts(activity, request));
    }

    @Override
    public CompletableFuture<TReadEvaluationStateResponse> readEvaluationState(TReadEvaluationStateRequest request) {
        if (!isReady()) {
            return replyAsyncReadEvaluationState(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        if ("".equals(request.getParentId())) {
            return readAlertEvaluationState(request);
        } else {
            return readSubAlertEvaluationState(request);
        }
    }

    @Override
    public CompletableFuture<TReadEvaluationStatsResponse> readEvaluationStats(TReadEvaluationStatsRequest request) {
        if (!isReady()) {
            return replyAsyncReadEvaluationStats(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        AlertActivity activity = activityById.get(request.getAlertId());

        if (activity == null) {
            return replyAsyncReadEvaluationStats(NOT_FOUND, "Alert with id " + request.getAlertId() + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(folderId)) {
                return replyAsyncReadEvaluationStats(NOT_FOUND, "Alert with id " + request.getAlertId() + " does not exist in folderId = " + folderId);
            }
        }

        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadEvaluationStats(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + request.getAlertId());
        }

        TEvaluationStats stats = activitySearch.getEvaluationStats(activity);
        TEvaluationStats muted = activitySearch.getMutedStats(activity, Set.of(), null);

        return replyAsyncReadEvaluationStats(stats, muted);
    }

    @Override
    public CompletableFuture<TExplainEvaluationResponse> explainEvaluation(TExplainEvaluationRequest request) {
        return completedFuture(TExplainEvaluationResponse.newBuilder()
                .setRequestStatus(INVALID_REQUEST)
                .setStatusMessage("Unsupported operation: " + request)
                .build());
    }

    @Override
    public CompletableFuture<TSimulateEvaluationResponse> simulateEvaluation(TSimulateEvaluationRequest request) {
        return completedFuture(TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(INVALID_REQUEST)
                .setStatusMessage("Unsupported operation: " + request)
                .build());
    }

    private CompletableFuture<TReadEvaluationStateResponse> readAlertEvaluationState(TReadEvaluationStateRequest request) {
        AlertActivity activity = activityById.get(request.getAlertId());
        if (activity == null) {
            return replyAsyncReadEvaluationState(NOT_FOUND, "Alert with id " + request.getAlertId() + " does not exist in project " + projectId);
        }

        if (!request.getFolderId().isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(request.getFolderId())) {
                return replyAsyncReadEvaluationState(NOT_FOUND, "Alert with id " + request.getAlertId() + " does not exist in folderId " + request.getFolderId());
            }
        }

        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadEvaluationState(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + request.getAlertId());
        }

        activity = ActivityFactory.unwrap(activity);
        final AlertProcessingState state;
        if (activity instanceof SimpleAlertActivity saa) {
            state = saa.getProcessingState();
        } else {
            state = null;
        }

        return replyAsyncReadEvaluationState(
                AlertConverter.stateToProto(Nullables.map(state, AlertProcessingState::evaluationState)),
                AlertConverter.muteStatusToProto(Nullables.map(state, AlertProcessingState::alertMuteStatus))
        );
    }

    private CompletableFuture<TReadEvaluationStateResponse> readSubAlertEvaluationState(TReadEvaluationStateRequest request) {
        AlertActivity activity = activityById.get(request.getParentId());
        if (activity == null) {
            return replyAsyncReadEvaluationState(NOT_FOUND, "Alert with id " + request.getParentId() + " does not exist");
        }

        if (!request.getFolderId().isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(request.getFolderId())) {
                return replyAsyncReadEvaluationState(NOT_FOUND, "Alert with id " + request.getParentId() + " does not exist in folderId " + request.getFolderId());
            }
        }

        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadEvaluationState(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + request.getParentId());
        }

        activity = ActivityFactory.unwrap(activity);
        if (!(activity instanceof MultiAlertActivity)) {
            return replyAsyncReadEvaluationState(NOT_FOUND, "It's not a multi alert: " + request.getParentId());
        }
        MultiAlertActivity parent = (MultiAlertActivity) activity;
        SubAlertActivity child = parent.getSubActivity(request.getAlertId());
        if (child == null) {
            return replyAsyncReadEvaluationState(NOT_FOUND, "Sub alert with id "
                    + request.getAlertId()
                    + " under alert "
                    + request.getParentId()
                    + " not exists");
        }

        var state = child.getProcessingState();

        return replyAsyncReadEvaluationState(
                AlertConverter.stateToProto(Nullables.map(state, AlertProcessingState::evaluationState)),
                AlertConverter.muteStatusToProto(Nullables.map(state, AlertProcessingState::alertMuteStatus))
        );
    }

    @Override
    public CompletableFuture<TReadNotificationStateResponse> readNotificationState(TReadNotificationStateRequest request) {
        if (!isReady()) {
            return replyAsyncReadNotificationState(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        if ("".equals(request.getParentId())) {
            return readAlertNotificationState(request);
        } else {
            return readSubAlertNotificationState(request);
        }
    }

    @Override
    public CompletableFuture<TReadNotificationStatsResponse> readNotificationStats(TReadNotificationStatsRequest request) {
        if (!isReady()) {
            return replyAsyncReadNotificationStats(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        AlertActivity activity = activityById.get(request.getAlertId());

        if (activity == null) {
            return replyAsyncReadNotificationStats(NOT_FOUND, "Alert with id " + request.getAlertId() + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(folderId)) {
                return replyAsyncReadNotificationStats(NOT_FOUND, "Alert with id " + request.getAlertId() + " does not exist in folderId = " + folderId);
            }
        }


        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadNotificationStats(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + request.getAlertId());
        }

        TNotificationStats stats = activitySearch.getNotificationStats(activity, Collections.emptySet());

        return replyAsyncReadNotificationStats(stats);
    }

    @Override
    public CompletableFuture<TReadProjectStatsResponse> readProjectStats(TReadProjectStatsRequest request) {
        if (!isReady()) {
            return completedFuture(TReadProjectStatsResponse.newBuilder()
                    .setRequestStatus(SHARD_NOT_INITIALIZED)
                    .setStatusMessage("Expected running state, but was " + state.get())
                    .build());
        }

        ensureSummaryStatisticCalculated();

        String folderId = request.getFolderId();
        int alertsCount;

        if (folderId.isEmpty()) {
            folderId = TOTAL_FOLDER;
            alertsCount = activityById.size();
        } else {
            alertsCount = alertsCountByFolder.getOrDefault(folderId, 0);
        }

        var evaluationSummaryStatistics = evaluationSummaryStatisticsByFolder.getOrDefault(folderId, EvaluationSummaryStatistics.EMPTY);
        var mutedSummaryStatistics = mutedSummaryStatisticsByFolder.getOrDefault(folderId, EvaluationSummaryStatistics.EMPTY);
        var notificationSummaryStatistics = notificationSummaryStatisticsByFolder.getOrDefault(folderId, NotificationSummaryStatistics.EMPTY);

        return completedFuture(TReadProjectStatsResponse.newBuilder()
                .setRequestStatus(OK)
                .setAlertsCount(alertsCount)
                .setEvaluationStats(SummaryStatisticsConverter.toProto(evaluationSummaryStatistics))
                .setMutedStats(SummaryStatisticsConverter.toProto(mutedSummaryStatistics))
                .setNotificationStats(SummaryStatisticsConverter.toProto(notificationSummaryStatistics))
                .build());
    }

    private CompletableFuture<TReadNotificationStateResponse> readAlertNotificationState(TReadNotificationStateRequest request) {
        String folderId = request.getFolderId();
        String alertId = request.getAlertId();
        AlertActivity activity = activityById.get(alertId);
        if (activity == null) {
            return replyAsyncReadNotificationState(NOT_FOUND, "Alert with id " + alertId + " does not exist in project " + projectId);
        }

        if (!folderId.isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(folderId)) {
                return replyAsyncReadNotificationState(NOT_FOUND, "Alert with id " + alertId + " does not exist in folderId " + folderId);
            }
        }

        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadNotificationState(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + alertId);
        }

        activity = ActivityFactory.unwrap(activity);
        if (activity instanceof SimpleAlertActivity one) {
            List<TNotificationState> states = one.getNotificationStates(Collections.emptySet())
                    .map(notificationConverter::stateToProto)
                    .collect(Collectors.toList());
            return replyAsyncReadNotificationState(states);
        }

        return replyAsyncReadNotificationState(Collections.emptyList());
    }

    private CompletableFuture<TReadNotificationStateResponse> readSubAlertNotificationState(TReadNotificationStateRequest request) {
        String folderId = request.getFolderId();
        String alertId = request.getAlertId();
        String parentId = request.getParentId();
        AlertActivity activity = activityById.get(parentId);
        if (activity == null) {
            return replyAsyncReadNotificationState(NOT_FOUND, "Alert with id " + parentId + " does not exist");
        }

        if (!folderId.isEmpty()) {
            if (!activity.getAlert().getFolderId().equals(folderId)) {
                return replyAsyncReadNotificationState(NOT_FOUND, "Alert with id " + parentId + " does not exist in folderId " + folderId);
            }
        }
        if (!validateServiceProvider(activity, request.getServiceProvider())) {
            return replyAsyncReadNotificationState(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + parentId);
        }
        activity = ActivityFactory.unwrap(activity);
        if (!(activity instanceof MultiAlertActivity)) {
            return replyAsyncReadNotificationState(NOT_FOUND, "It's not a multi alert: " + parentId);
        }

        MultiAlertActivity parent = (MultiAlertActivity) activity;
        SubAlertActivity child = parent.getSubActivity(alertId);
        if (child == null) {
            return replyAsyncReadNotificationState(NOT_FOUND, "Sub alert with id "
                    + alertId
                    + " under alert "
                    + parentId
                    + " not exists");
        }

        return replyAsyncReadNotificationState(child.getNotificationStates(Collections.emptySet())
                .map(notificationConverter::stateToProto)
                .collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<CreateAlertsFromTemplateResponse> createAlerts(CreateAlertsFromTemplateRequest request) {
        if (!isReady()) {
            return completedFuture(CreateAlertsFromTemplateResponse.newBuilder()
                    .setRequestStatusCode(SHARD_NOT_INITIALIZED)
                    .setStatusMessage("Expected running state, but was " + state.get())
                    .build());
        }
        return projectAlertServiceValidator.validateCreateAlertsFromTemplate(request)
                .thenCompose(validationResult -> {
                    if (validationResult != null) {
                        return completedFuture(CreateAlertsFromTemplateResponse.newBuilder()
                                .setRequestStatusCode(INVALID_REQUEST)
                                .setStatusMessage(validationResult)
                                .build());
                    }
                    List<AlertFromTemplatePersistent> alertBases = AlertConverter.protoToAlerts(clock.millis(), request);
                    return postInitializer.initializeTemplateAlertsFromPublishedTemplates(alertBases, request.getServiceProviderId())
                            .thenCompose(alerts -> {
                                var actorBody = new AlertCreator(alerts);
                                var runner = new AsyncActorRunner(actorBody, MoreExecutors.directExecutor(), 4);
                                return runner.start()
                                        .thenApply(unused -> CreateAlertsFromTemplateResponse.newBuilder()
                                                .setRequestStatusCode(OK)
                                                .addAllAlerts(alerts.stream()
                                                        .map(alert -> CreateAlertsFromTemplateResponse.Alert.newBuilder()
                                                                .setAlertId(actorBody.resolveId(alert.getId()))
                                                                .setTemplateId(alert.getTemplateId())
                                                                .putAllResourceParameters(alert.getLabels())
                                                                .build())
                                                        .collect(Collectors.toList()))
                                                .build());
                            });
                });
    }

    @Override
    public CompletableFuture<ListAlertLabelsResponse> listAlertLabels(ListAlertLabelsRequest request) {
        if (!isReady()) {
            return replyAsyncListAlertLabels(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }
        return CompletableFuture.completedFuture(ListAlertLabelsResponse.newBuilder()
                .setRequestStatusCode(ERequestStatusCode.OK)
                .addAllLabelKeys(alertLabels.entrySet().stream()
                        .map(entry -> entry.getValue().get() > 0 ? entry.getKey() : null)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList()))
                .build());
    }

    @Override
    public CompletableFuture<UpdateAlertTemplateVersionResponse> updateAlertTemplateVersion(UpdateAlertTemplateVersionRequest request) {
        if (!isReady()) {
            return replyAsyncUpdateAlertTemplateVersion(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        if (request.getTypeCase() == UpdateAlertTemplateVersionRequest.TypeCase.UPDATE_COUNT) {
            return updateAlertTemplateVersionByCount(request);
        }

        UpdateAlertTemplateVersionRequest.AlertData alertData = request.getAlertData();
        var id = alertData.getAlertId();
        AlertActivity prev = activityById.get(id);
        if (prev == null) {
            return replyAsyncUpdateAlertTemplateVersion(NOT_FOUND, "Alert with id " + id + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            if (!prev.getAlert().getFolderId().equals(folderId)) {
                return replyAsyncUpdateAlertTemplateVersion(NOT_FOUND, "Alert with id " + id + " does not exist in folderId = " + folderId);
            }
        }
        if (!validateServiceProvider(prev, request.getServiceProvider())) {
            return replyAsyncUpdateAlertTemplateVersion(ERequestStatusCode.NOT_AUTHORIZED, "Not authorized to access alert " + id);
        }

        if (prev.getAlert().getAlertType() != AlertType.FROM_TEMPLATE) {
            return replyAsyncUpdateAlertTemplateVersion(INVALID_REQUEST, "Can't update template version for alert: " + id);
        }

        AlertFromTemplatePersistent templateAlert = (AlertFromTemplatePersistent) prev.getAlert();
        if (!templateAlert.getTemplateId().equals(request.getTemplateId())) {
            return replyAsyncUpdateAlertTemplateVersion(INVALID_REQUEST, "Can't update alert to another template, for alert: " + id);
        }
        if (templateAlert.getTemplateVersionTag().equals(request.getTemplateVersionTag())) {
            return completedFuture(UpdateAlertTemplateVersionResponse.newBuilder()
                    .setRequestStatusCode(ERequestStatusCode.OK)
                    .setStatusMessage("Not Modified")
                    .setAlert(AlertConverter.alertToProto(prev.getAlert()))
                    .build());
        }
        return updateVersionInSingleAlert(prev, templateAlert, request.getTemplateVersionTag(), alertData.getUpdatedBy());
    }

    private CompletableFuture<UpdateAlertTemplateVersionResponse> updateVersionInSingleAlert(
            AlertActivity prev,
            AlertFromTemplatePersistent templateAlert,
            String templateVersionTag,
            String updatedBy)
    {
        long now = clock.millis();
        Alert updatedConverted = templateAlert.toBuilder()
                .setTemplateVersionTag(templateVersionTag)
                .setUpdatedAt(now)
                .setUpdatedBy(updatedBy == null ? templateAlert.getUpdatedBy() : updatedBy)
                .setVersion(prev.getAlert().getVersion() + 1)
                .build();

        return projectAlertServiceValidator.validateAlertVersionUpdate(updatedConverted)
                .thenCompose(validationResult -> {
                    if (validationResult != null) {
                        return replyAsyncUpdateAlertTemplateVersion(INVALID_REQUEST, validationResult);
                    }
                    return postInitializer.initializeVersionUpdate(updatedConverted)
                            .thenCompose(updated -> alertDao.update(updated, IdempotentOperation.NO_OPERATION)
                                    .thenCompose(maybePrev -> {
                                        if (maybePrev.isEmpty()) {
                                            // No record in db for an existing activity.
                                            prev.cancel();
                                            return replyAsyncUpdateAlertTemplateVersion(NOT_FOUND, "Alert with id " + updated.getId() + " does not exist");
                                        }

                                        Alert prevAlert = maybePrev.get();
                                        if (prevAlert.getVersion() == updated.getVersion() - 1 || prevAlert.equals(updated)) {
                                            // regular update
                                            return upsertActivity(updated)
                                                    .thenApply(unused -> alertVersionUpdatedMetricUpdate(updated))
                                                    .thenApply(unused -> UpdateAlertTemplateVersionResponse.newBuilder()
                                                            .setRequestStatusCode(ERequestStatusCode.OK)
                                                            .setAlert(AlertConverter.alertToProto(updated))
                                                            .build());
                                        }

                                        // db and activity are out of sync. Recreate activity from db
                                        return upsertActivity(prevAlert)
                                                .thenApply(unused -> replyUpdateAlertTemplateVersion(CONCURRENT_MODIFICATION, "Version mismatch for alert: " + updated.getId()));
                                    })
                                    .handle((result, e) -> {
                                        if (e != null) {
                                            logger.error("Failed to update alert version {}", updated.getKey(), e);
                                            return replyUpdateAlertTemplateVersion(ERequestStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                                        }
                                        return result;
                                    }));
                });
    }


    private void alertChanged(Alert prev, Alert alert) {
        // add new
        changeLabels(alert, 1);
        // remove old
        changeLabels(prev, -1);
    }

    private Void alertCountChanged(Alert alert, AlertActivity alertActivity, int count) {
        if (alert instanceof AlertFromTemplatePersistent fromTemplatePersistent) {
            var serviceProvider = "";
            if (alertActivity instanceof TemplateAlertActivity taa) {
                serviceProvider = taa.getTemplate().getServiceProviderId();
            }
            metrics.incrementAlertCount(fromTemplatePersistent.getTemplateId(), fromTemplatePersistent.getServiceProvider(), count, serviceProvider);
        }
        changeLabels(alert, count);
        return null;
    }

    private void changeLabels(Alert alert, int count) {
        for (String key : alert.getLabels().keySet()) {
            var counter = alertLabels.computeIfAbsent(key, s -> new AtomicInteger(0));
            counter.addAndGet(count);
        }
        if (alert instanceof AlertFromTemplatePersistent) {
            alertLabels.computeIfAbsent("templateId", s -> new AtomicInteger(0)).addAndGet(count);
            alertLabels.computeIfAbsent("templateVersionTag", s -> new AtomicInteger(0)).addAndGet(count);
            alertLabels.computeIfAbsent("serviceProviderId", s -> new AtomicInteger(0)).addAndGet(count);
        }
    }

    private Void alertVersionUpdatedMetricUpdate(Alert alert) {
        if (alert instanceof AlertFromTemplatePersistent fromTemplatePersistent) {
            alertFromTemplateMetrics.fromTemplateVersionUpdated(fromTemplatePersistent.getTemplateId(), fromTemplatePersistent.getServiceProvider());
        }
        return null;
    }

    private CompletableFuture<UpdateAlertTemplateVersionResponse> updateAlertTemplateVersionByCount(UpdateAlertTemplateVersionRequest request) {
        AtomicInteger atomicInteger = new AtomicInteger(0);
        Iterator<AlertActivity> iterator = activityById.values().iterator();
        AsyncActorBody body = () -> {
            while (true) {
                if (!iterator.hasNext() || atomicInteger.get() >= request.getUpdateCount()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }
                AlertActivity next = iterator.next();
                if (next instanceof TemplateAlertActivity taa) {
                    if (taa.getTemplate().getId().equals(request.getTemplateId()) &&
                            !taa.getTemplate().getTemplateVersionTag().equals(request.getTemplateVersionTag())) {
                        atomicInteger.incrementAndGet();
                        AlertFromTemplatePersistent templateAlert = (AlertFromTemplatePersistent) next.getAlert();
                        return updateVersionInSingleAlert(next, templateAlert, request.getTemplateVersionTag(), null);
                    }
                }
                return CompletableFuture.completedFuture(null);
            }
        };
        var runner = new AsyncActorRunner(body, executor, 1);
        return runner.start()
                .thenApply(unused -> UpdateAlertTemplateVersionResponse.newBuilder()
                        .setRequestStatusCode(ERequestStatusCode.OK)
                        .setUpdated(atomicInteger.get())
                        .build())
                .handle((response, throwable) -> {
                    if (throwable != null) {
                        logger.error("Failed to update alerts version ", throwable);
                        return replyUpdateAlertTemplateVersion(ERequestStatusCode.INTERNAL_ERROR, Throwables.getStackTraceAsString(throwable));
                    }
                    return response;
                });
    }

    public List<TPersistAlertState> snapshot() {
        State currentState = state.get();
        if (currentState != State.CLOSED && currentState != State.RUNNING) {
            throw new IllegalStateException("Shard " + projectId + " not initialized yet, current state: " + currentState);
        }

        return this.activityById.values()
                .parallelStream()
                .map(AlertActivity::dumpState)
                .collect(Collectors.toList());
    }

    @Override
    public void close() {
        var prev = this.state.getAndSet(State.CLOSED);
        if (prev == State.CLOSED) {
            return;
        }

        this.activityById.values().parallelStream().forEach(AlertActivity::cancel);
        Optional.ofNullable(refreshFuture).ifPresent(scheduledFuture -> scheduledFuture.cancel(true));
    }

    private boolean changeState(State from, State to) {
        if (this.state.compareAndSet(from, to)) {
            logger.info("{}: alerts change state {} -> {}", projectId, from, to);
            return true;
        }

        return false;
    }

    private boolean validateServiceProvider(AlertActivity activity, String serviceProvider) {
        if (StringUtils.isEmpty(serviceProvider)) {
            // user path
            return true;
        }
        // service provider can access only its alerts
        if (activity.getAlert() instanceof AlertFromTemplatePersistent templateAlert) {
            return serviceProvider.equals(templateAlert.getServiceProvider());
        }
        return false;
    }

    private String makeOperationId(AlertFromTemplatePersistent alert) {
        return Stream.concat(Stream.of(alert.getTemplateId()), alert.getLabels().entrySet().stream()
                .map(entry -> entry.getKey() + "_" + entry.getValue())
                .sorted())
                .collect(Collectors.joining(",", "", ""));
    }

    private enum State {
        IDLE,
        SCHEME_CREATING,
        FETCH_ALERTS,
        FETCH_STATE,
        START_ACTIVITIES,
        RUNNING,
        CLOSED,

    }
    private class AlertActivityLoader implements AsyncActorBody {

        private final Iterator<Alert> it;

        public AlertActivityLoader(List<Alert> alerts) {
            it = alerts.iterator();
        }
        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var alert = it.next();
            return activityFactory.makeActivity(alert)
                    .thenAccept(activity -> {
                        activityById.put(alert.getId(), activity);
                        quota.setAlertCount(activityById.size());
                        alertCountChanged(alert, activity, 1);
                    })
                    .exceptionally(e -> {
                        if (alert.getAlertType() == AlertType.FROM_TEMPLATE) {
                            var activity = activityFactory.makeFailedActivity(alert, e);
                            activityById.put(alert.getId(), activity);
                            quota.setAlertCount(activityById.size());
                            logger.error("Failed fetch template alert {} for project: {}", alert.getId(), projectId, e);
                            failedLoadingAlerts.put(alert.getId(), activity);
                            return null;
                        }

                        throw new RuntimeException(e);
                    });
        }

    }
    private class AlertCreator implements AsyncActorBody {
        private final Iterator<AlertFromTemplatePersistent> it;

        private final ConcurrentMap<String, TCreateAlertResponse> results = new ConcurrentHashMap<>();

        public AlertCreator(List<AlertFromTemplatePersistent> alerts) {
            it = alerts.iterator();
        }

        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            var alert = it.next();
            var operation = makeOperationId(alert);
            return createAlert(operation, alert)
                    .thenApply(response -> {
                        if (response.getRequestStatus() != OK) {
                            throw new RuntimeException(response.getStatusMessage());
                        }
                        results.put(alert.getId(), response);
                        return response;
                    })
                    .exceptionally(e -> {
                        throw new RuntimeException(e);
                    });
        }
        public String resolveId(String id) {
            var response = results.get(id);
            return Optional.ofNullable(response)
                    .map(TCreateAlertResponse::getAlert)
                    .map(TAlert::getId)
                    .orElse(id);
        }

    }

    private record QueryOperation(Optional<Alert> optionalAlert, boolean wasIdempotent){}
}
