package ru.yandex.solomon.alert.cluster.broker.alert;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.jns.client.JnsClient;
import ru.yandex.jns.dto.JnsListEscalationPolicy;
import ru.yandex.jns.dto.ListEscalationRequest;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.AlertActivity;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.protobuf.CreateAlertsFromTemplateRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteAlertRequest;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ProjectAlertServiceValidatorImpl implements ProjectAlertServiceValidator {

    private static final Logger logger = LoggerFactory.getLogger(ProjectAlertServiceValidatorImpl.class);
    private final AlertTemplateDao alertTemplateDao;
    private final AlertTemplateLastVersionDao alertTemplateLastVersionDao;
    private final JnsClient jnsClient;

    public ProjectAlertServiceValidatorImpl(
            AlertTemplateDao alertTemplateDao,
            AlertTemplateLastVersionDao alertTemplateLastVersionDao,
            JnsClient jnsClient)
    {
        this.alertTemplateDao = alertTemplateDao;
        this.alertTemplateLastVersionDao = alertTemplateLastVersionDao;
        this.jnsClient = jnsClient;
    }

    @Override
    public CompletableFuture<String> validateCreate(Alert alert) {
        CompletableFuture<String> result = CompletableFuture.completedFuture(null);
        switch (alert.getAlertType()) {
            case FROM_TEMPLATE -> result = result.thenCompose(s -> validateCreateAlertFromTemplate((AlertFromTemplatePersistent) alert));
        }
        return result.thenCompose(s -> {
            if (s != null) {
                return CompletableFuture.completedFuture(s);
            }
            return validateBase(alert);
        });
    }

    private CompletableFuture<String> validateBase(Alert alert) {
        if (alert.getEscalations().isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        return jnsClient.listEscalations(new ListEscalationRequest(alert.getProjectId(), ""))
                .handle((jnsListEscalationPolicy, throwable) -> {
                    if (throwable != null) {
                        logger.error("Error while validating escalations, let trust them", throwable);
                        return new JnsListEscalationPolicy("Error:" + throwable.getMessage(), "", List.of());
                    }
                    return jnsListEscalationPolicy;
                })
                .thenApply(result -> {
                    if (!StringUtils.isEmpty(result.error())) {
                        logger.error("Error while validating escalations, let trust them, {} - {}", result.error(), result.message());
                        return null;
                    }
                    var ids = Nullables.orEmpty(result.policies()).stream().map(JnsListEscalationPolicy.JnsEscalationPolicy::name).collect(Collectors.toSet());
                    for (String escalation : alert.getEscalations()) {
                        if (!ids.contains(escalation)) {
                            return String.format("No escalation '%s' in project '%s'", escalation, alert.getProjectId());
                        }
                    }
                    return null;
                });
    }

    @Override
    public CompletableFuture<String> validateCreateAlertsFromTemplate(CreateAlertsFromTemplateRequest request) {
        return alertTemplateLastVersionDao.find(request.getServiceProviderId(), "", 1000, "0")
                .thenCompose(page -> alertTemplateDao.findVersions(page.getItems()).thenApply(alertTemplates -> {
                    var map = alertTemplates.stream().collect(Collectors.toMap(AlertTemplate::getId, Function.identity()));
                    Set<String> neededTemplates = new HashSet<>(request.getTemplateIdsList());
                    for (String id : request.getTemplateIdsList()) {
                        if (!map.containsKey(id)) {
                            return "Hasn't published alert template with id " + id;
                        }
                    }
                    for (CreateAlertsFromTemplateRequest.Resource resourceId : request.getResourcesList()) {
                        for (AlertTemplate value : map.values()) {
                            if (!neededTemplates.contains(value.getId())) {
                                continue;
                            }
                            for (AlertTemplateParameter parameter : value.getParameters()) {
                                if (StringUtils.isEmpty(resourceId.getResourceParametersMap().get(parameter.getName()))) {
                                    return "Resource '" + resourceId.getResourceParametersMap() + "' hasn't parameter '" + parameter.getName() + "' for template '" + value.getId() + "'";
                                }
                            }
                        }
                    }
                    return null;
                }));
    }

    @Override
    public CompletableFuture<String> validateUpdate(Alert alert, AlertActivity prevActivity, boolean isUserRequest) {
        switch (alert.getAlertType()) {
            case FROM_TEMPLATE -> {
                return validateUpdateAlertFromTemplate((AlertFromTemplatePersistent) alert, prevActivity, isUserRequest);
            }
            default -> {
                return CompletableFuture.completedFuture(null);
            }
        }
    }

    @Override
    public CompletableFuture<String> validateAlertVersionUpdate(Alert alert) {
        switch (alert.getAlertType()) {
            case FROM_TEMPLATE -> {
                return validateAlertFromTemplateVersionUpdate((AlertFromTemplatePersistent) alert);
            }
            default -> {
                return CompletableFuture.completedFuture(null);
            }
        }
    }

    @Override
    public boolean validateDelete(TDeleteAlertRequest request, AlertActivity prevActivity) {
        switch (prevActivity.getAlert().getAlertType()) {
            default -> {
                return true;
            }
        }
    }

    private CompletableFuture<String> validateAlertFromTemplateVersionUpdate(AlertFromTemplatePersistent alert) {
        return alertTemplateDao.findById(alert.getTemplateId(), alert.getTemplateVersionTag())
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        return "Hasn't alert template with id " + alert.getTemplateId() + " and version " + alert.getTemplateVersionTag();
                    }
                    return null;
                });
    }

    private CompletableFuture<String> validateUpdateAlertFromTemplate(AlertFromTemplatePersistent alert, AlertActivity prevActivity, boolean isUserRequest) {
        return alertTemplateDao.findById(alert.getTemplateId(), alert.getTemplateVersionTag())
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        return "Hasn't alert template with id " + alert.getTemplateId() + " and version " + alert.getTemplateVersionTag();
                    }
                    AlertFromTemplatePersistent activityAlert = (AlertFromTemplatePersistent) prevActivity.getAlert();
                    if (!activityAlert.getTemplateId().equals(alert.getTemplateId())) {
                        return "Can't change template id, original was " + activityAlert.getTemplateId();
                    }
                    if (!activityAlert.getTemplateVersionTag().equals(alert.getTemplateVersionTag())) {
                        return "Can't change template version tag, original was " + activityAlert.getTemplateVersionTag();
                    }
                    if (!activityAlert.getServiceProvider().equals(alert.getServiceProvider())) {
                        return "Can't change alert of another service provider";
                    }
                    var result = validateAllParams(alert, alertTemplate.get());
                    if (result != null) {
                        return result;
                    }
                    if (isUserRequest) {
                        if (!activityAlert.getServiceProviderAnnotations().equals(alert.getServiceProviderAnnotations())) {
                            return "Can't change service provider annotations";
                        }
                    }
                    if (StringUtils.isEmpty(activityAlert.getServiceProvider())) {
                        // can change params if created by user
                        return null;
                    }
                    return validateParamsNotChanged(alert, activityAlert);
                });
    }

    private String validateParamsNotChanged(AlertFromTemplatePersistent alert, AlertFromTemplatePersistent activityAlert) {
        Map<String, AlertParameter> paramsMap = new HashMap<>();
        for (var parameter : activityAlert.getParameters()) {
            paramsMap.put(parameter.getName(), parameter);
        }
        var oldParams = new HashSet<>(activityAlert.getParameters());
        for (var alertParameter : alert.getParameters()) {
            boolean remove = oldParams.remove(alertParameter);
            if (!remove) {
                var param = paramsMap.get(alertParameter.getName());
                if (param != null && param.isValueValid()) {
                    return "Parameter " + alertParameter.getName() + " can't be changed, instead change alert template.";
                } else {
                    oldParams.remove(param);
                }
            }
        }
        if (oldParams.size() != 0) {
            return "Can't remove parameters from alert, instead change alert template.";
        }
        return null;
    }

    private CompletableFuture<String> validateCreateAlertFromTemplate(AlertFromTemplatePersistent alert) {
        return alertTemplateDao.findById(alert.getTemplateId(), alert.getTemplateVersionTag())
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        return "Hasn't alert template with id " + alert.getTemplateId() + " and version " + alert.getTemplateVersionTag();
                    }
                    return validateAllParams(alert, alertTemplate.get());
                });
    }

    private String validateAllParams(AlertFromTemplatePersistent alert, AlertTemplate alertTemplate) {
        Map<String, AlertTemplateParameter.ParameterValueType> nameTypeMap = new HashMap<>();
        // collect all template param names+types
        for (AlertTemplateParameter parameter : alertTemplate.getParameters()) {
            nameTypeMap.put(parameter.getName(), parameter.getType());
        }
        // validate param name+type against template
        for (var parameter : alert.getParameters()) {
            if (!parameter.isValueValid()) {
                return "Alert parameter " + parameter.getName() + " value must be specified";
            }
            var result = validate(parameter, nameTypeMap, "parameter");
            if (result != null) {
                return result;
            }
        }
        if (!nameTypeMap.isEmpty()) {
            // value must be presented if parameter not specified
            for (AlertTemplateParameter parameter : alertTemplate.getParameters()) {
                if (nameTypeMap.containsKey(parameter.getName())) {
                    if (!parameter.isDefaultValueValid()) {
                        return "Alert parameter " + parameter.getName() + " value must be specified";
                    }
                }
            }
        }
        nameTypeMap.clear();
        // collect all template thresholds names+types
        for (AlertTemplateParameter parameter : alertTemplate.getThresholds()) {
            nameTypeMap.put(parameter.getName(), parameter.getType());
        }
        // validate thresholds name+type against template
        for (var parameter : alert.getThresholds()) {
            var result = validate(parameter, nameTypeMap, "threshold");
            if (result != null) {
                return result;
            }
        }

        return null;
    }

    private String validate(AlertParameter parameter, Map<String, AlertTemplateParameter.ParameterValueType> nameTypeMap, String paramName) {
        String name = parameter.getName();
        AlertTemplateParameter.ParameterValueType type = parameter.getType();
        AlertTemplateParameter.ParameterValueType parameterValueType = nameTypeMap.remove(name);
        if (parameterValueType == null) {
            return "Alert template hasn't " + paramName + " or " + paramName + " duplicated: " + name;
        }

        if (parameterValueType != type) {
            return "Alert template has " + paramName + " type " + parameterValueType + ", but here another type " + type;
        }
        return null;
    }
}
