package ru.yandex.solomon.alert.cluster.broker.alert;

import java.time.Clock;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.cluster.project.ProjectAssignment;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
public class SnapshotProcess {
    private final Logger logger = LoggerFactory.getLogger(SnapshotProcess.class);

    private final Clock clock;
    private final ProjectAssignment assignment;
    private final AlertStatesDao dao;
    private final Supplier<List<TPersistAlertState>> snapshotSupplier;
    private final SnapshotProcessMetrics metrics;

    private final PingActorRunner actor;
    private final AtomicReference<CompletableFuture<Void>> flushSync = new AtomicReference<>(new CompletableFuture<>());
    private volatile long latestFlush;

    public SnapshotProcess(
            Clock clock,
            Executor executor,
            ScheduledExecutorService timer,
            ProjectAssignment assignment,
            AlertStatesDao dao,
            Supplier<List<TPersistAlertState>> snapshotSupplier,
            Duration snapshotInterval)
    {
        this.actor = PingActorRunner.newBuilder()
                .pingInterval(snapshotInterval)
                .onPing(this::actFlush)
                .executor(executor)
                .operation("snapshot " + assignment)
                .timer(timer)
                .build();

        this.clock = clock;
        this.assignment = assignment;
        this.dao = dao;
        this.snapshotSupplier = snapshotSupplier;
        this.latestFlush = clock.millis();
        this.metrics = new SnapshotProcessMetrics(() -> clock.millis() - latestFlush);
    }

    public void run() {
        actor.schedule();
    }

    public SnapshotProcessMetrics getMetrics() {
        return metrics;
    }

    public CompletableFuture<?> flush() {
        var future = flushSync.get();
        actor.forcePing();
        return future;
    }

    private CompletableFuture<?> actFlush(int attempt) {
        var flushSync = this.flushSync.getAndSet(new CompletableFuture<>());
        var future = prepareAndFlushSnapshot();
        CompletableFutures.whenComplete(future, flushSync);
        return future;
    }

    private CompletableFuture<Void> prepareAndFlushSnapshot() {
        try {
            return metrics.async.wrapFuture(() -> {
                var now = clock.instant();
                List<TPersistAlertState> snapshot = snapshotSupplier.get();
                metrics.setSize(snapshot.stream().mapToInt(TPersistAlertState::getSerializedSize).sum());
                var future = snapshot.isEmpty()
                        ? CompletableFuture.completedFuture(null)
                        : dao.save(assignment.getProjectId(), now, assignment.getSeqNo(), snapshot);
                return future.thenRun(() -> latestFlush = now.toEpochMilli());
            });
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<?> awaitFlush() {
        return flushSync.get();
    }

    public void cancel() {
        actor.close();
        flushSync.get().completeExceptionally(new CancellationException());
    }
}
