package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.util.EnumMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collector;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.HistogramCollector;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.registry.MetricId;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;

/**
 * @author Vladimir Gordiychuk
 */
public class ActivityMetrics implements MetricSupplier {
    private final MetricRegistry registry;

    private final GaugeInt64 simpleCount;
    private final GaugeInt64 multiCount;
    private final GaugeInt64 subCount;
    private final GaugeInt64 failedCount;

    private final EnumMap<AlertState, GaugeInt64> alertStatesCount;
    private final EnumMap<ActivityStatus, GaugeInt64> activityStatusCount;
    private final Histogram evaluationSinceLastSec;

    private ActivityMetrics(Builder builder) {
        registry = new MetricRegistry();

        simpleCount = gaugeInt64(registry, "activity.simpleAlert.count", builder.simpleCount);
        multiCount = gaugeInt64(registry, "activity.multiAlert.count", builder.multiCount);
        subCount = gaugeInt64(registry, "activity.subAlert.count", builder.subCount);
        failedCount = gaugeInt64(registry, "activity.failedAlert.count", builder.failedCount);

        alertStatesCount = new EnumMap<>(AlertState.class);
        for (var state : AlertState.VALUES) {
            alertStatesCount.put(
                state,
                gaugeInt64(
                    registry,
                    "activity.alert.state.count",
                    Labels.of("state", state.name()),
                    builder.alertStatesCount.get(state)));
        }

        activityStatusCount = new EnumMap<>(ActivityStatus.class);
        for (var status : ActivityStatus.VALUES) {
            activityStatusCount.put(
                status,
                gaugeInt64(
                    registry,
                    "activity.status.count",
                    Labels.of("status", status.name()),
                    builder.activityStatusCount.get(status)));
        }

        evaluationSinceLastSec = registry.histogramCounter(
            "activity.evaluation.since_last_seconds",
            builder.evaluationSinceLastSec);
    }

    private static GaugeInt64 gaugeInt64(MetricRegistry registry, String name, long val) {
        return gaugeInt64(registry, name, Labels.of(), val);
    }

    private static GaugeInt64 gaugeInt64(MetricRegistry registry, String name, Labels labels, long val) {
        var id = new MetricId(name, labels);
        var gauge = new GaugeInt64(val);

        return (GaugeInt64) registry.putMetricIfAbsent(id, gauge);
    }

    public static ActivityMetrics empty() {
        return new Builder().build();
    }

    public static Collector<AlertActivity, Builder, ActivityMetrics> collector(long nowMillis) {
        return Collector.of(
            Builder::new,
            (metrics, activity) -> activity.fillMetrics(metrics, nowMillis),
            (left, right) -> {
                left.combine(right);
                return left;
            },
            Builder::build,
            Collector.Characteristics.UNORDERED);
    }

    public void combine(ActivityMetrics that) {
        this.simpleCount.combine(that.simpleCount);
        this.multiCount.combine(that.multiCount);
        this.subCount.combine(that.subCount);
        this.failedCount.combine(that.failedCount);

        for (var state : AlertState.VALUES) {
            this.alertStatesCount.get(state)
                .combine(that.alertStatesCount.get(state));
        }
        for (var status : ActivityStatus.VALUES) {
            this.activityStatusCount.get(status)
                .combine(that.activityStatusCount.get(status));
        }
        this.evaluationSinceLastSec.combine(that.evaluationSinceLastSec);
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }

    static class Builder extends Accumulator {
        private long simpleCount;
        private long multiCount;
        private long subCount;
        private long failedCount;

        private Builder() {
        }

        void add(SimpleAlertActivity activity, long nowMillis) {
            simpleCount += 1L;
            addCommonMonitored(activity, nowMillis);
        }

        void add(MultiAlertActivity activity, long nowMillis) {
            multiCount += 1L;
            combine(
                activity.getSubActivities()
                    .parallelStream()
                    .collect(SubAccumulator.collector(nowMillis)));
        }

        void add(FailedAlertActivity activity, long nowMillis) {
            failedCount += 1L;
            addCommonMonitored(activity, nowMillis);
        }

        void combine(Builder that) {
            this.simpleCount += that.simpleCount;
            this.multiCount += that.multiCount;
            this.subCount += that.subCount;
            this.failedCount += that.failedCount;

            this.alertStatesCount.addAll(that.alertStatesCount);
            this.activityStatusCount.addAll(that.activityStatusCount);
            this.evaluationSinceLastSec.combine(that.evaluationSinceLastSec);
        }

        void combine(SubAccumulator that) {
            this.subCount += that.subCount;

            this.alertStatesCount.addAll(that.alertStatesCount);
            this.activityStatusCount.addAll(that.activityStatusCount);
            this.evaluationSinceLastSec.combine(that.evaluationSinceLastSec);
        }

        ActivityMetrics build() {
            return new ActivityMetrics(this);
        }
    }

    private static class SubAccumulator extends Accumulator {

        long subCount;

        void add(SubAlertActivity activity, long nowMillis) {
            subCount += 1L;
            addCommonMonitored(activity, nowMillis);
        }

        void combine(SubAccumulator that) {
            this.subCount += that.subCount;

            this.alertStatesCount.addAll(that.alertStatesCount);
            this.activityStatusCount.addAll(that.activityStatusCount);
            this.evaluationSinceLastSec.combine(that.evaluationSinceLastSec);
        }

        static Collector<SubAlertActivity, SubAccumulator, SubAccumulator> collector(long nowMillis) {
            return Collector.of(
                SubAccumulator::new,
                (acc, activity) -> acc.add(activity, nowMillis),
                (left, right) -> {
                    left.combine(right);
                    return left;
                },
                Collector.Characteristics.UNORDERED);
        }
    }

    private abstract static class Accumulator {

        final EnumMapToLong<AlertState> alertStatesCount;
        final EnumMapToLong<ActivityStatus> activityStatusCount;
        final HistogramCollector evaluationSinceLastSec;

        Accumulator() {
            this.alertStatesCount = new EnumMapToLong<>(AlertState.class);
            this.activityStatusCount = new EnumMapToLong<>(ActivityStatus.class);
            this.evaluationSinceLastSec = Histograms.exponential(15, 2, 1);
        }

        void addCommonMonitored(MonitoredAlertActivity activity, long nowMillis) {
            AlertState alertState = activity.getAlert().getState();
            alertStatesCount.addAndGet(alertState, 1L);
            if (alertState == AlertState.ACTIVE) {
                activityStatusCount.addAndGet(activity.getStatus(), 1L);
                addEvaluationLag(activity.getLatestEvaluation(), nowMillis);
            }
        }

        private void addEvaluationLag(@Nullable EvaluationState evaluation, long nowMillis) {
            if (evaluation == null) {
                return;
            }

            var lagSeconds = TimeUnit.MILLISECONDS.toSeconds(nowMillis - evaluation.getLatestEval().toEpochMilli());
            evaluationSinceLastSec.collect(lagSeconds);
        }
    }
}
