package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.MoreObjects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.cluster.broker.evaluation.EvaluationObserver;
import ru.yandex.solomon.alert.cluster.broker.mute.MuteMatcher;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.mute.domain.AffectingMute;
import ru.yandex.solomon.alert.mute.domain.MuteStatus;
import ru.yandex.solomon.alert.notification.NotificationState;
import ru.yandex.solomon.alert.notification.state.StatefulNotificationChannel;
import ru.yandex.solomon.alert.protobuf.TPersistEvaluationState;
import ru.yandex.solomon.alert.protobuf.TPersistNotificationState;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.alert.rule.AlertProcessingState;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class EvaluationObserverActivity implements EvaluationObserver, MonitoredAlertActivity {
    private static final Logger logger = LoggerFactory.getLogger(SubAlertActivity.class);
    private static final AtomicReferenceFieldUpdater<EvaluationObserverActivity, ActivityStatus> STATUS_FIELD =
            AtomicReferenceFieldUpdater.newUpdater(EvaluationObserverActivity.class, ActivityStatus.class, "status");

    protected final Alert alert;
    protected final SimpleActivitiesFactory factory;
    protected final Map<String, StatefulNotificationChannel> notificationChannels;
    @Nullable
    protected volatile AlertProcessingState alertProcessingState;
    protected volatile ActivityStatus status = ActivityStatus.IDLE;
    @Nullable
    protected volatile Flow.Subscription subscription;
    protected volatile Throwable latestError;

    protected EvaluationObserverActivity(
            Alert alert,
            SimpleActivitiesFactory factory,
            Map<String, StatefulNotificationChannel> notificationChannels)
    {
        this.alert = alert;
        this.factory = factory;
        this.notificationChannels = notificationChannels;
    }

    @Override
    public void run() {
        if (alert.getState() != AlertState.ACTIVE) {
            return;
        }

        if (isCanceled()) {
            return;
        }

        if (STATUS_FIELD.compareAndSet(this, ActivityStatus.IDLE, ActivityStatus.ASSIGN)) {
            status = ActivityStatus.ASSIGN;
            factory.getAssignmentService().assign(factory.getAssignment(), alert, this);
        }
    }

    @ManagerMethod
    @SuppressWarnings("unused")
    public void forceRun() {
        factory.getAssignmentService().assign(factory.getAssignment(), alert, this);
    }

    @Override
    public ActivityStatus getStatus() {
        return status;
    }

    protected ActivityStatus actualizeStatus(ActivityStatus next) {
        ActivityStatus status = STATUS_FIELD.get(this);
        if (status == next) {
            return next;
        }

        if (status == ActivityStatus.CANCELED) {
            return status;
        }

        if (!STATUS_FIELD.compareAndSet(this, status, next)) {
            return STATUS_FIELD.get(this);
        }

        return next;
    }

    @Override
    public void cancel() {
        status = ActivityStatus.CANCELED;
        for (StatefulNotificationChannel channel : notificationChannels.values()) {
            channel.close();
        }
        Flow.Subscription sub = subscription;
        if (sub != null) {
            sub.cancel();
            subscription = null;
        }
    }

    @Override
    public boolean isCanceled() {
        return status == ActivityStatus.CANCELED;
    }

    @Override
    public void onSubscribe(Flow.Subscription s) {
        subscription = s;
        if (status == ActivityStatus.CANCELED) {
            s.cancel();
            subscription = null;
        } else {
            s.request(Long.MAX_VALUE);
        }
    }

    @Override
    public void onError(Throwable throwable) {
        latestError = throwable;
        actualizeStatus(ActivityStatus.ERROR);
    }

    @Override
    public void onComplete() {
        actualizeStatus(ActivityStatus.IDLE);
    }

    @Override
    public void onNext(EvaluationState value) {
        if (actualizeStatus(ActivityStatus.RUNNING) == ActivityStatus.CANCELED) {
            logger.info("{} skip by canceled activity", alert.getKey());
            return;
        }

        if (skipReporting(value.getStatus())) {
            logger.info("{} skip report {}", alert.getKey(), value);
            return;
        }

        AlertMuteStatus mutes = matchMutes(factory.getMuteMatcher(), value.getLatestEval());
        alertProcessingState = new AlertProcessingState(value, mutes);
        var isSentToDefault = false;
        for (StatefulNotificationChannel channel : notificationChannels.values()) {
            if (channel.isDefault() && !isSentToDefault) {
                isSentToDefault = true;
                channel.send(alertProcessingState);
            } else if (!channel.isDefault()) {
                channel.send(alertProcessingState);
            }
        }
    }

    public AlertMuteStatus matchMutes(MuteMatcher matcher, Instant latestEval) {
        final Labels subAlertLabels;
        final String alertId;
        if (alert.getAlertType() == AlertType.SUB_ALERT) {
            SubAlert subAlert = (SubAlert) alert;
            subAlertLabels = subAlert.getGroupKey();
            alertId = subAlert.getParent().getId();
        } else {
            subAlertLabels = Labels.of();
            alertId = alert.getId();
        }

        List<AffectingMute> affectingMutes = matcher.match(alertId, subAlertLabels, latestEval);

        boolean alertIsMuted = affectingMutes.stream()
                .anyMatch(affectingMute -> affectingMute.status() == MuteStatus.ACTIVE);
        var alertMuteCode = alertIsMuted ? AlertMuteStatus.MuteStatusCode.MUTED :
                AlertMuteStatus.MuteStatusCode.NOT_MUTED;

        return new AlertMuteStatus(alertMuteCode, affectingMutes);
    }

    @Nullable
    @Override
    public EvaluationState getLatestEvaluation() {
        return EvaluationObserver.super.getLatestEvaluation();
    }

    @Nullable
    @Override
    final public AlertProcessingState getProcessingState() {
        return alertProcessingState;
    }

    protected boolean skipReporting(EvaluationStatus status) {
        return switch (status.getErrorCode()) {
            case DEADLINE, UNAVAILABLE -> true;
            default -> false;
        };
    }

    public Stream<NotificationState> getNotificationStates(Set<String> notificationIds) {
        if (notificationChannels.isEmpty()) {
            return Stream.empty();
        }

        Stream<NotificationState> result = notificationChannels.values()
                .stream()
                .map(StatefulNotificationChannel::getLatestNotificationState);

        if (!notificationIds.isEmpty()) {
            result = result.filter(state -> notificationIds.contains(state.getKey().getNotificationId()));
        }

        return result;
    }

    protected List<TPersistNotificationState> dumpNotifications() {
        if (notificationChannels.isEmpty()) {
            return Collections.emptyList();
        }

        List<TPersistNotificationState> result = new ArrayList<>(notificationChannels.size());
        for (StatefulNotificationChannel channel : notificationChannels.values()) {
            result.add(channel.dumpState());
        }
        return result;
    }

    protected void restoreEvaluationState(TPersistEvaluationState proto) {
        if (proto.getLatestEvalMillis() == 0) {
            return;
        }

        var evaluationState = EvaluationState.newBuilder(alert)
                .setSince(Instant.ofEpochMilli(proto.getSinceMillis()))
                .setLatestEval(Instant.ofEpochMilli(proto.getLatestEvalMillis()))
                .setStatus(AlertConverter.protoToStatus(proto.getStatus()))
                .setPreviousStatus(AlertConverter.protoToStatus(proto.getPreviousStatus()))
                .build();

        AlertMuteStatus muteState = AlertConverter.protoToMuteStatus(proto.getMuteStatus());

        alertProcessingState = new AlertProcessingState(evaluationState, muteState);
    }

    protected void restoreNotificationsState(List<TPersistNotificationState> list) {
        for (TPersistNotificationState proto : list) {
            StatefulNotificationChannel channel = notificationChannels.get(proto.getNotificationChannelId());
            if (channel != null) {
                channel.restoreState(proto);
            }
        }
    }

    @Override
    public String toString() {
        final var frozen = Optional.ofNullable(alertProcessingState);
        return MoreObjects.toStringHelper(this.getClass().getSimpleName())
                .add("key", alert.getKey())
                .add("status", frozen.map(AlertProcessingState::evaluationState)
                        .map(EvaluationState::getStatus).orElse(null))
                .add("mutes", frozen.map(AlertProcessingState::alertMuteStatus).orElse(null))
                .omitNullValues()
                .toString();
    }
}
