package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;
import com.google.common.base.Throwables;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.cluster.broker.mute.MuteMatcher;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.notification.state.StatefulNotificationChannel;
import ru.yandex.solomon.alert.protobuf.PersistFailedAlertState;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.alert.protobuf.TPersistNotificationState;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.alert.rule.EvaluationState;

/**
 * @author Alexey Trushkin
 */
public class FailedAlertActivity implements AlertActivity, MonitoredAlertActivity {

    private static final AlertMuteStatus ALERT_MUTE_STATUS = new AlertMuteStatus(AlertMuteStatus.MuteStatusCode.NOT_MUTED, List.of());
    private final EvaluationStatus status;
    private final EvaluationState evaluationState;
    private final Alert alert;
    private final Throwable loadingException;
    private final Map<String, StatefulNotificationChannel> channels;
    private volatile boolean canceled;

    public FailedAlertActivity(Alert alert, Throwable loadingException, Map<String, StatefulNotificationChannel> channels) {
        this.alert = alert;
        this.loadingException = loadingException;
        this.channels = channels;
        status = EvaluationStatus.ERROR.withDescription(Throwables.getStackTraceAsString(loadingException));
        evaluationState = EvaluationState.newBuilder(alert)
                .setStatus(status)
                .setSince(Instant.now())
                .build();
    }

    @Override
    public boolean isCanceled() {
        return canceled;
    }

    @Override
    public Alert getAlert() {
        return alert;
    }

    @Override
    public int countSubAlerts() {
        return 0;
    }

    @Override
    public void run() {
        // do nothing
    }

    @Override
    public void cancel() {
        canceled = true;
    }

    @Override
    public TPersistAlertState dumpState() {
        return TPersistAlertState.newBuilder()
                .setId(alert.getId())
                .setVersion(alert.getVersion())
                .setPersistFailedAlertState(PersistFailedAlertState.newBuilder()
                        .addAllNotificationStates(dumpNotifications())
                        .build())
                .build();
    }

    private List<TPersistNotificationState> dumpNotifications() {
        if (channels.isEmpty()) {
            return Collections.emptyList();
        }

        List<TPersistNotificationState> result = new ArrayList<>(channels.size());
        for (StatefulNotificationChannel channel : channels.values()) {
            result.add(channel.dumpState());
        }
        return result;
    }

    @Override
    public void restore(TPersistAlertState state) {
        if (alert.getVersion() != state.getVersion()) {
            return;
        }
        if (!state.hasPersistFailedAlertState()) {
            return;
        }

        var failedState = state.getPersistFailedAlertState();
        restoreNotificationsState(failedState.getNotificationStatesList());
    }

    protected void restoreNotificationsState(List<TPersistNotificationState> list) {
        for (TPersistNotificationState proto : list) {
            StatefulNotificationChannel channel = channels.get(proto.getNotificationChannelId());
            if (channel != null) {
                channel.restoreState(proto);
            }
        }
    }

    @Override
    public void appendAlertMetrics(MetricConsumer consumer) {
        consumer.onMetricBegin(MetricType.IGAUGE);
        consumer.onLabelsBegin(3);
        consumer.onLabel("sensor", "alert.evaluation.status");
        consumer.onLabel("projectId", alert.getProjectId());
        consumer.onLabel("alertId", alert.getId());
        consumer.onLabelsEnd();
        consumer.onLong(0, AlertStatusCodec.encode(status, ALERT_MUTE_STATUS));
        consumer.onMetricEnd();
    }

    @Override
    public void appendEvaluationStatistics(EvaluationSummaryStatistics summary) {
        summary.add(evaluationState);
    }

    @Override
    public void appendMutedStatistics(EvaluationSummaryStatistics summary, Set<String> muteIds, @Nullable MuteMatcher matcher) {
        // do nothing
    }

    @Override
    public void appendNotificationStatistics(NotificationSummaryStatistics summary, Set<String> notificationIds) {
        channels.values()
                .stream()
                .map(StatefulNotificationChannel::getLatestNotificationState)
                .filter(state -> notificationIds.contains(state.getKey().getNotificationId()))
                .forEach(summary::add);
    }

    @Override
    public void fillMetrics(ActivityMetrics.Builder metrics, long nowMillis) {
        metrics.add(this, nowMillis);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this.getClass().getSimpleName())
                .add("key", alert.getKey())
                .add("cancelled", canceled)
                .toString() + '@' + Integer.toHexString(System.identityHashCode(this));
    }

    @Override
    public ActivityStatus getStatus() {
        return ActivityStatus.ERROR;
    }

    @Override
    public EvaluationState getLatestEvaluation() {
        return evaluationState;
    }
}
