package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.base.MoreObjects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.cluster.broker.mute.MuteMatcher;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.domain.AlertingLabelsAllocator;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.alert.protobuf.TPersistMultiAlertState;
import ru.yandex.solomon.alert.protobuf.TPersistSubAlertState;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.alert.rule.AlertProcessingState;
import ru.yandex.solomon.alert.unroll.MultiAlertUtils;
import ru.yandex.solomon.alert.unroll.UnrollExecutor;
import ru.yandex.solomon.alert.unroll.UnrollResult;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.util.collection.enums.EnumMapToLong;

/**
 * @author Vladimir Gordiychuk
 */
public class MultiAlertActivity implements AlertActivity, UnrollExecutor.UnrollConsumer {
    private static final Logger logger = LoggerFactory.getLogger(MultiAlertActivity.class);

    private final Alert parent;
    private final SimpleActivitiesFactory factory;
    private final Map<String, SubAlertActivity> subAlertById = new ConcurrentHashMap<>();
    private volatile boolean canceled;

    public MultiAlertActivity(Alert parent, SimpleActivitiesFactory factory) {
        this.parent = parent;
        this.factory = factory;
    }

    @Override
    public boolean isCanceled() {
        return canceled;
    }

    @Override
    public void accept(UnrollResult result) {
        if (canceled) {
            return;
        }

        Map<String, Labels> fresh = result.labels.stream()
                .collect(Collectors.toMap(key -> MultiAlertUtils.getAlertId(parent, key), Function.identity()));

        // Delete obsolete sub alerts
        if (result.allowDelete) {
            Iterator<Map.Entry<String, SubAlertActivity>> it = subAlertById.entrySet().iterator();
            while (it.hasNext()) {
                Map.Entry<String, SubAlertActivity> entry = it.next();
                if (!fresh.containsKey(entry.getKey())) {
                    logger.info("{} alert doesn't have group {} anymore", parent.getKey(), entry.getKey());
                    entry.getValue().cancel();
                    it.remove();
                }
            }
        }

        // Add new sub alerts
        for (Map.Entry<String, Labels> entry : fresh.entrySet()) {
            if (subAlertById.containsKey(entry.getKey())) {
                continue;
            }

            logger.info("{} alert have a new group {}", parent.getKey(), entry.getValue());
            SubAlert subAlert = SubAlert.newBuilder()
                    .setId(MultiAlertUtils.getAlertId(parent, entry.getValue()))
                    .setParent(parent)
                    .setGroupKey(entry.getValue())
                    .build();

            SubAlertActivity activity = new SubAlertActivity(subAlert, this, factory);
            subAlertById.put(subAlert.getId(), activity);
            activity.run();
        }

        if (canceled) {
            cancel();
        }
    }

    @Override
    public Alert getAlert() {
        return parent;
    }

    @Override
    public int countSubAlerts() {
        return subAlertById.size();
    }

    @Override
    public void run() {
        if (canceled) {
            return;
        }

        if (parent.getState() != AlertState.ACTIVE) {
            return;
        }

        subAlertById.values().forEach(SubAlertActivity::run);
        UnrollExecutor unrollExecutor = factory.getUnrollExecutor();
        if (subAlertById.isEmpty()) {
            // alert not unrolling yet at all, do it as fast as possible
            unrollExecutor.unrollNow(parent, this);
        } else {
            unrollExecutor.unroll(parent, this);
        }
    }

    @Override
    public void cancel() {
        canceled = true;
        subAlertById.values().forEach(SubAlertActivity::cancel);
    }

    public Collection<SubAlertActivity> getSubActivities() {
        return subAlertById.values();
    }

    public SubAlertActivity getSubActivity(String id) {
        return subAlertById.get(id);
    }

    @Override
    public TPersistAlertState dumpState() {
        return TPersistAlertState.newBuilder()
                .setId(parent.getId())
                .setVersion(parent.getVersion())
                .setMultiAlertState(TPersistMultiAlertState.newBuilder()
                        .addAllSubAlerts(dumpSubAlertsState())
                        .build())
                .build();
    }

    private List<TPersistSubAlertState> dumpSubAlertsState() {
        if (subAlertById.isEmpty()) {
            return Collections.emptyList();
        }

        return subAlertById.values()
                .stream()
                .map(SubAlertActivity::dumpState)
                .collect(Collectors.toList());
    }

    @Override
    public void restore(TPersistAlertState state) {
        if (parent.getVersion() != state.getVersion()) {
            return;
        }
        if (state.hasPersistFailedAlertState()) {
            return;
        }

        for (TPersistSubAlertState subState : state.getMultiAlertState().getSubAlertsList()) {
            Labels key = LabelConverter.protoToLabels(subState.getLabelsList(), AlertingLabelsAllocator.I);
            SubAlert subAlert = SubAlert.newBuilder()
                    .setId(MultiAlertUtils.getAlertId(parent, key))
                    .setParent(parent)
                    .setGroupKey(key)
                    .build();

            SubAlertActivity activity = new SubAlertActivity(subAlert, this, factory);
            SubAlertActivity prev = subAlertById.putIfAbsent(subAlert.getId(), activity);
            if (prev != null) {
                continue;
            }

            activity.restore(subState);
        }
    }

    @Override
    public void appendAlertMetrics(MetricConsumer consumer) {
        if (subAlertById.isEmpty()) {
            return;
        }

        EnumMapToLong<EvaluationStatus.Code> countByCode = new EnumMapToLong<>(EvaluationStatus.Code.class);
        EnumMapToLong<EvaluationStatus.Code> countMutedByCode = new EnumMapToLong<>(EvaluationStatus.Code.class);
        for (SubAlertActivity activity : subAlertById.values()) {
            AlertProcessingState state = activity.getProcessingState();
            if (state == null) {
                continue;
            }

            appendSubAlertMetric(state, consumer);
            var evalStatusCode = state.evaluationState().getStatus().getCode();
            countByCode.addAndGet(evalStatusCode, 1);
            if (state.alertMuteStatus().statusCode() == AlertMuteStatus.MuteStatusCode.MUTED) {
                countMutedByCode.addAndGet(evalStatusCode, 1);
            }
        }

        appendMultiAlertMetrics(countByCode, countMutedByCode, consumer);
    }

    @Override
    public void appendEvaluationStatistics(EvaluationSummaryStatistics summary) {
        EvaluationSummaryStatistics multiSummary = subAlertById.values()
                .parallelStream()
                .map(SubAlertActivity::getLatestEvaluation)
                .collect(EvaluationSummaryStatistics::new, EvaluationSummaryStatistics::add, EvaluationSummaryStatistics::combine);
        summary.combine(multiSummary);
    }

    @Override
    public void appendMutedStatistics(EvaluationSummaryStatistics summary, Set<String> muteIds, @Nullable MuteMatcher matcher) {
        EvaluationSummaryStatistics multiSummary;
        if (matcher == null) {
            multiSummary = subAlertById.values()
                    .parallelStream()
                    .map(SubAlertActivity::getProcessingState)
                    .filter(Objects::nonNull)
                    .filter(state -> state.alertMuteStatus().containsAnyOf(muteIds))
                    .map(AlertProcessingState::evaluationState)
                    .collect(EvaluationSummaryStatistics::new, EvaluationSummaryStatistics::add, EvaluationSummaryStatistics::combine);
        } else {
            multiSummary = subAlertById.values()
                    .parallelStream()
                    .filter(saa -> saa.matchMutes(matcher, Instant.EPOCH).isMuted())
                    .map(SubAlertActivity::getLatestEvaluation)
                    .collect(EvaluationSummaryStatistics::new, EvaluationSummaryStatistics::add, EvaluationSummaryStatistics::combine);
        }
        summary.combine(multiSummary);
    }

    @Override
    public void appendNotificationStatistics(NotificationSummaryStatistics summary, Set<String> notificationIds) {
        NotificationSummaryStatistics multiSummary = subAlertById.values()
                .parallelStream()
                .flatMap(sub -> sub.getNotificationStates(notificationIds))
                .collect(NotificationSummaryStatistics::new, NotificationSummaryStatistics::add, NotificationSummaryStatistics::combine);
        summary.combine(multiSummary);
    }

    @Override
    public void fillMetrics(ActivityMetrics.Builder metrics, long nowMillis) {
        metrics.add(this, nowMillis);
    }

    private void appendMultiAlertMetrics(EnumMapToLong<EvaluationStatus.Code> countByCode, EnumMapToLong<EvaluationStatus.Code> countMutedByCode, MetricConsumer consumer) {
        for (EvaluationStatus.Code code : EvaluationStatus.Code.values()) {
            consumer.onMetricBegin(MetricType.IGAUGE);
            consumer.onLabelsBegin(4);
            consumer.onLabel("sensor", "multiAlert.evaluation.status");
            consumer.onLabel("projectId", parent.getProjectId());
            consumer.onLabel("alertId", parent.getId());
            consumer.onLabel("status", code.name());
            consumer.onLabelsEnd();
            consumer.onLong(0, countByCode.get(code));
            consumer.onMetricEnd();
        }
        for (EvaluationStatus.Code code : EvaluationStatus.Code.values()) {
            consumer.onMetricBegin(MetricType.IGAUGE);
            consumer.onLabelsBegin(4);
            consumer.onLabel("sensor", "multiAlert.evaluation.status");
            consumer.onLabel("projectId", parent.getProjectId());
            consumer.onLabel("alertId", parent.getId());
            consumer.onLabel("status", code.name() + ":MUTED");
            consumer.onLabelsEnd();
            consumer.onLong(0, countMutedByCode.get(code));
            consumer.onMetricEnd();
        }
    }

    private void appendSubAlertMetric(AlertProcessingState state, MetricConsumer consumer) {
        consumer.onMetricBegin(MetricType.IGAUGE);
        consumer.onLabelsBegin(4);
        consumer.onLabel("sensor", "alert.evaluation.status");
        consumer.onLabel("projectId", parent.getProjectId());
        consumer.onLabel("parentId", parent.getId());
        consumer.onLabel("alertId", state.evaluationState().getAlertId());
        consumer.onLabelsEnd();
        consumer.onLong(0, AlertStatusCodec.encode(state));
        consumer.onMetricEnd();
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this.getClass().getSimpleName())
            .add("key", parent.getKey())
            .add("subAlerts", subAlertById.size())
            .add("cancelled", canceled)
            .omitNullValues()
            .toString() + '@' + Integer.toHexString(System.identityHashCode(this));
    }
}
