package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.time.Instant;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.solomon.alert.cluster.broker.mute.MuteMatcher;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.protobuf.TPersistAlertState;
import ru.yandex.solomon.alert.protobuf.TPersistSimpleAlertState;
import ru.yandex.solomon.alert.rule.AlertMuteStatus;
import ru.yandex.solomon.alert.rule.AlertProcessingState;

/**
 * @author Vladimir Gordiychuk
 */
public class SimpleAlertActivity extends EvaluationObserverActivity implements AlertActivity {
    public SimpleAlertActivity(Alert alert, SimpleActivitiesFactory factory) {
        super(alert, factory, factory.getNotificationChannelFactory().prepareChannels(alert));
    }

    @Override
    public Alert getAlert() {
        return alert;
    }

    @Override
    public int countSubAlerts() {
        return 0;
    }

    @Override
    public TPersistAlertState dumpState() {
        return TPersistAlertState.newBuilder()
            .setId(alert.getId())
            .setVersion(alert.getVersion())
            .setSimpleAlertState(TPersistSimpleAlertState.newBuilder()
                .setEvaluation(PersistStateConverter.processingStateToProto(alertProcessingState))
                .addAllNotifications(dumpNotifications())
                .build())
            .build();
    }

    @Override
    public void appendAlertMetrics(MetricConsumer consumer) {
        AlertProcessingState state = getProcessingState();
        if (state == null) {
            return;
        }

        consumer.onMetricBegin(MetricType.IGAUGE);
        consumer.onLabelsBegin(3);
        consumer.onLabel("sensor", "alert.evaluation.status");
        consumer.onLabel("projectId", alert.getProjectId());
        consumer.onLabel("alertId", alert.getId());
        consumer.onLabelsEnd();
        consumer.onLong(0, AlertStatusCodec.encode(state));
        consumer.onMetricEnd();
    }

    @Override
    public void appendEvaluationStatistics(EvaluationSummaryStatistics summary) {
        summary.add(getLatestEvaluation());
    }

    @Override
    public void appendMutedStatistics(EvaluationSummaryStatistics summary, Set<String> muteIds, @Nullable MuteMatcher matcher) {
        if (matcher == null) {
            AlertMuteStatus muteStatus = getMuteStatus();
            if (muteStatus != null && muteStatus.containsAnyOf(muteIds)) {
                summary.add(getLatestEvaluation());
            }
        } else {
            if (matchMutes(matcher, Instant.EPOCH).isMuted()) {
                summary.add(getLatestEvaluation());
            }
        }
    }

    @Override
    public void appendNotificationStatistics(NotificationSummaryStatistics summary, Set<String> notificationIds) {
        getNotificationStates(notificationIds).forEach(summary::add);
    }

    @Override
    public void fillMetrics(ActivityMetrics.Builder metrics, long nowMillis) {
        metrics.add(this, nowMillis);
    }

    @Override
    public void restore(TPersistAlertState proto) {
        if (alert.getVersion() != proto.getVersion()) {
            return;
        }
        if (proto.hasPersistFailedAlertState()) {
            return;
        }

        TPersistSimpleAlertState simpleState = proto.getSimpleAlertState();
        restoreEvaluationState(simpleState.getEvaluation());
        restoreNotificationsState(simpleState.getNotificationsList());
    }
}
