package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertInterner;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class TemplateActivityFactory {

    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(3)
            .withDelay(1_000)
            .withMaxDelay(60_000);
    private final AlertTemplateDao alertTemplateDao;
    private final TemplateAlertFactory templateAlertFactory;
    private final SimpleActivitiesFactory simpleActivitiesFactory;

    public TemplateActivityFactory(
            AlertTemplateDao alertTemplateDao,
            TemplateAlertFactory templateFactory,
            SimpleActivitiesFactory simpleActivitiesFactory)
    {
        this.alertTemplateDao = alertTemplateDao;
        this.templateAlertFactory = templateFactory;
        this.simpleActivitiesFactory = simpleActivitiesFactory;
    }

    public CompletableFuture<AlertActivity> makeActivity(AlertFromTemplatePersistent alert) {
        return CompletableFutures.safeCall(() -> RetryCompletableFuture.runWithRetries(() ->
                alertTemplateDao.findById(alert.getTemplateId(), alert.getTemplateVersionTag())
                        .thenApply(alertTemplate -> {
                            if (alertTemplate.isEmpty()) {
                                throw new IllegalArgumentException("No alert template with id " + alert.getTemplateId());
                            }
                            return makeActivity(alert, alertTemplate.get());
                        }), RETRY_CONFIG));
    }

    @NotNull
    private AlertActivity makeActivity(AlertFromTemplatePersistent alert, AlertTemplate template) {
        Alert templated = templateAlertFactory.createAlertFromTemplate(alert, template);
        return new TemplateAlertActivity(
                (AlertFromTemplatePersistent) AlertInterner.I.intern(alert),
                AlertInterner.I.intern(templated),
                simpleActivitiesFactory,
                template);
    }
}
