package ru.yandex.solomon.alert.cluster.broker.alert.activity;

import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.domain.AbstractAlertBuilder;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.domain.template.AlertParameter;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.alert.template.domain.expression.ExpressionAlertTemplate;
import ru.yandex.solomon.alert.template.domain.threshold.TemplatePredicateRule;
import ru.yandex.solomon.alert.template.domain.threshold.ThresholdAlertTemplate;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class TemplateAlertFactory {

    @Deprecated
    private static final String PARAMETER_TEMPLATE_PREFIX = "templateParameter.";
    private static final String PARAMETERS_TEMPLATE_PREFIX = "templateParameters.";

    private final TemplateFactory templateFactory;

    public TemplateAlertFactory(TemplateFactory templateFactory) {
        this.templateFactory = templateFactory;
    }

    public Alert createAlertFromTemplate(AlertFromTemplatePersistent alert, AlertTemplate template) {
        alert.setServiceProviderLabels(getLabels(template));
        var params = prepareParams(alert, template);
        AbstractAlertBuilder<?, ?> builder;
        switch (template.getAlertTemplateType()) {
            case THRESHOLD -> builder = createThresholdAlertBuilder((ThresholdAlertTemplate) template, params);
            case EXPRESSION -> builder = createExpressionAlertBuilder((ExpressionAlertTemplate) template, params);
            default -> throw new IllegalArgumentException("Cant create alert from template with type " + alert.getAlertType());
        }
        builder.setId(alert.getId())
                .setProjectId(alert.getProjectId())
                .setFolderId(alert.getFolderId())
                .setName(alert.getName())
                .setDescription(alert.getDescription())
                .setVersion(alert.getVersion())
                .setCreatedBy(alert.getCreatedBy())
                .setCreatedAt(alert.getCreatedAt())
                .setUpdatedBy(alert.getUpdatedBy())
                .setUpdatedAt(alert.getUpdatedAt())
                .setState(alert.getState())
                .setLabels(getLabels(alert, template))
                .setNotificationChannels(alert.getNotificationChannels())
                .setEscalations(alert.getEscalations())
                .setAnnotations(prepareValueFromParams(alert.getAnnotations(), params))
                .setServiceProviderAnnotations(prepareValueFromParams(alert.getServiceProviderAnnotations(), params))
                // from template and params
                .setObtainedFromTemplate(true)
                .setGroupByLabels(prepareValueFromParams(template.getGroupByLabels(), params))
                .setPeriod(Duration.ofMillis(template.getPeriodMillis()))
                .setDelaySeconds(template.getDelaySeconds())
                .setSeverity(template.getSeverity())
                .setResolvedEmptyPolicy(template.getResolvedEmptyPolicy())
                .setNoPointsPolicy(template.getNoPointsPolicy());
        return builder.build();
    }

    private Map<String, String> getLabels(AlertFromTemplatePersistent alert, AlertTemplate template) {
        var labels = new HashMap<>(alert.getLabels());
        labels.put("templateId", template.getId());
        labels.put("templateVersionTag", template.getTemplateVersionTag());
        labels.put("serviceProviderId", template.getServiceProviderId());
        return labels;
    }

    private Map<String, String> getLabels(AlertTemplate template) {
        Map<String, String> labels = new HashMap<>();
        labels.put("templateId", template.getId());
        labels.put("templateVersionTag", template.getTemplateVersionTag());
        labels.put("serviceProviderId", template.getServiceProviderId());
        return labels;
    }

    private Map<String, Object> prepareParams(AlertFromTemplatePersistent alert, AlertTemplate template) {
        SkipUnknownValuesMap result = new SkipUnknownValuesMap();
        // fill template params and their defaults
        for (AlertTemplateParameter parameter : template.getParameters()) {
            prepareParam(parameter, result);
        }
        for (AlertTemplateParameter parameter : template.getThresholds()) {
            prepareParam(parameter, result);
        }
        // override param defaults from alert
        for (AlertParameter parameter : alert.getParameters()) {
            prepareParam(parameter, result);
        }
        for (AlertParameter parameter : alert.getThresholds()) {
            prepareParam(parameter, result);
        }
        return result;
    }

    @Deprecated
    private String prepareParamNameOld(String name) {
        return PARAMETER_TEMPLATE_PREFIX + name;
    }

    private String prepareParamName(String name) {
        return PARAMETERS_TEMPLATE_PREFIX + name;
    }

    private void prepareParam(AlertParameter parameter,
                              SkipUnknownValuesMap result)
    {
        if (!result.originalContainsKey(prepareParamNameOld(parameter.getName())) && !result.originalContainsKey(prepareParamName(parameter.getName()))) {
            // skip unknown params
            return;
        }
        if (parameter instanceof AlertParameter.IntegerParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), param.getValue());
            result.put(prepareParamName(parameter.getName()), param.getValue());
        } else if (parameter instanceof AlertParameter.DoubleParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), param.getValue());
            result.put(prepareParamName(parameter.getName()), param.getValue());
        } else if (parameter instanceof AlertParameter.TextListParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), String.join("|", param.getValue()));
            result.put(prepareParamName(parameter.getName()), String.join("|", param.getValue()));
        } else if (parameter instanceof AlertParameter.TextParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), param.getValue());
            result.put(prepareParamName(parameter.getName()), param.getValue());
        } else if (parameter instanceof AlertParameter.LabelListParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), String.join("|", param.getValues()));
            result.put(prepareParamName(parameter.getName()), String.join("|", param.getValues()));
        } else {
            throw new IllegalArgumentException("Unknown alert parameter type: " + parameter.getClass());
        }
    }

    private void prepareParam(AlertTemplateParameter parameter,
                              Map<String, Object> result)
    {
        if (parameter instanceof AlertTemplateParameter.IntegerParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), param.getDefaultValue());
            result.put(prepareParamName(parameter.getName()), param.getDefaultValue());
        } else if (parameter instanceof AlertTemplateParameter.DoubleParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), param.getDefaultValue());
            result.put(prepareParamName(parameter.getName()), param.getDefaultValue());
        } else if (parameter instanceof AlertTemplateParameter.TextListParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), String.join("|", param.getDefaultValues()));
            result.put(prepareParamName(parameter.getName()), String.join("|", param.getDefaultValues()));
        } else if (parameter instanceof AlertTemplateParameter.TextParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), param.getDefaultValue());
            result.put(prepareParamName(parameter.getName()), param.getDefaultValue());
        } else if (parameter instanceof AlertTemplateParameter.LabelListParameterValue param) {
            result.put(prepareParamNameOld(parameter.getName()), String.join("|", param.getDefaultValues()));
            result.put(prepareParamName(parameter.getName()), String.join("|", param.getDefaultValues()));
        } else {
            throw new IllegalArgumentException("Unknown alert template parameter type: " + parameter.getClass());
        }
    }

    private AbstractAlertBuilder<?, ?> createExpressionAlertBuilder(
            ExpressionAlertTemplate template,
            Map<String, Object> params)
    {
        return ExpressionAlert.newBuilder()
                .setProgram(prepareValueFromParams(template.getProgram(), params));
    }

    private AbstractAlertBuilder<?, ?> createThresholdAlertBuilder(
            ThresholdAlertTemplate template,
            Map<String, Object> params)
    {
        return ThresholdAlert.newBuilder()
                .setTransformations(prepareValueFromParams(template.getTransformations(), params))
                .setSelectors(prepareValueFromParams(template.getSelectors(), params))
                .setPredicateRules(template.getPredicateRules().stream()
                        .map(templatePredicateRule -> PredicateRule
                                .onThreshold(prepareValueFromParams(templatePredicateRule, params))
                                .withThresholdType(templatePredicateRule.getThresholdType())
                                .withComparison(templatePredicateRule.getComparison())
                                .withTargetStatus(templatePredicateRule.getTargetStatus())));
    }

    private double prepareValueFromParams(TemplatePredicateRule templatePredicateRule, Map<String, Object> params) {
        if (!StringUtils.isEmpty(templatePredicateRule.getThresholdParameterTemplate())) {
            String s = prepareValueFromParams(templatePredicateRule.getThresholdParameterTemplate(), params);
            try {
                return Double.parseDouble(s);
            } catch (Exception ignore) {
            }
        }
        return templatePredicateRule.getThreshold();
    }

    private String prepareValueFromParams(String value, Map<String, Object> params) {
        return templateFactory.createTemplate(value).process(params);
    }

    private List<String> prepareValueFromParams(List<String> values, Map<String, Object> params) {
        return values.stream()
                .map(s -> prepareValueFromParams(s, params))
                .collect(Collectors.toList());
    }

    private Map<String, String> prepareValueFromParams(Map<String, String> values, Map<String, Object> params) {
        Map<String, String> result = new HashMap<>(values.size());
        for (var entry : values.entrySet()) {
            result.put(prepareValueFromParams(entry.getKey(), params), prepareValueFromParams(entry.getValue(), params));
        }
        return result;
    }

    // save unknown params to render them later
    private static class SkipUnknownValuesMap extends HashMap<String, Object> {

        @Override
        public boolean containsKey(Object key) {
            return true;
        }

        @Override
        public Object get(Object key) {
            var value = super.get(key);
            if (value == null) {
                return "{{" + key + "}}";
            }
            return value;
        }

        public boolean originalContainsKey(String prepareParamName) {
            return super.containsKey(prepareParamName);
        }
    }
}
