package ru.yandex.solomon.alert.cluster.broker.alert.activity.search;

import java.time.Instant;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.api.converters.NotificationConverter;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.ActivityFactory;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.AlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.FailedAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.MultiAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SimpleAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SubAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.TemplateAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.mute.MuteMatcher;
import ru.yandex.solomon.alert.domain.AlertState;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.template.AlertFromTemplatePersistent;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.protobuf.EAlertState;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TNotificationStatus;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.alert.template.domain.AlertTemplateType;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class ActivityFilters {

    private final NotificationConverter notificationConverter;

    public ActivityFilters(NotificationConverter notificationConverter) {
        this.notificationConverter = notificationConverter;
    }

    public Predicate<AlertActivity> filterByName(String name) {
        return activity -> StringUtils.containsIgnoreCase(activity.getAlert().getName(), name);
    }

    public Predicate<AlertActivity> filterByState(Set<AlertState> states) {
        return activity -> states.contains(activity.getAlert().getState());
    }

    public Predicate<AlertActivity> filterByState(List<EAlertState> states) {
        return filterByState(AlertConverter.protosToAlertState(states));
    }

    public Predicate<AlertActivity> filterByType(Set<AlertType> types) {
        return activity -> {
            if (types.contains(activity.getAlert().getAlertType())) {
                return true;
            }
            if (activity instanceof TemplateAlertActivity taa) {
                AlertTemplateType alertTemplateType = taa.getTemplate().getAlertTemplateType();
                if (alertTemplateType == AlertTemplateType.EXPRESSION) {
                    return types.contains(AlertType.EXPRESSION);
                } else if (alertTemplateType == AlertTemplateType.THRESHOLD) {
                    return types.contains(AlertType.THRESHOLD);
                }
            }
            return false;
        };
    }

    public Predicate<AlertActivity> filterByType(List<EAlertType> types) {
        return filterByType(AlertConverter.protosToTypes(types));
    }

    public Predicate<AlertActivity> filterByNotificationIds(List<String> notificationIds) {
        Set<String> ids = ImmutableSet.copyOf(notificationIds);
        return activity -> activity.getAlert().getNotificationChannels().keySet().stream().anyMatch(ids::contains);
    }

    public Predicate<AlertActivity> filterByCreatedServiceProvider(String serviceProvider) {
        return activity -> {
            if (activity.getAlert() instanceof AlertFromTemplatePersistent templateAlert) {
                return serviceProvider.equals(templateAlert.getServiceProvider());
            }
            // skip other alerts
            return false;
        };
    }

    public Predicate<AlertActivity> filterByTemplateServiceProvider(String templateServiceProvider) {
        return activity -> {
            if("-".equals(templateServiceProvider)) {
                return !(activity instanceof TemplateAlertActivity);
            } else {
                if (activity instanceof TemplateAlertActivity taa) {
                    return taa.getTemplate().getServiceProviderId().equals(templateServiceProvider);
                }
            }
            // skip other alerts
            return false;
        };
    }

    public Predicate<AlertActivity> filterByEvaluationStatus(List<TEvaluationStatus.ECode> list) {
        EnumSet<EvaluationStatus.Code> codes = EnumSet.noneOf(EvaluationStatus.Code.class);
        for (TEvaluationStatus.ECode item : list) {
            codes.add(AlertConverter.protoToStatusCode(item));
        }
        return filterByEvaluationStatus(codes);
    }

    public Predicate<AlertActivity> filterByNotificationStatus(
        Set<String> notificationIds,
        List<TNotificationStatus.ECode> list)
    {
        EnumSet<NotificationStatus.Code> codes = EnumSet.noneOf(NotificationStatus.Code.class);
        for (TNotificationStatus.ECode item : list) {
            codes.add(notificationConverter.protoToStatusCode(item));
        }

        return filterByNotificationStatus(notificationIds, codes);
    }

    public Predicate<AlertActivity> filterByEvaluationStatus(Set<EvaluationStatus.Code> codes) {
        return activity -> {
            activity = ActivityFactory.unwrap(activity);
            if (activity instanceof SimpleAlertActivity one) {
                EvaluationState evaluation = one.getLatestEvaluation();
                return evaluation != null && codes.contains(evaluation.getStatus().getCode());
            } else if (activity instanceof MultiAlertActivity many) {
                for (SubAlertActivity child : many.getSubActivities()) {
                    EvaluationState evaluation = child.getLatestEvaluation();
                    if (evaluation == null) {
                        continue;
                    }

                    if (codes.contains(evaluation.getStatus().getCode())) {
                        return true;
                    }
                }
            }

            return false;
        };
    }

    public Predicate<AlertActivity> filterByNotificationStatus(
        Set<String> notificationIds,
        Set<NotificationStatus.Code> codes)
    {
        return activity -> {
            activity = ActivityFactory.unwrap(activity);
            if (activity instanceof SimpleAlertActivity one) {
                return one.getNotificationStates(notificationIds)
                        .map(notificationState -> notificationState.getLatestStatus().getCode())
                        .anyMatch(codes::contains);
            } else if (activity instanceof MultiAlertActivity many) {
                return many.getSubActivities()
                        .stream()
                        .flatMap(subActivity -> subActivity.getNotificationStates(notificationIds))
                        .map(notificationState -> notificationState.getLatestStatus().getCode())
                        .anyMatch(codes::contains);
            }
            return false;
        };
    }

    private Predicate<AlertActivity> filterByMuteId(Set<String> muteIds) {
        return activity -> {
            activity = ActivityFactory.unwrap(activity);
            if (activity instanceof SimpleAlertActivity one) {
                var status = one.getMuteStatus();
                return status != null && status.containsAnyOf(muteIds);
            } else if (activity instanceof MultiAlertActivity many) {
                return many.getSubActivities().stream()
                        .anyMatch(subActivity -> {
                            var status = subActivity.getMuteStatus();
                            return status != null && status.containsAnyOf(muteIds);
                        });
            }
            return false;
        };
    }

    private Predicate<AlertActivity> filterByMute(MuteMatcher dt) {
        return activity -> {
            activity = ActivityFactory.unwrap(activity);
            if (activity instanceof SimpleAlertActivity one) {
                return one.matchMutes(dt, Instant.EPOCH).isMuted();
            } else if (activity instanceof MultiAlertActivity many) {
                return many.getSubActivities().stream()
                        .anyMatch(subActivity -> subActivity.matchMutes(dt, Instant.EPOCH).isMuted());
            }
            return false;
        };
    }

    public Predicate<SubAlertActivity> filterSubAlertBySelector(Selectors selectors) {
        return activity -> selectors.match(activity.getAlert().getGroupKey());
    }

    public Predicate<SubAlertActivity> filterSubAlertByEvaluationStatus(List<TEvaluationStatus.ECode> list) {
        EnumSet<EvaluationStatus.Code> codes = EnumSet.noneOf(EvaluationStatus.Code.class);
        for (TEvaluationStatus.ECode item : list) {
            codes.add(AlertConverter.protoToStatusCode(item));
        }

        return filterSubAlertByEvaluationStatus(codes);
    }

    public Predicate<SubAlertActivity> filterSubAlertByNotificationStatus(
        Set<String> notificationIds,
        List<TNotificationStatus.ECode> list)
    {
        EnumSet<NotificationStatus.Code> codes = EnumSet.noneOf(NotificationStatus.Code.class);
        for (TNotificationStatus.ECode item : list) {
            codes.add(notificationConverter.protoToStatusCode(item));
        }

        return filterSubAlertByNotificationStatus(notificationIds, codes);
    }

    public Predicate<SubAlertActivity> filterSubAlertByEvaluationStatus(Set<EvaluationStatus.Code> codes) {
        return activity -> {
            EvaluationState evaluation = activity.getLatestEvaluation();
            return evaluation != null && codes.contains(evaluation.getStatus().getCode());
        };
    }

    public Predicate<SubAlertActivity> filterSubAlertByNotificationStatus(
        Set<String> notificationIds,
        Set<NotificationStatus.Code> codes)
    {
        return activity -> activity.getNotificationStates(notificationIds)
                .anyMatch(state -> codes.contains(state.getLatestStatus().getCode()));
    }

    public Predicate<AlertActivity> filterByLabels(String labelsSelector) {
        var selectors = Selectors.parse(labelsSelector);
        return activity -> selectors.match(activity.getAlert().getLabels());
    }

    @VisibleForTesting
    public Predicate<AlertActivity> filterBy(TListAlertRequest request) {
        return filterBy(request, null);
    }

    public Predicate<AlertActivity> filterBy(TListAlertRequest request, @Nullable MuteMatcher matcher) {
        Predicate<AlertActivity> predicate = filterByFolderId(request.getFolderId())
                .and(removeFailed());
        if (!request.getFilterByName().isEmpty()) {
            predicate = predicate.and(filterByName(request.getFilterByName()));
        }

        if (request.getFilterByTypeCount() > 0) {
            predicate = predicate.and(filterByType(request.getFilterByTypeList()));
        }

        if (request.getFilterByStateCount() > 0) {
            predicate = predicate.and(filterByState(request.getFilterByStateList()));
        }

        if (request.getFilterByNotificationIdCount() > 0) {
            predicate = predicate.and(filterByNotificationIds(request.getFilterByNotificationIdList()));
        }

        if (request.getFilterByEvaluationStatusCount() > 0) {
            predicate = predicate.and(filterByEvaluationStatus(request.getFilterByEvaluationStatusList()));
        }

        if (request.getFilterByNotificationStatusCount() > 0) {
            Set<String> notificationIds = ImmutableSet.copyOf(request.getFilterByNotificationIdList());
            predicate = predicate.and(filterByNotificationStatus(notificationIds, request.getFilterByNotificationStatusList()));
        }

        switch (request.getFilterByMuteCase()) {
            case FILTER_BY_MUTE_REFERENCE -> {
                Set<String> muteIds = ImmutableSet.copyOf(request.getFilterByMuteReference().getIdsList());
                if (!muteIds.isEmpty()) {
                    predicate = predicate.and(filterByMuteId(muteIds));
                }
            }
            case FILTER_BY_INLINED_MUTE -> {
                predicate = predicate.and(filterByMute(matcher));
            }
        }

        if (!StringUtils.isEmpty(request.getFilterCreatedByServiceProvider())) {
            predicate = predicate.and(filterByCreatedServiceProvider(request.getFilterCreatedByServiceProvider()));
        }

        if (!"".equals(request.getTemplateProvider())) {
            predicate = predicate.and(filterByTemplateServiceProvider(request.getTemplateProvider()));
        }

        if (!StringUtils.isEmpty(request.getLabelsSelector())) {
            predicate = predicate.and(filterByLabels(request.getLabelsSelector()));
        }

        return predicate;
    }

    private Predicate<AlertActivity> removeFailed() {
        return activity -> !(activity instanceof FailedAlertActivity);
    }

    private Predicate<AlertActivity> filterByFolderId(String folderId) {
        if (folderId.isEmpty()) {
            return ignore -> true;
        } else {
            return entry -> folderId.equals(entry.getAlert().getFolderId());
        }
    }

    public Predicate<SubAlertActivity> filterBy(TListSubAlertRequest request) {
        Predicate<SubAlertActivity> predicate = (ignore) -> true;
        if (request.getFilterByLabelsCount() > 0) {
            Selectors selectors = LabelSelectorConverter.protoToSelectors(request.getFilterByLabelsList());
            predicate = predicate.and(filterSubAlertBySelector(selectors));
        }

        if (request.getFilterByEvaluationStatusCount() > 0) {
            List<TEvaluationStatus.ECode> codes = request.getFilterByEvaluationStatusList();
            predicate = predicate.and(filterSubAlertByEvaluationStatus(codes));
        }

        if (request.getFilterByNotificationStatusCount() > 0) {
            Set<String> notificationIds = ImmutableSet.copyOf(request.getFilterByNotificationIdList());
            List<TNotificationStatus.ECode> codes = request.getFilterByNotificationStatusList();
            predicate = predicate.and(filterSubAlertByNotificationStatus(notificationIds, codes));
        }

        return predicate;
    }
}
