package ru.yandex.solomon.alert.cluster.broker.alert.activity.search;

import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.api.converters.MuteConverter;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.ActivityFactory;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.AlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.EvaluationSummaryStatistics;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.MultiAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.NotificationSummaryStatistics;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SimpleAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SubAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.SummaryStatisticsConverter;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.TemplateAlertActivity;
import ru.yandex.solomon.alert.cluster.broker.mute.MuteMatcher;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.mute.domain.AffectingMute;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.mute.domain.MuteStatus;
import ru.yandex.solomon.alert.protobuf.AlertMuteStatus;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlertList;
import ru.yandex.solomon.alert.protobuf.TEvaluationStats;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TListAlert;
import ru.yandex.solomon.alert.protobuf.TListAlertList;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListAlertResponse;
import ru.yandex.solomon.alert.protobuf.TListSubAlert;
import ru.yandex.solomon.alert.protobuf.TListSubAlertRequest;
import ru.yandex.solomon.alert.protobuf.TListSubAlertResponse;
import ru.yandex.solomon.alert.protobuf.TNotificationStats;
import ru.yandex.solomon.alert.rule.AlertProcessingState;
import ru.yandex.solomon.alert.unroll.MultiAlertUnrollFactory;
import ru.yandex.solomon.labels.protobuf.LabelConverter;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class ActivitySearch {

    private final ActivityFilters activityFilters;

    public ActivitySearch(ActivityFilters activityFilters) {
        this.activityFilters = activityFilters;
    }

    public TListAlertResponse listAlerts(Collection<AlertActivity> source, TListAlertRequest request) {
        final int offset = getOffset(request.getPageToken());
        final int pageSize = getPageSize(request.getPageSize());

        Set<String> notificationIds = ImmutableSet.copyOf(request.getFilterByNotificationIdList());
        Set<String> muteIds = switch (request.getFilterByMuteCase()) {
            case FILTER_BY_MUTE_REFERENCE -> ImmutableSet.copyOf(request.getFilterByMuteReference().getIdsList());
            case FILTER_BY_INLINED_MUTE, FILTERBYMUTE_NOT_SET -> ImmutableSet.of();
        };
        MuteMatcher matcher = switch (request.getFilterByMuteCase()) {
            case FILTER_BY_INLINED_MUTE -> new StubMatcher(MuteConverter.INSTANCE.protoToMute(request.getFilterByInlinedMute()));
            case FILTER_BY_MUTE_REFERENCE, FILTERBYMUTE_NOT_SET -> null;
        };

        MutableInt cursor = new MutableInt(offset);
        var activities = source.stream()
                .sorted(ActivitySorts.orderBy(request))
                .skip(offset)
                .peek(ignore -> cursor.increment())
                .filter(activityFilters.filterBy(request, matcher))
                .limit(pageSize + 1)
                .collect(Collectors.toList());

        final String nextToken;
        if (activities.size() > pageSize) {
            activities = activities.subList(0, pageSize);
            cursor.decrement();
            nextToken = String.valueOf(cursor.intValue());
        } else {
            nextToken = "";
        }

        var builder = TListAlertResponse.newBuilder();
        if (request.getFullResultModel()) {
            builder.setAlertList(TAlertList.newBuilder()
                    .addAllAlerts(activities.stream()
                            .map(AlertActivity::getAlert)
                            .map(AlertConverter::alertToProto)
                            .collect(Collectors.toList()))
                    .build());
        } else {
            var listAlerts = activities.stream()
                    .map(activity -> activityToListItem(activity, notificationIds, muteIds, matcher))
                    .collect(Collectors.toList());
            builder.addAllAlerts(listAlerts)
                    .setListAlertList(TListAlertList.newBuilder()
                            .addAllAlerts(listAlerts)
                            .build());
        }

        return builder
                .setRequestStatus(ERequestStatusCode.OK)
                .setNextPageToken(nextToken)
                .build();
    }

    public TListSubAlertResponse listSubAlerts(AlertActivity source, TListSubAlertRequest request) {
        source = ActivityFactory.unwrap(source);
        if (source instanceof MultiAlertActivity many) {
            return listSubAlerts(many.getSubActivities(), request);
        }
        return TListSubAlertResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .build();
    }

    public TListSubAlertResponse listSubAlerts(Collection<SubAlertActivity> source, TListSubAlertRequest request) {
        final int offset = getOffset(request.getPageToken());
        final int pageSize = getPageSize(request.getPageSize());

        Set<String> notificationIds = ImmutableSet.copyOf(request.getFilterByNotificationIdList());
        Set<String> annotationKeys = ImmutableSet.copyOf(request.getAnnotationKeysList());

        MutableInt cursor = new MutableInt(offset);
        List<TListSubAlert> result = source.stream()
                .filter(activityFilters.filterBy(request))
                .sorted(ActivitySorts.orderBy(request))
                .skip(offset)
                .limit(pageSize)
                .peek(ignore -> cursor.increment())
                .map(activity -> activityToListSubAlert(activity, notificationIds, annotationKeys))
                .collect(Collectors.toList());

        String nextToken = result.size() < pageSize
                ? ""
                : String.valueOf(cursor.intValue());

        return TListSubAlertResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .addAllAlerts(result)
                .setNextPageToken(nextToken)
                .build();
    }

    private int getPageSize(int pageSize) {
        return pageSize <= 0 ? 10 : pageSize;
    }

    private int getOffset(String token) {
        return token.isEmpty() ? 0 : Integer.parseInt(token);
    }

    private TListAlert activityToListItem(
            AlertActivity originalActivity,
            Set<String> notificationIds,
            Set<String> muteIds,
            @Nullable MuteMatcher matcher)
    {
        Alert alert = originalActivity.getAlert();
        AlertProcessingState state;
        var unwrap = ActivityFactory.unwrap(originalActivity);
        if (unwrap instanceof SimpleAlertActivity saa) {
            state = saa.getProcessingState();
        } else {
            state = null;
        }
        var labels = new HashMap<>(alert.getLabels());
        if (unwrap instanceof TemplateAlertActivity taa) {
            labels.put("templateId", taa.getTemplate().getId());
            labels.put("templateVersionTag", taa.getTemplate().getTemplateVersionTag());
            labels.put("serviceProviderId", taa.getTemplate().getServiceProviderId());
        }
        var builder = TListAlert.newBuilder()
                .setId(alert.getId())
                .setProjectId(alert.getProjectId())
                .setFolderId(alert.getFolderId())
                .setName(alert.getName())
                .setAlertState(AlertConverter.alertStateToProto(alert.getState()))
                .setAlertType(AlertConverter.typeToProto(alert.getAlertType()))
                .setMultiAlert(MultiAlertUnrollFactory.isSupportUnrolling(unwrap.getAlert()))
                .setEvaluationStats(getEvaluationStats(unwrap))
                .setMutedStats(getMutedStats(unwrap, muteIds, matcher))
                .setNotificationStats(getNotificationStats(unwrap, notificationIds))
                .addAllNotificationChannelIds(alert.getNotificationChannels().keySet().stream().sorted().collect(Collectors.toList()))
                .putAllLabels(labels)
                .putAllConfiguredNotificationChannels(alert.getNotificationChannels().entrySet().stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, e -> AlertConverter.channelConfigToProto(e.getValue()))))
                .setEvaluationStatusCode(TEvaluationStatus.ECode.forNumber(state != null ? state.evaluationState().getStatus().getCode().getNumber() : 0))
                .setMuteStatusCode(AlertMuteStatus.Code.forNumber(state != null ? state.alertMuteStatus().statusCode().getNumber() : 0))
                .setStatusSinceTimestamp(state != null ? state.evaluationState().getLatestEval().toEpochMilli() : 0L);
        if (originalActivity instanceof TemplateAlertActivity taa) {
            builder.setTemplateData(TListAlert.FromTemplateData.newBuilder()
                    .setTemplateType(AlertConverter.typeToProto(taa.getDelegateAlert().getAlertType()))
                    .setTemplateServiceProviderId(taa.getTemplate().getServiceProviderId())
                    .build());
        }
        return builder.build();
    }

    private TListSubAlert activityToListSubAlert(
        SubAlertActivity activity,
        Set<String> notificationIds,
        Set<String> annotationKeys)
    {
        SubAlert alert = activity.getAlert();
        int evaluationStatusCode = -1;
        AlertMuteStatus.Code muteStatusCode = AlertMuteStatus.Code.UNKNOWN;
        long latestEvalMillis = 0;
        Map<String, String> annotations = Collections.emptyMap();
        AlertProcessingState state = activity.getProcessingState();
        if (state != null) {
            evaluationStatusCode = state.evaluationState().getStatus().getCode().getNumber();
            latestEvalMillis = state.evaluationState().getLatestEval().toEpochMilli();
            if (!annotationKeys.isEmpty()) {
                annotations = state.evaluationState().getStatus().getAnnotations().entrySet()
                    .stream()
                    .filter(entry -> annotationKeys.contains(entry.getKey()))
                    .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
            }
            muteStatusCode = AlertMuteStatus.Code.forNumber(state.alertMuteStatus().statusCode().getNumber());
        }

        return TListSubAlert.newBuilder()
                .setId(alert.getId())
                .setProjectId(alert.getProjectId())
                .setFolderId(alert.getParent().getFolderId())
                .setParentId(alert.getParent().getId())
                .addAllLabels(LabelConverter.labelsToProtoList(alert.getGroupKey()))
                .setEvaluationStatusCodeValue(evaluationStatusCode)
                .setMuteStatusCode(muteStatusCode)
                .setNotificationStats(getNotificationStats(activity, notificationIds))
                .setLatestEvalMillis(latestEvalMillis)
                .putAllAnnotations(annotations)
                .build();
    }

    public TEvaluationStats getEvaluationStats(AlertActivity activity) {
        EvaluationSummaryStatistics summary = new EvaluationSummaryStatistics();
        activity.appendEvaluationStatistics(summary);
        return SummaryStatisticsConverter.toProto(summary);
    }

    public TEvaluationStats getMutedStats(AlertActivity activity, Set<String> muteIds, @Nullable MuteMatcher matcher) {
        EvaluationSummaryStatistics summary = new EvaluationSummaryStatistics();
        activity.appendMutedStatistics(summary, muteIds, matcher);
        return SummaryStatisticsConverter.toProto(summary);
    }

    public TNotificationStats getNotificationStats(AlertActivity activity, Set<String> notificationIds) {
        NotificationSummaryStatistics summary = new NotificationSummaryStatistics();
        activity.appendNotificationStatistics(summary, notificationIds);
        return SummaryStatisticsConverter.toProto(summary);
    }

    private TNotificationStats getNotificationStats(SubAlertActivity activity, Set<String> notificationIds) {
        NotificationSummaryStatistics summary = new NotificationSummaryStatistics();
        activity.getNotificationStates(notificationIds).forEach(summary::add);
        return SummaryStatisticsConverter.toProto(summary);
    }

    private static class StubMatcher implements MuteMatcher {
        private final Mute dt;

        public StubMatcher(Mute dt) {
            this.dt = dt;
        }

        public List<AffectingMute> match(String alertId, Labels subAlertLabels, Instant evaluatedAt) {
            if (dt.matches(alertId, subAlertLabels)) {
                return List.of(new AffectingMute("whatever", MuteStatus.ACTIVE));
            } else {
                return List.of();
            }
        }
    }
}
