package ru.yandex.solomon.alert.cluster.broker.evaluation;

import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;

import io.grpc.Status;
import io.grpc.Status.Code;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.cluster.project.ProjectAssignment;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.protobuf.THeartbeatResponse;

/**
 * @author Vladimir Gordiychuk
 */
public class EvaluationAssignmentServiceImpl implements EvaluationAssignmentService, TaskEvaluationTracker<Task>, AutoCloseable {
    private final EvaluationAssignmentServiceMetrics metrics;
    private final long noMessageTimeoutMillis;

    private final OneStreamPerHost oneStreamPerHost;

    private volatile boolean closed;

    public EvaluationAssignmentServiceImpl(
        ClusterDiscovery<GrpcTransport> discovery,
        ScheduledExecutorService timer,
        Executor executor,
        MetricRegistry registry,
        long noMessageTimeoutMillis)
    {
        this.metrics = new EvaluationAssignmentServiceMetrics(registry);
        this.noMessageTimeoutMillis = noMessageTimeoutMillis;
        this.oneStreamPerHost = new OneStreamPerHost(discovery, timer, executor, this, registry);
    }

    @Override
    public void update(THeartbeatResponse heartbeat) {
        this.oneStreamPerHost.update(heartbeat);
    }

    @Override
    public void assign(ProjectAssignment assignment, Alert alert, EvaluationObserver observer) {
        assign(new Task(assignment, alert, observer));
    }

    @Override
    public void close() {
        closed = true;
    }

    @Override
    public void onTaskComplete(Task value) {
        actualizeMetrics();
        if (!value.isCanceled()) {
            metrics.restartCount.inc();
            metrics.incStatus(Code.OK);
            assign(value);
        }
    }

    @Override
    public void onTaskCompleteWarmup(Task value) {
        actualizeMetrics();
    }

    @Override
    public void onTaskError(Task task, Status status) {
        actualizeMetrics();
        if (status.getCode() != Status.Code.ABORTED) {
            metrics.restartCount.inc();
            metrics.incStatus(status.getCode());
            assign(task);
        }
    }

    @Override
    public long noMessageTimeoutMillis() {
        return noMessageTimeoutMillis;
    }

    private void assign(Task task) {
        if (task.isCanceled() || closed) {
            return;
        }

        task.observer.onSubscribe(task);
        metrics.warmupSize.add(1);
        oneStreamPerHost.assign(task);
    }

    private void actualizeMetrics() {
        metrics.warmupSize.set(oneStreamPerHost.getWarmupSize());
    }
}
