package ru.yandex.solomon.alert.cluster.broker.evaluation;

import java.util.concurrent.Flow;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

import io.grpc.Status;
import io.netty.util.Timeout;
import io.netty.util.TimerTask;

import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.alert.util.Async;

/**
 * @author Vladimir Gordiychuk
 */
class EvaluationStreamObserver implements Flow.Subscriber<EvaluationState>, TimerTask {
    private static final AtomicReferenceFieldUpdater<EvaluationStreamObserver, State> fieldState =
            AtomicReferenceFieldUpdater.newUpdater(EvaluationStreamObserver.class, State.class, "state");

    private final Task task;
    private final TaskEvaluationTracker<EvaluationStreamObserver> tracker;
    private volatile boolean warmup;
    private volatile Timeout timeout;
    private volatile State state = State.RUNNING;
    private volatile Flow.Subscription subscription;

    public EvaluationStreamObserver(Task task, TaskEvaluationTracker<EvaluationStreamObserver> tracker) {
        this.task = task;
        this.tracker = tracker;
        this.warmup = true;
        // TODO: remove it after migrate on one grpc stream for multiple evaluation (@gordiychuk)
        this.timeout = Async.runAfter(this, tracker.noMessageTimeoutMillis(), TimeUnit.MILLISECONDS);
    }

    public void cancel() {
        if (!fieldState.compareAndSet(this, State.RUNNING, State.CANCELED)) {
            return;
        }

        timeout.cancel();
        task.onError(Status.CANCELLED.asException());
        Flow.Subscription s = subscription;
        if (s != null) {
            s.cancel();
        }
    }

    public Task getTask() {
        return task;
    }

    public boolean isWarmup() {
        return warmup;
    }

    @Override
    public void onSubscribe(Flow.Subscription s) {
        subscription = s;
        if (state != State.RUNNING) {
            s.cancel();
        } else {
            s.request(Long.MAX_VALUE);
        }
    }

    @Override
    public void onNext(EvaluationState value) {
        timeout.cancel();
        if (state != State.RUNNING) {
            throw abortCausedBy(state.name());
        }

        if (task.isCanceled()) {
            onComplete();
            throw abortCausedBy("task cancellation");
        }

        try {
            if (warmup) {
                warmup = false;
                tracker.onTaskCompleteWarmup(this);
            }

            task.onNext(value);
            // when into stream not received message, it's indicate about network problem
            // so we should force cancel evaluation and try on another node
            timeout = Async.runAfter(this, tracker.noMessageTimeoutMillis(), TimeUnit.MILLISECONDS);
        } catch (Throwable e) {
            onError(e);
        }
    }

    @Override
    public void onError(Throwable e) {
        if (fieldState.compareAndSet(this, State.RUNNING, State.ERROR)) {
            reportError(e);
        }
    }

    @Override
    public void onComplete() {
        if (fieldState.compareAndSet(this, State.RUNNING, State.COMPLETE)) {
            tracker.onTaskComplete(this);
        }
    }

    @Override
    public void run(Timeout timeout) {
        if (timeout.isCancelled()) {
            return;
        }

        if (!fieldState.compareAndSet(this, State.RUNNING, State.DEADLINE)) {
            return;
        }

        reportError(Status.DEADLINE_EXCEEDED.withDescription("Cancel stuck evaluation: " + timeout).asException());
    }

    private void reportError(Throwable e) {
        task.onError(e);
        tracker.onTaskError(this, Status.fromThrowable(e));
        Flow.Subscription sub = subscription;
        if (sub != null) {
            sub.cancel();
        }
    }

    private RuntimeException abortCausedBy(String cause) {
        throw Status.ABORTED
                .withDescription("Evaluation for alert " + task.alert.getKey() + " not actual anymore: " + cause)
                .asRuntimeException();
    }

    private enum State {
        RUNNING, CANCELED, ERROR, DEADLINE, COMPLETE
    }
}
