package ru.yandex.solomon.alert.cluster.broker.evaluation;

import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicInteger;

import io.grpc.Status;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.cluster.project.AssignmentConverter;
import ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter;
import ru.yandex.solomon.alert.cluster.server.grpc.evaluation.ClientAssignReq;
import ru.yandex.solomon.alert.cluster.server.grpc.evaluation.ClientEvaluationCluster;
import ru.yandex.solomon.alert.cluster.server.grpc.evaluation.MapSubscriber;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.Evaluation;
import ru.yandex.solomon.alert.protobuf.THeartbeatResponse;
import ru.yandex.solomon.alert.rule.EvaluationState;

/**
 * @author Vladimir Gordiychuk
 */
public class OneStreamPerHost implements TaskEvaluationTracker<EvaluationStreamObserver>, AutoCloseable {
    private final ClientEvaluationCluster cluster;
    private final TaskEvaluationTracker<Task> tracker;
    private final AtomicInteger warmupSize = new AtomicInteger();

    public OneStreamPerHost(
            ClusterDiscovery<GrpcTransport> discovery,
            ScheduledExecutorService timer,
            Executor executor,
            TaskEvaluationTracker<Task> tracker,
            MetricRegistry registry)
    {
        this.cluster = new ClientEvaluationCluster(discovery, 5, timer, executor, registry);
        this.tracker = tracker;
    }

    public void update(THeartbeatResponse heartbeat) {
        for (var node : heartbeat.getMembersList()) {
            cluster.setActive(node.getAddress(), node.getActive());
        }
    }

    public void assign(Task task) {
        var seqNo = AssignmentConverter.toProto(task.assignment.getSeqNo());
        var trackSubscriber = new EvaluationStreamObserver(task, this);
        var subscriber = new MapSubscriber<Evaluation, EvaluationState>(trackSubscriber, GrpcEvaluationStreamConverter::toState);
        var req = new ClientAssignReq(seqNo, task.alert, task.observer.getLatestEvaluation(), subscriber);
        warmupSize.incrementAndGet();
        cluster.assign(req);
    }

    @Override
    public void close() {
        cluster.close();
    }

    @Override
    public void onTaskComplete(EvaluationStreamObserver value) {
        if (value.isWarmup()) {
            warmupSize.decrementAndGet();
        }
        tracker.onTaskComplete(value.getTask());
    }

    @Override
    public void onTaskCompleteWarmup(EvaluationStreamObserver value) {
        warmupSize.decrementAndGet();
        tracker.onTaskCompleteWarmup(value.getTask());
    }

    @Override
    public void onTaskError(EvaluationStreamObserver value, Status status) {
        if (value.isWarmup()) {
            warmupSize.decrementAndGet();
        }
        tracker.onTaskError(value.getTask(), status);
    }

    public int getWarmupSize() {
        return warmupSize.get();
    }

    @Override
    public long noMessageTimeoutMillis() {
        return tracker.noMessageTimeoutMillis();
    }
}
