package ru.yandex.solomon.alert.cluster.broker.mute;

import java.util.EnumMap;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.mute.domain.MuteStatus;
import ru.yandex.solomon.util.collection.enums.EnumMapToInt;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class MuteServiceMetrics implements MetricSupplier {
    private final Runnable actualize;
    private final MetricRegistry registry;

    private final EnumMap<MuteStatus, GaugeInt64> countByStatus = new EnumMap<>(MuteStatus.class);
    public final Rate cpuTimeNanos;

    public MuteServiceMetrics() {
        this(() -> { /* no-op */ });
    }

    MuteServiceMetrics(Runnable actualize) {
        this.actualize = actualize;
        registry = new MetricRegistry();

        cpuTimeNanos = new Rate();
        registry.lazyRate("alerting.mute.cpu_use_us", () -> TimeUnit.NANOSECONDS.toMicros(cpuTimeNanos.get()));
        // EnumMetrics does not allow for enum values blacklist
        for (var status : MuteStatus.class.getEnumConstants()) {
            if (status == MuteStatus.UNKNOWN || status == MuteStatus.ARCHIVED) {
                continue;
            }
            countByStatus.put(status, registry.gaugeInt64("alerting.mute.count", Labels.of("status", status.name())));
        }
    }

    public void combine(MuteServiceMetrics muteServiceMetrics) {
        cpuTimeNanos.combine(muteServiceMetrics.cpuTimeNanos);
        for (var e : countByStatus.entrySet()) {
            e.getValue().combine(muteServiceMetrics.countByStatus.get(e.getKey()));
        }
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        actualize.run();
        registry.append(tsMillis, commonLabels, consumer);
    }

    public void setCountByStatus(EnumMapToInt<MuteStatus> countMutesByStatus) {
        for (var e : countByStatus.entrySet()) {
            e.getValue().set(countMutesByStatus.get(e.getKey()));
        }
    }
}
