package ru.yandex.solomon.alert.cluster.broker.mute;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.GuardedBy;

import com.google.common.annotations.VisibleForTesting;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.api.converters.MuteConverter;
import ru.yandex.solomon.alert.cluster.broker.mute.search.MuteSearch;
import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.mute.domain.AffectingMute;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.mute.domain.MuteStatus;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.CreateMuteResponse;
import ru.yandex.solomon.alert.protobuf.DeleteMuteRequest;
import ru.yandex.solomon.alert.protobuf.DeleteMuteResponse;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesResponse;
import ru.yandex.solomon.alert.protobuf.MutesStats;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteResponse;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteStatsResponse;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteResponse;
import ru.yandex.solomon.alert.util.RateLimitedRunner;
import ru.yandex.solomon.util.collection.enums.EnumMapToInt;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static ru.yandex.solomon.alert.cluster.broker.mute.ProtoReply.replyCreateMute;
import static ru.yandex.solomon.alert.cluster.broker.mute.ProtoReply.replyDeleteMute;
import static ru.yandex.solomon.alert.cluster.broker.mute.ProtoReply.replyReadMute;
import static ru.yandex.solomon.alert.cluster.broker.mute.ProtoReply.replyUpdateMute;
import static ru.yandex.solomon.idempotency.IdempotentOperation.NO_OPERATION;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class ProjectMuteService implements AutoCloseable, MuteMatcher {
    private static final Logger logger = LoggerFactory.getLogger(ProjectMuteService.class);

    private final String projectId;
    private final Clock clock;
    private final EntitiesDao<Mute> dao;

    // Each mute belongs exactly to one of the following maps
    private final ReadWriteLock rwLock = new ReentrantReadWriteLock(true);
    @GuardedBy("rwLock")
    private final Map<String, Mute> mutesById = new HashMap<>();
    @GuardedBy("rwLock")
    private final Map<String, Mute> obsoleteMutesById = new HashMap<>();

    private final RateLimitedRunner cleanupRunner = new RateLimitedRunner(Duration.ofMinutes(10), this::doCleanupMutes);

    private final AtomicReference<State> state = new AtomicReference<>(State.IDLE);
    private final MuteSearch search;
    private final MuteConverter converter;
    private final MuteServiceMetrics metrics;

    private final RateLimitedRunner statsRefreshRunner = new RateLimitedRunner(Duration.ofSeconds(15), this::refreshStats);
    @Nullable
    private volatile Map<String, EnumMapToInt<MuteStatus>> muteStatsByFolder = null;
    private static final String TOTAL_FOLDER = "";
    private static final EnumMapToInt<MuteStatus> EMPTY_MUTE_STATS = new EnumMapToInt<>(MuteStatus.class, 0);

    public ProjectMuteService(
            String projectId,
            Clock clock,
            EntitiesDao<Mute> mutesDao,
            MuteSearch search,
            MuteConverter converter)
    {
        this.projectId = projectId;
        this.clock = clock;
        this.dao = mutesDao;
        this.search = search;
        this.converter = converter;
        this.metrics = new MuteServiceMetrics(this::actualizeMetrics);
    }

    public CompletableFuture<CreateMuteResponse> createMute(CreateMuteRequest request) {
        return ensureRunningExecute(ProjectMuteService::createMuteImpl, request);
    }

    public CompletableFuture<ReadMuteResponse> readMute(ReadMuteRequest request) {
        return ensureRunningExecute(ProjectMuteService::readMuteImpl, request);
    }

    public CompletableFuture<UpdateMuteResponse> updateMute(UpdateMuteRequest request) {
        return ensureRunningExecute(ProjectMuteService::updateMuteImpl, request);
    }

    public CompletableFuture<DeleteMuteResponse> deleteMute(DeleteMuteRequest request) {
        return ensureRunningExecute(ProjectMuteService::deleteMuteImpl, request);
    }

    public CompletableFuture<ListMutesResponse> listMutes(ListMutesRequest request) {
        return ensureRunningExecute(ProjectMuteService::listMutesImpl, request);
    }

    public CompletableFuture<ReadMuteStatsResponse> readMuteStats(ReadMuteStatsRequest request) {
        return ensureRunningExecute(ProjectMuteService::readMuteStatsImpl, request);
    }

    private CompletableFuture<Void> purgeMute(Mute mute) {
        removeMute(mute.getId());
        return dao.deleteById(mute.getProjectId(), mute.getId(), NO_OPERATION)
                .exceptionally(throwable -> {
                    logger.error("Could not cleanup mute {}:{} from db by TTL", mute.getProjectId(), mute.getId(), throwable);
                    return null;
                });
    }

    private CompletableFuture<CreateMuteResponse> createMuteImpl(CreateMuteRequest request) {
        Instant now = clock.instant();
        var builder = converter.protoToMute(request.getMute())
                .toBuilder()
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setVersion(1);

        if (builder.getFrom() == Instant.EPOCH) {
            builder.setFrom(now);
        }

        if (!builder.getFrom().isBefore(builder.getTo())) {
            return failedFuture(failure(Status.INVALID_ARGUMENT, "expected from_millis < to_millis for mute " + builder.getId()));
        }

        var mute = builder.build();
        var old = getMute(mute.getId());

        CompletableFuture<Void> future = completedFuture(null);

        if (old != null) {
            if (old.isDeletedByTtl(now)) {
                future = purgeMute(old);
            } else {
                return failedFuture(failure(Status.ALREADY_EXISTS, "mute with id " + mute.getId() + " already exists"));
            }
        }

        return future.thenCompose(aVoid -> dao.insert(mute, NO_OPERATION)
                .handle((maybePrev, e) -> {
                    if (e != null) {
                        logger.error("Failed create mute {} request into project {}", mute, mute.getProjectId(), e);
                        throw failure(Status.INTERNAL.withCause(e), "failed to create mute");
                    }

                    if (maybePrev.isPresent()) {
                        // DB and muteById out of sync

                        Mute prev = maybePrev.get();

                        if (!prev.equals(mute)) {
                            // make mute (better late than never) and return conflict
                            upsertMute(prev, now);
                            throw failure(Status.ALREADY_EXISTS, "mute with id " + mute.getId() + " already exists");
                        }
                    }
                    // prev = null or prev = mute

                    upsertMute(mute, now);
                    return replyCreateMute(converter.muteToProto(mute, mute.getStatusAt(now)));
                }));
    }

    private Mute getMute(String muteId) {
        rwLock.readLock().lock();
        try {
            Mute dt = mutesById.get(muteId);
            if (dt != null) {
                return dt;
            }
            return obsoleteMutesById.get(muteId);
        } finally {
            rwLock.readLock().unlock();
        }
    }

    private CompletableFuture<Mute> getMuteOrThrowIfNotFound(String id, String folderId, Instant now) {
        Mute mute = getMute(id);
        if (mute == null || mute.isDeletedByTtl(now)) {
            CompletableFuture<Void> future = (mute != null) ? purgeMute(mute) : completedFuture(null);
            return future.thenCompose(aVoid ->
                    failedFuture(failure(Status.NOT_FOUND, "mute with id " + id + " does not exist")));
        }

        if (!folderId.isEmpty()) {
            if (!mute.getFolderId().equals(folderId)) {
                return failedFuture(failure(Status.NOT_FOUND, "mute with id " + id + " does not exist in folderId = " + folderId));
            }
        }

        return completedFuture(mute);
    }

    private CompletableFuture<ReadMuteResponse> readMuteImpl(ReadMuteRequest request) {
        Instant now = clock.instant();
        return getMuteOrThrowIfNotFound(request.getId(), request.getFolderId(), now)
                .thenApply(mute -> {
                    var status = mute.getStatusAt(now);
                    return replyReadMute(converter.muteToProto(mute, status));
                });
    }

    private CompletableFuture<UpdateMuteResponse> updateMuteImpl(UpdateMuteRequest request) {
        Instant now = clock.instant();
        Mute mute = converter.protoToMute(request.getMute());
        return getMuteOrThrowIfNotFound(mute.getId(), mute.getFolderId(), now)
                .thenCompose(previous -> {
                    if (mute.getVersion() != -1 && previous.getVersion() != mute.getVersion()) {
                        return failedFuture(failure(Status.FAILED_PRECONDITION, "version mismatch for mute: " + mute.getId()));
                    }

                    Mute updated = mute.toBuilder()
                            .setFolderId(previous.getFolderId())
                            .setCreatedAt(Instant.ofEpochMilli(previous.getCreatedAt()))
                            .setCreatedBy(previous.getCreatedBy())
                            .setUpdatedAt(now)
                            .setVersion(previous.getVersion() + 1)
                            .build();

                    return dao.update(updated, NO_OPERATION)
                            .handle((maybePrev, e) -> {
                                if (e != null) {
                                    logger.error("Failed to update mute {} in project {}", updated.getId(), updated.getProjectId(), e);
                                    throw failure(Status.INTERNAL.withCause(e), "failed to update mute");
                                }

                                if (maybePrev.isEmpty()) {
                                    // Was removed in db but not from memory -> remove from memory
                                    removeMute(previous.getId());
                                    throw failure(Status.NOT_FOUND, "mute with id " + mute.getId() + " does not exist");
                                }

                                Mute prev = maybePrev.get();
                                if (prev.getVersion() == updated.getVersion() - 1 || prev.equals(updated)) {
                                    // regular update or retry update
                                    upsertMute(updated, now);
                                    return replyUpdateMute(converter.muteToProto(updated, updated.getStatusAt(now)));
                                }

                                // States in db and mem are out of sync,
                                upsertMute(mute, now);
                                throw failure(Status.FAILED_PRECONDITION, "version mismatch for mute: " + mute.getId());
                            });
                });
    }

    private CompletableFuture<DeleteMuteResponse> deleteMuteImpl(DeleteMuteRequest request) {
        Instant now = clock.instant();
        return getMuteOrThrowIfNotFound(request.getId(), request.getFolderId(), now)
                .thenCompose(mute -> dao.deleteById(projectId, mute.getId(), NO_OPERATION)
                        .handle((ignore, e) -> {
                            if (e != null) {
                                logger.error("Failed to delete mute {} in project {}", mute.getId(), mute.getProjectId(), e);
                                throw failure(Status.INTERNAL.withCause(e), "failed to delete mute");
                            }

                            removeMute(mute.getId());
                            return replyDeleteMute();
                        }));
    }

    private CompletableFuture<ListMutesResponse> listMutesImpl(ListMutesRequest request) {
        rwLock.readLock().lock();
        try {
            return completedFuture(search.listMutes(mutesById.values(), obsoleteMutesById.values(), request, clock.instant()));
        } finally {
            rwLock.readLock().unlock();
        }
    }

    private void refreshStats(long nowMillis) {
        Map<String, EnumMapToInt<MuteStatus>> muteStatsByFolder = new HashMap<>();

        rwLock.readLock().lock();
        try {
            Stream.concat(mutesById.values().stream(), obsoleteMutesById.values().stream())
                    .forEach(mute -> {
                        var status = mute.getStatusAt(nowMillis);
                        String folderId = mute.getFolderId();
                        if (!folderId.isEmpty()) {
                            appendStatsForFolder(folderId, muteStatsByFolder, status);
                        }
                        appendStatsForFolder(TOTAL_FOLDER, muteStatsByFolder, status);
                    });
        } finally {
            rwLock.readLock().unlock();
        }

        this.muteStatsByFolder = muteStatsByFolder;
    }

    private void appendStatsForFolder(String folderId, Map<String, EnumMapToInt<MuteStatus>> muteStatsByFolder, MuteStatus status) {
        var muteStats = muteStatsByFolder.computeIfAbsent(folderId, ignore -> new EnumMapToInt<>(MuteStatus.class, 0));
        muteStats.addAndGet(status, 1);
    }

    private CompletableFuture<ReadMuteStatsResponse> readMuteStatsImpl(ReadMuteStatsRequest request) {
        long nowMillis = clock.millis();
        statsRefreshRunner.runIfNecessary(nowMillis);

        // In case of race at first init
        if (muteStatsByFolder == null) {
            refreshStats(nowMillis);
        }

        var muteStats = muteStatsByFolder.getOrDefault(request.getFolderId(), EMPTY_MUTE_STATS);
        return completedFuture(ReadMuteStatsResponse.newBuilder()
                .setMutesStats(MutesStats.newBuilder()
                        .setCountPending(muteStats.get(MuteStatus.PENDING))
                        .setCountActive(muteStats.get(MuteStatus.ACTIVE))
                        .setCountExpired(muteStats.get(MuteStatus.EXPIRED))
                        .setCountArchived(muteStats.get(MuteStatus.ARCHIVED)))
                .build());
    }

    @Override
    public void close() {
        var prev = state.getAndSet(State.CLOSED);
        if (prev != State.CLOSED) {
            // TODO: cleanup
        }
    }

    private StatusRuntimeException failure(Status status, String msg) {
        return status.withDescription(msg).asRuntimeException();
    }

    public <ReqT, RespT> CompletableFuture<RespT> ensureRunningExecute(BiFunction<ProjectMuteService, ReqT, CompletableFuture<RespT>> fn, ReqT request) {
        var currentState = state.get();
        if (currentState == State.RUNNING) {
            return fn.apply(this, request);
        }
        return failedFuture(failure(Status.UNAVAILABLE, "expected RUNNING shard, but was in state " + state.get()));
    }

    public boolean isReady() {
        return state.get() == State.RUNNING;
    }

    public CompletableFuture<?> run() {
        CompletableFuture<?> init = new CompletableFuture<>();

        CompletableFuture<?> result = init
                .thenCompose(ignore -> createSchema(State.IDLE))
                .thenCompose(ignore -> fetchMutes(State.SCHEME_CREATING))
                .thenAccept(ignore -> {
                    State current = state.get();
                    if (current == State.CLOSED || current == State.RUNNING) {
                        return;
                    }

                    if (!changeState(State.FETCH_MUTES, State.RUNNING)) {
                        throw new IllegalStateException("Expected state " + State.FETCH_MUTES + " but was " + state.get());
                    }
                });

        init.complete(null);
        return result;
    }

    private CompletableFuture<?> createSchema(State from) {
        if (!changeState(from, State.SCHEME_CREATING)) {
            return completedFuture(null);
        }

        return dao.createSchema(projectId)
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed to create mutes schema for project: " + projectId, e);
                        changeState(State.SCHEME_CREATING, from);
                    }
                });
    }

    private CompletableFuture<?> fetchMutes(State from) {
        if (!changeState(from, State.FETCH_MUTES)) {
            return completedFuture(null);
        }

        Instant now = clock.instant();
        return dao.find(projectId, value -> upsertMute(value, now))
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed to fetch mutes for project: " + projectId, e);
                        changeState(State.FETCH_MUTES, from);
                    }
                });
    }

    private void upsertMute(Mute mute, Instant now) {
        if (mute.isDeletedByTtl(now)) {
            removeMute(mute.getId());
        }
        rwLock.writeLock().lock();
        try {
            if (mute.getStatusAt(now) == MuteStatus.ARCHIVED) {
                obsoleteMutesById.put(mute.getId(), mute);
                mutesById.remove(mute.getId());
            } else {
                mutesById.put(mute.getId(), mute);
                obsoleteMutesById.remove(mute.getId());
            }
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    private void removeMute(String id) {
        rwLock.writeLock().lock();
        try {
            mutesById.remove(id);
            obsoleteMutesById.remove(id);
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    private boolean changeState(State from, State to) {
        if (this.state.compareAndSet(from, to)) {
            logger.info("{}: mutes change state {} -> {}", projectId, from, to);
            return true;
        }

        return false;
    }

    @Override
    public List<AffectingMute> match(String alertId, Labels subAlertLabels, Instant evaluatedAt) {
        // TODO: indices to speedup scan
        long evaluatedAtMillis = evaluatedAt.toEpochMilli();
        cleanupRunner.runIfNecessary(evaluatedAtMillis);

        long startNanos = System.nanoTime();
        rwLock.readLock().lock();
        try {
            if (mutesById.isEmpty()) {
                return List.of();
            }
            return mutesById.entrySet().stream()
                    .filter(e -> e.getValue().getStatusAt(evaluatedAt) != MuteStatus.ARCHIVED && e.getValue().matches(alertId, subAlertLabels))
                    .map(e -> new AffectingMute(e.getKey(), e.getValue().getStatusAt(evaluatedAtMillis)))
                    .collect(Collectors.toUnmodifiableList());
        } finally {
            rwLock.readLock().unlock();
            metrics.cpuTimeNanos.add(System.nanoTime() - startNanos);
        }
    }

    @VisibleForTesting
    public int getRegularMutesCount() {
        rwLock.readLock().lock();
        try {
            return mutesById.size();
        } finally {
            rwLock.readLock().unlock();
        }
    }

    private void doCleanupMutes(long nowMillis) {
        Instant now = Instant.ofEpochMilli(nowMillis);
        // Optimistic no-op check under read lock
        rwLock.readLock().lock();
        try {
            boolean haveObsolete = mutesById.entrySet().stream()
                    .anyMatch(e -> e.getValue().getStatusAt(now) == MuteStatus.ARCHIVED);
            if (!haveObsolete) {
                return;
            }
        } finally {
            rwLock.readLock().unlock();
        }

        rwLock.writeLock().lock();
        try {
            mutesById.entrySet().removeIf(e -> {
                if (e.getValue().getStatusAt(now) == MuteStatus.ARCHIVED) {
                    obsoleteMutesById.put(e.getKey(), e.getValue());
                    return true;
                }
                return false;
            });
            obsoleteMutesById.entrySet().removeIf(e -> e.getValue().isDeletedByTtl(now));
            logger.info("Cleaning up mutes in project {}. Regular count: {}, Obsolete count: {}", projectId,
                    mutesById.size(), obsoleteMutesById.size());
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    public MuteServiceMetrics getMetrics() {
        return metrics;
    }

    private void actualizeMetrics() {
        Instant now = Instant.now();
        EnumMapToInt<MuteStatus> countMutesByStatus = new EnumMapToInt<>(MuteStatus.class, 0);
        rwLock.readLock().lock();
        try {
            for (var dt : mutesById.values()) {
                countMutesByStatus.addAndGet(dt.getStatusAt(now), 1);
            }
        } finally {
            rwLock.readLock().unlock();
        }
        metrics.setCountByStatus(countMutesByStatus);
    }

    private enum State {
        IDLE,
        SCHEME_CREATING,
        FETCH_MUTES,
        RUNNING,
        CLOSED,
    }
}
