package ru.yandex.solomon.alert.cluster.broker.mute.search;

import java.time.Instant;
import java.util.Collection;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.solomon.alert.api.converters.MuteConverter;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.protobuf.ListMutesRequest;
import ru.yandex.solomon.alert.protobuf.ListMutesResponse;

import static ru.yandex.solomon.alert.cluster.broker.mute.search.MuteFilters.filterBy;
import static ru.yandex.solomon.alert.cluster.broker.mute.search.MuteSorts.orderBy;

/**
 * @author Ivan Tsybulin
 */
final public class MuteSearch {

    private final MuteConverter converter;

    public MuteSearch(MuteConverter converter) {
        this.converter = converter;
    }

    public ListMutesResponse listMutes(Collection<Mute> source, Collection<Mute> secondSource, ListMutesRequest request, Instant now) {
        final int offset = getOffset(request.getPageToken());
        final int pageSize = getPageSize(request.getPageSize());

        MutableInt cursor = new MutableInt(offset);
        var result = Stream.concat(source.stream(), secondSource.stream())
                .filter(filterBy(request, now))
                .sorted(orderBy(request))
                .skip(offset)
                .peek(ignore -> cursor.increment())
                .limit(pageSize + 1)
                .map(dt -> converter.muteToProto(dt, dt.getStatusAt(now)))
                .collect(Collectors.toList());

        final String nextToken;
        if (result.size() > pageSize) {
            result = result.subList(0, pageSize);
            cursor.decrement();
            nextToken = String.valueOf(cursor.intValue());
        } else {
            nextToken = "";
        }

        return ListMutesResponse.newBuilder()
                .addAllMutes(result)
                .setNextPageToken(nextToken)
                .build();
    }

    private static int getPageSize(int pageSize) {
        return pageSize <= 0 ? 10 : pageSize;
    }

    private static int getOffset(String token) {
        return token.isEmpty() ? 0 : Integer.parseInt(token);
    }
}
