package ru.yandex.solomon.alert.cluster.broker.notification;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.name.resolver.client.Resource;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ChannelValidationService {

    public CompletableFuture<Result> validateSeverities(List<TNotification> notifications) {
        return CompletableFutures.safeCall(() -> {
            var result = new Result(new HashMap<>(notifications.size()));
            CompletableFuture<Void> done = CompletableFuture.completedFuture(null);
            for (TNotification tNotification : notifications) {
                done = done.thenAccept(unused -> {
                    var validations = validate(tNotification);
                    result.channels.put(tNotification.getId(), validations);
                });
            }
            return done
                    .thenApply(unused -> result);
        });
    }

    private List<ResultRow> validate(TNotification tNotification) {
        if (tNotification.getDefaultForAlertSeverityList().isEmpty()) {
            return List.of();
        } else {
            List<ResultRow> rows = new ArrayList<>();
            for (Severity severity : tNotification.getDefaultForAlertSeverityList()) {
                switch (severity) {
                    case SEVERITY_DISASTER -> rows.add(new ResultRow(AlertSeverity.DISASTER, "no valid escalation", false));
                    case SEVERITY_CRITICAL -> rows.add(new ResultRow(AlertSeverity.CRITICAL, "no valid escalation", false));
                    case SEVERITY_INFO -> rows.add(new ResultRow(AlertSeverity.INFO, "", true));
                }
            }
            return rows;
        }
    }

    public CompletableFuture<Boolean> validateChannelsForSeverity(
            List<String> channelsId,
            AlertSeverity alertSeverity,
            Resource.Severity resourceSeverity)
    {
        if (channelsId.isEmpty()) {
            return CompletableFuture.completedFuture(false);
        }
        switch (resourceSeverity) {
            case CRITICAL -> {
                switch (alertSeverity) {
                    case INFO -> {
                        return CompletableFuture.completedFuture(true);
                    }
                    default -> {
                        //todo validate escalation
                        return CompletableFuture.completedFuture(false);
                    }
                }
            }
            case NORMAL, NON_CRITICAL -> {
                return CompletableFuture.completedFuture(true);
            }
            // validate like highly critical resource
            default -> {
                switch (alertSeverity) {
                    case INFO -> {
                        return CompletableFuture.completedFuture(true);
                    }
                    default -> {
                        //todo validate escalation
                        return CompletableFuture.completedFuture(false);
                    }
                }
            }
        }
    }

    public static record Result(Map<String, List<ResultRow>> channels) {

    }

    public static record ResultRow(AlertSeverity severity, String error, boolean result) {

    }

}
