package ru.yandex.solomon.alert.cluster.broker.notification;

import java.net.ConnectException;
import java.net.SocketException;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.util.collection.enums.EnumMapToAtomicLong;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DelegateNotificationChannel implements NotificationChannel {
    private static final AtomicReferenceFieldUpdater<DelegateNotificationChannel, NotificationChannel> DELEGATE_FIELD =
            AtomicReferenceFieldUpdater.newUpdater(DelegateNotificationChannel.class, NotificationChannel.class, "delegate");

    private final EnumMapToAtomicLong<NotificationStatus.Code> stats = new EnumMapToAtomicLong<>(NotificationStatus.Code.class);
    private volatile NotificationChannel delegate;

    public DelegateNotificationChannel(NotificationChannel delegate) {
        this.delegate = delegate;
    }

    public void updateDelegate(NotificationChannel actual) {
        NotificationChannel prev = DELEGATE_FIELD.getAndSet(this, actual);
        prev.close();
    }

    @Override
    @Nonnull
    public String getId() {
        return delegate.getId();
    }

    @Nonnull
    @Override
    public String getProjectId() {
        return delegate.getProjectId();
    }

    @Override
    public NotificationType getType() {
        return delegate.getType();
    }

    @Override
    public Notification getNotification() {
        return delegate.getNotification();
    }

    @Override
    @Nonnull
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        return CompletableFutures.safeCall(() -> delegate.send(latestSuccessSend, event))
                .whenComplete((status, e) -> {
                    if (e != null) {
                        status = classifyException(e).withDescription(Throwables.getStackTraceAsString(e));
                    }

                    stats.addAndGet(status.getCode(), 1L);
                });
    }

    private NotificationStatus classifyException(Throwable e) {
        var cause = e;
        while (cause != null) {
            if (cause instanceof ConnectException) {
                return NotificationStatus.ERROR_ABLE_TO_RETRY;
            }

            if (cause instanceof SocketException) {
                return NotificationStatus.ERROR_ABLE_TO_RETRY;
            }

            cause = cause.getCause();
        }

        return NotificationStatus.ERROR;
    }

    @Override
    public DispatchRule getDispatchRule(ChannelConfig channelConfigOverride) {
        return delegate.getDispatchRule(channelConfigOverride);
    }

    @Override
    public boolean isDefault() {
        return delegate.isDefault();
    }

    public NotificationChannel getDelegate() {
        return delegate;
    }

    @Override
    public void close() {
        delegate.close();
    }

    public void appendNotificationMetrics(String projectId, MetricConsumer consumer) {
        for (NotificationStatus.Code code: NotificationStatus.Code.values()) {
            long count = stats.get(code);
            if (count == 0) {
                continue;
            }

            consumer.onMetricBegin(MetricType.COUNTER);
            consumer.onLabelsBegin(4);
            consumer.onLabel("sensor", "channel.notification.status");
            consumer.onLabel("projectId", projectId);
            consumer.onLabel("channelId", delegate.getId());
            consumer.onLabel("status", code.name());
            consumer.onLabelsEnd();
            consumer.onLong(0, count);
            consumer.onMetricEnd();
        }
    }

    @Override
    public String toString() {
        return delegate.toString();
    }
}
