package ru.yandex.solomon.alert.cluster.broker.notification;

import java.time.Clock;
import java.time.Instant;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.jns.client.JnsClient;
import ru.yandex.jns.dto.JnsListEscalationPolicy;
import ru.yandex.jns.dto.ListEscalationRequest;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.solomon.alert.api.converters.NotificationConverter;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.cluster.broker.notification.search.NotificationSearch;
import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.dao.NotificationsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.NotificationServiceMetrics;
import ru.yandex.solomon.alert.notification.channel.DevNullOrDefaultsNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationChannelFactory;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.state.StatefulNotificationChannel;
import ru.yandex.solomon.alert.notification.state.StatefulNotificationChannelFactory;
import ru.yandex.solomon.alert.protobuf.EscalationView;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TCreateNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TDeleteNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TListEscalationsRequest;
import ru.yandex.solomon.alert.protobuf.TListEscalationsResponse;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsResponse;
import ru.yandex.solomon.alert.protobuf.TReadNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TReadNotificationResponse;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsResponse;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationRequest;
import ru.yandex.solomon.alert.protobuf.TUpdateNotificationResponse;
import ru.yandex.solomon.idempotency.IdempotentOperation;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyAsyncCreateNotification;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyAsyncDeleteNotification;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyAsyncListEscalations;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyAsyncListNotification;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyAsyncResolveNotificationDetails;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyAsyncUpdateNotification;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyCreateNotification;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyResolveListEscalations;
import static ru.yandex.solomon.alert.cluster.broker.notification.ProtoReply.replyUpdateNotification;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.CONCURRENT_MODIFICATION;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.INTERNAL_ERROR;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.INVALID_REQUEST;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.NODE_UNAVAILABLE;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.NOT_FOUND;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.OK;
import static ru.yandex.solomon.alert.protobuf.ERequestStatusCode.SHARD_NOT_INITIALIZED;
import static ru.yandex.solomon.idempotency.IdempotentOperation.NO_OPERATION;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ProjectNotificationService implements NotificationApi, AlertStatefulNotificationChannelsFactory, AutoCloseable {
    private final static Logger logger = LoggerFactory.getLogger(ProjectNotificationService.class);
    private final ConcurrentMap<String, Notification> notificationById = new ConcurrentHashMap<>();
    private final ConcurrentMap<String, DelegateNotificationChannel> notificationChannelById = new ConcurrentHashMap<>();
    private final DefaultChannelsHolder defaultChannelsHolder = new DefaultChannelsHolder();

    private final String projectId;
    private final Clock clock;
    private final NotificationsDao notificationDao;
    private final NotificationChannelFactory channelFactory;
    private final NotificationConverter notificationConverter;
    private final NotificationSearch notificationSearch;
    private final JnsClient jnsClient;
    private final StatefulNotificationChannelFactory statefulChannelFactory;
    private final NotificationServiceMetrics metrics;
    private final AtomicReference<State> state = new AtomicReference<>(State.IDLE);

    public ProjectNotificationService(
            String projectId,
            Clock clock,
            EntitiesDao<Notification> notificationDao,
            NotificationChannelFactory channelFactory,
            StatefulNotificationChannelFactory statefulChannelFactory,
            NotificationConverter notificationConverter,
            NotificationSearch notificationSearch,
            JnsClient jnsClient)
    {
        this.projectId = projectId;
        this.clock = clock;
        this.notificationDao = (NotificationsDao) notificationDao;
        this.channelFactory = channelFactory;
        this.notificationConverter = notificationConverter;
        this.notificationSearch = notificationSearch;
        this.jnsClient = jnsClient;
        this.metrics = new NotificationServiceMetrics();
        this.statefulChannelFactory = statefulChannelFactory;
    }

    public NotificationServiceMetrics getMetrics() {
        return metrics;
    }

    public boolean isReady() {
        return state.get() == State.RUNNING;
    }

    public CompletableFuture<?> run() {
        CompletableFuture<?> init = new CompletableFuture<>();

        CompletableFuture<?> result = init
                .thenCompose(ignore -> createSchema(State.IDLE))
                .thenCompose(ignore -> fetchNotifications(State.SCHEME_CREATING))
                .thenAccept(ignore -> {
                    State current = state.get();
                    if (current == State.CLOSED || current == State.RUNNING) {
                        return;
                    }

                    if (!changeState(State.FETCH_NOTIFICATIONS, State.RUNNING)) {
                        throw new IllegalStateException("Expected state " + State.FETCH_NOTIFICATIONS + " but was " + state.get());
                    }
                });

        init.complete(null);
        return result;
    }

    private CompletableFuture<?> createSchema(State from) {
        if (!changeState(from, State.SCHEME_CREATING)) {
            return completedFuture(null);
        }

        return notificationDao.createSchema(projectId)
                .whenComplete((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed create schema for project: " + projectId, e);
                        changeState(State.SCHEME_CREATING, from);
                    }
                });
    }

    private CompletableFuture<?> fetchNotifications(State from) {
        if (!changeState(from, State.FETCH_NOTIFICATIONS)) {
            return completedFuture(null);
        }

        return notificationDao.find(projectId, value -> {
            notificationById.put(value.getId(), value);
            if (value.isDefaultForProject()) {
                // create delegate for defaults anyway
                notificationChannelById.computeIfAbsent(value.getId(), id -> {
                    return createChannel(value.getFolderId(), value.getId(), value);
                });
            }
        }).whenComplete((ignore, e) -> {
            if (e != null) {
                logger.error("Failed fetch notifications for project: " + projectId, e);
                changeState(State.FETCH_NOTIFICATIONS, from);
            }
        });
    }

    @Override
    public CompletableFuture<TCreateNotificationResponse> createNotification(TCreateNotificationRequest request) {
        if (!isReady()) {
            return replyAsyncCreateNotification(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        Instant now = clock.instant();
        Notification notification = notificationConverter.protoToNotification(request.getNotification())
                .toBuilder()
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .setVersion(1)
                .build();

        if (notificationById.containsKey(notification.getId())) {
            return replyAsyncCreateNotification(INVALID_REQUEST, "Notification with id " + notification.getId() + " already exists");
        }

        return notificationDao.insert(notification, IdempotentOperation.NO_OPERATION)
                .handle((maybePrev, e) -> {
                    if (e != null) {
                        logger.error("Failed create notification {} request into project {}", notification, notification.getProjectId(), e);
                        return replyCreateNotification(INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                    }

                    if (maybePrev.isPresent()) {
                        // DB and notificationById out of sync

                        Notification prev = maybePrev.get();

                        if (!prev.equals(notification)) {
                            // make notification (better late than never) and return conflict
                            upsertNotification(prev);

                            return replyCreateNotification(INVALID_REQUEST, "Notification with id " + notification.getId() + " already exists");
                        }
                    }
                    // prev = null or prev = notification

                    upsertNotification(notification);
                    return replyCreateNotification(notificationConverter.notificationToProto(notification));
                });
    }

    private void upsertNotification(Notification notification) {
        // FIXME: Maybe there's a bug in this code https://st.yandex-team.ru/SOLOMON-6428
        actualizeChannel(notification);
        notificationById.put(notification.getId(), notification);
    }

    @Override
    public CompletableFuture<TReadNotificationResponse> readNotification(TReadNotificationRequest request) {
        if (!isReady()) {
            return ProtoReply.replyAsyncReadNotification(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        Notification notification = notificationById.get(request.getNotificationId());
        if (notification == null) {
            return ProtoReply.replyAsyncReadNotification(NOT_FOUND, "Notification with id " + request.getNotificationId() + " does not exist");
        }

        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            if (!notification.getFolderId().equals(folderId)) {
                return ProtoReply.replyAsyncReadNotification(NOT_FOUND, "Notification with id " + request.getNotificationId() + " does not exist in folderId = " + folderId);
            }
        }

        return ProtoReply.replyAsyncReadNotification(notificationConverter.notificationToProto(notification));
    }

    @Override
    public CompletableFuture<TUpdateNotificationResponse> updateNotification(TUpdateNotificationRequest request) {
        if (!isReady()) {
            return replyAsyncUpdateNotification(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        Notification notification = notificationConverter.protoToNotification(request.getNotification());
        Notification prev = notificationById.get(notification.getId());
        if (prev == null) {
            return replyAsyncUpdateNotification(NOT_FOUND, "Notification with id " + notification.getId() + " does not exist");
        }

        if (!notification.getFolderId().isEmpty()) {
            if (!prev.getFolderId().equals(notification.getFolderId())) {
                return replyAsyncUpdateNotification(NOT_FOUND, "Notification with id " + notification.getId() + " does not exist in folderId = " + notification.getFolderId());
            }
        }

        if (notification.getVersion() != -1 && prev.getVersion() != notification.getVersion()) {
            return replyAsyncUpdateNotification(CONCURRENT_MODIFICATION, "Version mismatch for notification: " + notification.getId());
        }

        Instant now = clock.instant();
        Notification updated = notification.toBuilder()
                .setFolderId(prev.getFolderId())
                .setCreatedAt(prev.getCreatedAt())
                .setCreatedBy(prev.getCreatedBy())
                .setUpdatedAt(now)
                .setVersion(prev.getVersion() + 1)
                .build();

        var set = new HashSet<>(prev.getDefaultForSeverity());
        set.removeAll(updated.getDefaultForSeverity());
        return notificationDao.updateWithValidations(updated, NO_OPERATION, set)
                .handle((maybePrev, e) -> {
                    if (e != null) {
                        logger.error("Failed to update notification {} into project {}", updated.getId(), updated.getProjectId(), e);
                        var cause = Throwables.getRootCause(e);
                        if (cause instanceof IllegalArgumentException iae) {
                            return replyUpdateNotification(INVALID_REQUEST, iae.getMessage());
                        }
                        return replyUpdateNotification(INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                    }

                    if (maybePrev.isEmpty()) {
                        // No record in db for an existing activity.
                        removeNotification(prev.getId());
                        return replyUpdateNotification(NOT_FOUND, "Notification with id " + notification.getId() + " does not exist");
                    }

                    Notification prevInDb = maybePrev.get();
                    Notification next;
                    if (prevInDb.getVersion() == updated.getVersion() - 1 || prevInDb.equals(updated)) {
                        // regular update
                        upsertNotification(updated);
                        return replyUpdateNotification(notificationConverter.notificationToProto(updated));
                    }

                    // db and activity are out of sync. Recreate activity from db
                    upsertNotification(prevInDb);
                    return replyUpdateNotification(CONCURRENT_MODIFICATION, "Version mismatch for notification: " + notification.getId());
                });
    }

    @Override
    public CompletableFuture<TDeleteNotificationResponse> deleteNotification(TDeleteNotificationRequest request) {
        if (!isReady()) {
            return replyAsyncDeleteNotification(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        String notificationId = request.getNotificationId();
        if (!notificationById.containsKey(notificationId)) {
            return replyAsyncDeleteNotification(NOT_FOUND, "Notification with id " + notificationId + " does not exist");
        }

        Notification notification = notificationById.get(notificationId);
        String folderId = request.getFolderId();
        if (!folderId.isEmpty()) {
            if (!notification.getFolderId().equals(folderId)) {
                return replyAsyncDeleteNotification(NOT_FOUND, "Notification with id " + notificationId + " does not exist in folderId = " + folderId);
            }
        }

        return notificationDao.deleteByIdWithValidations(projectId, notificationId, NO_OPERATION, notification.getDefaultForSeverity())
                .handle((ignore, e) -> {
                    if (e != null) {
                        logger.error("Failed to delete notification {} into project {}", request.getNotificationId(), request.getProjectId(), e);
                        var cause = Throwables.getRootCause(e);
                        if (cause instanceof IllegalArgumentException iae) {
                            return ProtoReply.replyDeleteNotification(INVALID_REQUEST, iae.getMessage());
                        }
                        return ProtoReply.replyDeleteNotification(INTERNAL_ERROR, Throwables.getStackTraceAsString(e));
                    }

                    removeNotification(notificationId);

                    return ProtoReply.replyDeleteNotification(notificationId);
                });
    }

    private void removeNotification(String notificationId) {
        Notification prev = notificationById.remove(notificationId);
        if (prev != null) {
            DelegateNotificationChannel delegate = notificationChannelById.get(notificationId);
            if (delegate != null) {
                defaultChannelsHolder.remove(prev);
                delegate.updateDelegate(new DevNullOrDefaultsNotificationChannel(
                        notificationId,
                        projectId,
                        defaultChannelsHolder.defaultChannelsMap.values()));
            }
        }
    }

    @Override
    public CompletableFuture<TListNotificationsResponse> listNotification(TListNotificationsRequest request) {
        if (!isReady()) {
            return replyAsyncListNotification(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        return completedFuture(notificationSearch.listNotifications(notificationById.values(), request));
    }

    @Override
    public CompletableFuture<TResolveNotificationDetailsResponse> resolveNotificationDetails(TResolveNotificationDetailsRequest request) {
        if (!isReady()) {
            return replyAsyncResolveNotificationDetails(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        return completedFuture(notificationSearch.resolveNotificationDetails(notificationById.values(), request));
    }

    @Override
    public CompletableFuture<TListEscalationsResponse> listEscalations(TListEscalationsRequest request) {
        if (!isReady()) {
            return replyAsyncListEscalations(SHARD_NOT_INITIALIZED, "Expected running state, but was " + state.get());
        }

        return jnsClient.listEscalations(new ListEscalationRequest(request.getProjectId(), request.getFilterByName()))
                .handle((jnsListEscalationPolicy, throwable) -> {
                    if (throwable != null) {
                        logger.error("Error while validating escalations, let trust them", throwable);
                        return new JnsListEscalationPolicy("Error:" + throwable.getMessage(), "", List.of());
                    }
                    return jnsListEscalationPolicy;
                })
                .thenApply(result -> {
                    if (!StringUtils.isEmpty(result.error())) {
                        logger.error("Error while validating escalations, let trust them, {} - {}", result.error(), result.message());
                        return replyResolveListEscalations(NODE_UNAVAILABLE, "Expected running state, but was " + state.get());
                    }
                    return TListEscalationsResponse.newBuilder()
                            .setRequestStatus(OK)
                            .addAllEscalationViews(result.policies().stream()
                                    .map(policy -> EscalationView.newBuilder()
                                            .setId(policy.name())
                                            .setTitle(policy.title())
                                            .build())
                                    .collect(Collectors.toList()))
                            .build();
                });
    }

    @Override
    public Map<String, StatefulNotificationChannel> prepareChannels(Alert alert) {
        if (alert.getNotificationChannels().isEmpty()) {
            return Collections.emptyMap();
        }

        Map<String, StatefulNotificationChannel> result = new HashMap<>();
        for (var channelWithConfig : alert.getNotificationChannels().entrySet()) {
            String channelId = channelWithConfig.getKey();
            ChannelConfig config = channelWithConfig.getValue();
            result.put(channelId, prepareChannel(alert, channelId, config));
        }

        return result;
    }

    private StatefulNotificationChannel prepareChannel(Alert alert, String channelId, ChannelConfig alertChannelConfig) {
        NotificationChannel channel = getOrCreateNotificationChannel(alert.getFolderId(), channelId);
        return statefulChannelFactory.create(alert, channel, alertChannelConfig, metrics);
    }


    private DelegateNotificationChannel getOrCreateNotificationChannel(String folderId, String channelId) {
        DelegateNotificationChannel delegate = notificationChannelById.get(channelId);
        if (delegate != null) {
            return delegate;
        }

        return notificationChannelById.computeIfAbsent(channelId, id -> {
            Notification notification = notificationById.get(id);
            return createChannel(folderId, id, notification);
        });
    }

    private DelegateNotificationChannel createChannel(String folderId, String id, @Nullable Notification notification) {
        final NotificationChannel channel;
        if (notification != null && notification.getFolderId().equals(folderId)) {
            channel = channelFactory.createChannel(notification);
        } else {
            channel = new DevNullOrDefaultsNotificationChannel(
                    id,
                    projectId,
                    defaultChannelsHolder.defaultChannelsMap.values());
        }

        var result = new DelegateNotificationChannel(channel);
        if (notification != null && notification.getFolderId().equals(folderId) && notification.isDefaultForProject()) {
            defaultChannelsHolder.update(notification, result);
        }
        return result;
    }

    private void actualizeChannel(Notification update) {
        DelegateNotificationChannel delegate = notificationChannelById.get(update.getId());
        if (delegate != null) {
            delegate.updateDelegate(channelFactory.createChannel(update));
            defaultChannelsHolder.update(update, delegate);
        } else if (update.isDefaultForProject()) {
            // create delegate for defaults anyway
            notificationChannelById.computeIfAbsent(update.getId(), id -> {
                return createChannel(update.getFolderId(), update.getId(), update);
            });
        }
    }

    @Override
    public void close() {
        var prev = state.getAndSet(State.CLOSED);
        if (prev == State.CLOSED) {
            return;
        }

        notificationChannelById.values().parallelStream().forEach(DelegateNotificationChannel::close);
    }

    private boolean changeState(State from, State to) {
        if (this.state.compareAndSet(from, to)) {
            logger.info("{}: notifications change state {} -> {}", projectId, from, to);
            return true;
        }

        return false;
    }

    public void appendNotificationMetrics(MetricConsumer consumer) {
        notificationChannelById.values().forEach(channel -> channel.appendNotificationMetrics(projectId, consumer));
    }

    @VisibleForTesting
    public DefaultChannelsHolder getDefaultChannelsHolder() {
        return defaultChannelsHolder;
    }

    @VisibleForTesting
    public DelegateNotificationChannel getNotificationChannelById(String id) {
        return notificationChannelById.get(id);
    }

    private enum State {
        IDLE,
        SCHEME_CREATING,
        FETCH_NOTIFICATIONS,
        RUNNING,
        CLOSED,
    }

    public static class DefaultChannelsHolder {
        private final ConcurrentMap<String, DelegateNotificationChannel> defaultChannelsMap = new ConcurrentHashMap<>();

        @VisibleForTesting
        public ConcurrentMap<String, DelegateNotificationChannel> getDefaultChannelsMap() {
            return defaultChannelsMap;
        }

        public void remove(Notification notification) {
            defaultChannelsMap.remove(notification.getId());
        }

        public void update(Notification notification, DelegateNotificationChannel channel) {
            if (notification.isDefaultForProject()) {
                defaultChannelsMap.putIfAbsent(notification.getId(), channel);
            } else {
                remove(notification);
            }
        }
    }
}
