package ru.yandex.solomon.alert.cluster.broker.notification.search;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.alert.protobuf.Severity;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.notification.ENotificationChannelType;

/**
 * @author Vladimir Gordiychuk
 */
class NotificationFilters {
    private NotificationFilters() {
    }

    private static Predicate<Notification> filterByFolderId(String folderId) {
        if (folderId.isEmpty()) {
            return ignore -> true;
        } else {
            return entry -> folderId.equals(entry.getFolderId());
        }
    }

    private static Predicate<Notification> filterByName(String name) {
        return entry -> StringUtils.containsIgnoreCase(entry.getName(), name);
    }

    private static Predicate<Notification> filterByType(Set<NotificationType> types) {
        return entry -> types.contains(entry.getType());
    }

    private static Predicate<Notification> filterByType(List<ENotificationChannelType> protos) {
        EnumSet<NotificationType> types = EnumSet.noneOf(NotificationType.class);
        for (ENotificationChannelType proto : protos) {
            types.add(NotificationType.forNumber(proto.getNumber()));
        }

        return filterByType(types);
    }

    private static Predicate<Notification> filterBySeverity(List<Severity> protos) {
        EnumSet<AlertSeverity> severities = EnumSet.noneOf(AlertSeverity.class);
        for (var proto : protos) {
            severities.add(AlertSeverity.forNumber(proto.getNumber()));
        }

        return entry -> {
            for (AlertSeverity severity : entry.getDefaultForSeverity()) {
                if (severities.contains(severity)) {
                    return true;
                }
            }
            return false;
        };
    }

    private static Predicate<Notification> filterById(Set<String> notificationIds) {
        return entry -> notificationIds.contains(entry.getId());
    }

    public static Predicate<Notification> filterBy(TListNotificationsRequest request) {
        Predicate<Notification> predicate = filterByFolderId(request.getFolderId());
        if (!request.getFilterByName().isEmpty()) {
            predicate = predicate.and(filterByName(request.getFilterByName()));
        }

        if (request.getFilterByTypeCount() > 0) {
            predicate = predicate.and(filterByType(request.getFilterByTypeList()));
        }

        if (request.getFilterByDefaultSeverityCount() > 0) {
            predicate = predicate.and(filterBySeverity(request.getFilterByDefaultSeverityList()));
        }

        return predicate;
    }

    public static Predicate<Notification> filterBy(TResolveNotificationDetailsRequest request) {
        Set<String> notificationIds = request.getNotificationIdsList().stream().collect(Collectors.toUnmodifiableSet());

        Predicate<Notification> predicate = filterByFolderId(request.getFolderId());

        return predicate.and(filterById(notificationIds));
    }
}
