package ru.yandex.solomon.alert.cluster.broker.notification.search;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.solomon.alert.api.converters.NotificationConverter;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.TListNotificationsResponse;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsRequest;
import ru.yandex.solomon.alert.protobuf.TResolveNotificationDetailsResponse;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.alert.protobuf.notification.TNotificationDetails;

import static ru.yandex.solomon.alert.cluster.broker.notification.search.NotificationFilters.filterBy;
import static ru.yandex.solomon.alert.cluster.broker.notification.search.NotificationSorts.orderBy;

/**
 * @author Vladimir Gordiychuk
 */
public class NotificationSearch {

    private final NotificationConverter notificationConverter;

    public NotificationSearch(NotificationConverter notificationConverter) {
        this.notificationConverter = notificationConverter;
    }

    public  TListNotificationsResponse listNotifications(Collection<Notification> source, TListNotificationsRequest request) {
        final int offset = getOffset(request.getPageToken());
        final int pageSize = getPageSize(request.getPageSize());

        MutableInt cursor = new MutableInt(offset);
        List<TNotification> result = source.stream()
                .filter(filterBy(request))
                .sorted(orderBy(request))
                .skip(offset)
                .peek(ignore -> cursor.increment())
                .limit(pageSize + 1)
                .map(notificationConverter::notificationToProto)
                .collect(Collectors.toList());

        final String nextToken;
        if (result.size() > pageSize) {
            result = result.subList(0, pageSize);
            cursor.decrement();
            nextToken = String.valueOf(cursor.intValue());
        } else {
            nextToken = "";
        }

        return TListNotificationsResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK)
                .addAllNotification(result)
                .setNextPageToken(nextToken)
                .build();
    }

    public TResolveNotificationDetailsResponse resolveNotificationDetails(Collection<Notification> source, TResolveNotificationDetailsRequest request) {
        List<TNotificationDetails> result = source.stream()
            .filter(filterBy(request))
            .map(notificationConverter::notificationToDetailsProto)
            .collect(Collectors.toList());

        return TResolveNotificationDetailsResponse.newBuilder()
            .setRequestStatus(ERequestStatusCode.OK)
            .addAllNotificationDetails(result)
            .build();
    }

    private  int getPageSize(int pageSize) {
        return pageSize <= 0 ? 10 : pageSize;
    }

    private  int getOffset(String token) {
        return token.isEmpty() ? 0 : Integer.parseInt(token);
    }
}
