package ru.yandex.solomon.alert.cluster.project;

import java.util.Collections;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import ru.yandex.solomon.alert.domain.StringInterner;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;
import ru.yandex.solomon.alert.protobuf.TProjectAssignmentResponse;
import ru.yandex.solomon.balancer.AssignmentSeqNo;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public final class AssignmentConverter {
    private AssignmentConverter() {
    }

    public static TProjectAssignmentResponse toProto(AssignmentSnapshot snapshot) {
        return TProjectAssignmentResponse.newBuilder()
                .addAllAssignEntries(snapshot.getAssignmentByProjectId()
                        .values()
                        .stream()
                        .map(AssignmentConverter::toProto)
                        .collect(toList()))
                .setSeqNo(toProto(snapshot.getAssignmentSeqNo()))
                .build();
    }

    public static TAssignmentSeqNo toProto(AssignmentSeqNo seqNo) {
        return TAssignmentSeqNo.newBuilder()
                .setProjectSeqNo(seqNo.getAssignSeqNo())
                .setLeaderSeqNo(seqNo.getLeaderSeqNo())
                .build();
    }

    private static TProjectAssignmentResponse.Entry toProto(ProjectAssignment assignment) {
        return TProjectAssignmentResponse.Entry.newBuilder()
                .setProjectId(assignment.getProjectId())
                .setSeqNo(toProto(assignment.getSeqNo()))
                .setAddress(assignment.getAddress())
                .build();
    }

    public static AssignmentSnapshot fromProto(TProjectAssignmentResponse proto) {
        Map<String, ProjectAssignment> assignments = proto.getAssignEntriesList()
                .stream()
                .map(AssignmentConverter::fromProto)
                .collect(Collectors.toMap(ProjectAssignment::getProjectId, Function.identity()));
        AssignmentSeqNo seqNo = fromProto(proto.getSeqNo());
        return new AssignmentSnapshot(Collections.unmodifiableMap(assignments), seqNo);
    }

    private static ProjectAssignment fromProto(TProjectAssignmentResponse.Entry proto) {
        return new ProjectAssignment(
            StringInterner.I.intern(proto.getProjectId()),
            StringInterner.I.intern(proto.getAddress()),
            fromProto(proto.getSeqNo()));
    }

    public static AssignmentSeqNo fromProto(TAssignmentSeqNo proto) {
        return new AssignmentSeqNo(proto.getLeaderSeqNo(), proto.getProjectSeqNo());
    }
}
