package ru.yandex.solomon.alert.cluster.server.grpc;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.alert.evaluation.EvaluationService;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusRequest;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusResponse;
import ru.yandex.solomon.selfmon.ng.ProcSelfMon;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class EvaluationServerStatusHandler implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(EvaluationServerStatusHandler.class);

    private final EvaluationService evaluationService;
    private final PingActorRunner actor;

    private final Meter cpuMs = Meter.of(ExpMovingAverage.oneMinute());
    private long prevUtimeMs;

    public EvaluationServerStatusHandler(EvaluationService evaluationService, ScheduledExecutorService timer, Executor executor) {
        this.evaluationService = evaluationService;
        this.actor = PingActorRunner.newBuilder()
                .onPing(this::act)
                .operation("resource_usage_watcher")
                .pingInterval(Duration.ofSeconds(5))
                .timer(timer)
                .executor(executor)
                .build();
        actor.schedule();
    }

    CompletableFuture<Void> forceAct() {
        return actor.forcePing();
    }

    public EvaluationServerStatusResponse handle(EvaluationServerStatusRequest request) {
        var statistics = evaluationService.statistics();
        return EvaluationServerStatusResponse.newBuilder()
                .setNode(HostUtils.getFqdn())
                .setAssignmentsCount(statistics.assignments())
                .setEvaluationRate(statistics.evaluationRate())
                .setCpuNanos(millisToNanos(cpuMs.getRate(TimeUnit.SECONDS)))
                .build();
    }

    public CompletableFuture<?> act(int attempt) {
        updateCpu();
        return CompletableFuture.completedFuture(null);
    }

    private double millisToNanos(double millis) {
        return millis * 1e+6;
    }

    private void updateCpu() {
        try {
            long utimeMs = ProcSelfMon.getUtimeMs();
            long delta = utimeMs - prevUtimeMs;
            if (delta > 0) {
                cpuMs.mark(delta);
            }
            prevUtimeMs = utimeMs;
        } catch (Throwable e) {
            // ProcSelfMon can not work on ci https://paste.yandex-team.ru/631268
            logger.error("prepare node summary failed: ", e);
        }
    }

    @Override
    public void close() throws Exception {
        actor.close();
    }
}
