package ru.yandex.solomon.alert.cluster.server.grpc;

import javax.annotation.ParametersAreNonnullByDefault;

import io.grpc.stub.StreamObserver;

import ru.yandex.grpc.utils.StreamObservers;
import ru.yandex.solomon.alert.cluster.balancer.AlertingBalancer;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusRequest;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusResponse;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage;
import ru.yandex.solomon.alert.protobuf.TAlertingClusterServiceGrpc;
import ru.yandex.solomon.alert.protobuf.TAssignProjectRequest;
import ru.yandex.solomon.alert.protobuf.TAssignProjectResponse;
import ru.yandex.solomon.alert.protobuf.THeartbeatRequest;
import ru.yandex.solomon.alert.protobuf.THeartbeatResponse;
import ru.yandex.solomon.alert.protobuf.TProjectAssignmentRequest;
import ru.yandex.solomon.alert.protobuf.TProjectAssignmentResponse;
import ru.yandex.solomon.alert.protobuf.TUnassignProjectRequest;
import ru.yandex.solomon.alert.protobuf.TUnassignProjectResponse;

import static ru.yandex.grpc.utils.StreamObservers.asyncComplete;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class GrpcAlertingClusterService extends TAlertingClusterServiceGrpc.TAlertingClusterServiceImplBase {
    private final AlertingBalancer balancer;
    private final AlertingLocalShards localShards;
    private final GrpcEvaluationService grpcEvaluationService;

    public GrpcAlertingClusterService(
        AlertingBalancer balancer,
        AlertingLocalShards localShards,
        GrpcEvaluationService grpcEvaluationService)
    {
        this.balancer = balancer;
        this.localShards = localShards;
        this.grpcEvaluationService = grpcEvaluationService;
    }

    @Override
    public void heartbeat(THeartbeatRequest request, StreamObserver<THeartbeatResponse> responseObserver) {
        StreamObservers.asyncComplete(balancer.receiveHeartbeat(request), responseObserver);
    }

    @Override
    public void assignProject(TAssignProjectRequest request, StreamObserver<TAssignProjectResponse> responseObserver) {
        asyncComplete(localShards.assignShard(request), responseObserver);
    }

    @Override
    public void unassignProject(TUnassignProjectRequest request, StreamObserver<TUnassignProjectResponse> responseObserver) {
        asyncComplete(localShards.unassignShard(request), responseObserver);
    }

    @Override
    public void projectAssignment(TProjectAssignmentRequest request, StreamObserver<TProjectAssignmentResponse> responseObserver) {
        asyncComplete(balancer.getAssignmentSnapshot(request), responseObserver);
    }

    @Override
    public void evaluationServerStatus(EvaluationServerStatusRequest request, StreamObserver<EvaluationServerStatusResponse> responseObserver) {
        asyncComplete(grpcEvaluationService.evaluationStatus(request), responseObserver);
    }

    @Override
    public StreamObserver<EvaluationStreamClientMessage> evaluationStream(StreamObserver<EvaluationStreamServerMessage> responseObserver) {
        return grpcEvaluationService.evaluationStream(responseObserver);
    }
}
