package ru.yandex.solomon.alert.cluster.server.grpc;

import java.time.Clock;
import java.util.List;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.grpc.utils.StreamObservers;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.protobuf.TAssignProjectRequest;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;
import ru.yandex.solomon.alert.protobuf.TUnassignProjectRequest;
import ru.yandex.solomon.balancer.BalancerServiceGrpc;
import ru.yandex.solomon.balancer.TAssignShardRequest;
import ru.yandex.solomon.balancer.TAssignShardResponse;
import ru.yandex.solomon.balancer.TBalancerServiceGrpc;
import ru.yandex.solomon.balancer.TCreateShardRequest;
import ru.yandex.solomon.balancer.TCreateShardResponse;
import ru.yandex.solomon.balancer.TNodeSummary;
import ru.yandex.solomon.balancer.TPingRequest;
import ru.yandex.solomon.balancer.TPingResponse;
import ru.yandex.solomon.balancer.TShardSummary;
import ru.yandex.solomon.balancer.TShardSummary.EStatus;
import ru.yandex.solomon.balancer.TUnassignShardRequest;
import ru.yandex.solomon.balancer.TUnassignShardResponse;
import ru.yandex.solomon.balancer.protobuf.TBalancerResources;
import ru.yandex.solomon.selfmon.ng.ProcSelfMon;
import ru.yandex.solomon.util.host.HostUtils;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcBalancerService extends TBalancerServiceGrpc.TBalancerServiceImplBase {
    private static final Logger logger = LoggerFactory.getLogger(GrpcBalancerService.class);

    private final Clock clock;
    private final AlertingLocalShards localShards;
    private final long startedAt;

    public GrpcBalancerService(AlertingLocalShards localShards, Clock clock) {
        this.clock = clock;
        this.startedAt = clock.millis();
        this.localShards = localShards;
    }

    @Override
    public void assignShard(TAssignShardRequest request, StreamObserver<TAssignShardResponse> responseObserver) {
        var future = localShards.assignShard(TAssignProjectRequest.newBuilder()
                .setProjectId(request.getShardId())
                .setSeqNo(TAssignmentSeqNo.newBuilder()
                        .setLeaderSeqNo(request.getAssignmentSeqNo().getLeaderSeqNo())
                        .setProjectSeqNo(request.getAssignmentSeqNo().getAssignSeqNo())
                        .build())
                .setExpiredAt(request.getExpiredAt())
                .build())
                .thenApply(ignore -> TAssignShardResponse.getDefaultInstance());

        StreamObservers.asyncComplete(future, responseObserver);
    }

    @Override
    public void unassignShard(TUnassignShardRequest request, StreamObserver<TUnassignShardResponse> responseObserver) {
        var future = localShards.unassignShard(TUnassignProjectRequest.newBuilder()
                .setProjectId(request.getShardId())
                .setSeqNo(TAssignmentSeqNo.newBuilder()
                        .setLeaderSeqNo(request.getAssignmentSeqNo().getLeaderSeqNo())
                        .setProjectSeqNo(request.getAssignmentSeqNo().getAssignSeqNo())
                        .build())
                .setGracefull(request.getGraceful())
                .setExpiredAt(request.getExpiredAt())
                .build())
                .thenApply(ignore -> TUnassignShardResponse.getDefaultInstance());

        StreamObservers.asyncComplete(future, responseObserver);
    }

    @Override
    public void ping(TPingRequest request, StreamObserver<TPingResponse> responseObserver) {
        try {
            responseObserver.onNext(TPingResponse.newBuilder()
                    .setNode(HostUtils.getFqdn())
                    .setNodeSummary(prepareNodeSummary())
                    .addAllShardSummary(prepareShardSummary())
                    .build());

            responseObserver.onCompleted();
        } catch (Throwable e) {
            responseObserver.onError(e);
        }
    }

    private TNodeSummary prepareNodeSummary() {
        try {
            return TNodeSummary.newBuilder()
                    .setMemoryBytes(ProcSelfMon.getRssBytes())
                    .setUtimeMillis(ProcSelfMon.getUtimeMs())
                    .setUpTimeMillis(clock.millis() - startedAt)
                    .build();
        } catch (Throwable e) {
            // ProcSelfMon can not work on ci https://paste.yandex-team.ru/631268
            logger.error("prepare node summary failed: ", e);
            return TNodeSummary.getDefaultInstance();
        }
    }

    private List<TShardSummary> prepareShardSummary() {
        return localShards.stream()
                .map(shard -> TShardSummary.newBuilder()
                        .setShardId(shard.getProjectId())
                        .setStatus(shard.isReady() ? EStatus.READY : EStatus.LOADING)
                        .setUpTimeMillis(clock.millis() - shard.createdAt)
                        .setResources(TBalancerResources.newBuilder()
                                .setAlertsCount(shard.alertsCount())
                                .build())
                        .build())
                .collect(Collectors.toList());
    }

    public Proxy createProxy() {
        return new Proxy(this);
    }

    public static class Proxy extends BalancerServiceGrpc.BalancerServiceImplBase {
        private final GrpcBalancerService delegate;

        public Proxy(GrpcBalancerService delegate) {
            this.delegate = delegate;
        }

        @Override
        public void assignShard(TAssignShardRequest request, StreamObserver<TAssignShardResponse> responseObserver) {
            delegate.assignShard(request, responseObserver);
        }

        @Override
        public void unassignShard(TUnassignShardRequest request, StreamObserver<TUnassignShardResponse> responseObserver) {
            delegate.unassignShard(request, responseObserver);
        }

        @Override
        public void createShard(TCreateShardRequest request, StreamObserver<TCreateShardResponse> responseObserver) {
            delegate.createShard(request, responseObserver);
        }

        @Override
        public void ping(TPingRequest request, StreamObserver<TPingResponse> responseObserver) {
            delegate.ping(request, responseObserver);
        }
    }
}
