package ru.yandex.solomon.alert.cluster.server.grpc;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;

import ru.yandex.grpc.utils.StreamObservers;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.evaluation.EvaluationService;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusRequest;
import ru.yandex.solomon.alert.protobuf.EvaluationServerStatusResponse;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcEvaluationService implements AutoCloseable {

    private final EvaluationService evaluationService;
    private final AssignmentTracker assignmentTracker;
    private final EvaluationServerStatusHandler evaluationServerStatusHandler;
    private final Executor executor;

    private final Set<GrpcEvaluationStream> activeStreams = Sets.newConcurrentHashSet();
    private volatile boolean closed;

    public GrpcEvaluationService(
            EvaluationService evaluationService,
            AssignmentTracker assignmentTracker,
            EvaluationServerStatusHandler evaluationServerStatusHandler,
            Executor executor) {
        this.evaluationService = evaluationService;
        this.assignmentTracker = assignmentTracker;
        this.evaluationServerStatusHandler = evaluationServerStatusHandler;
        this.executor = executor;
    }

    public StreamObserver<EvaluationStreamClientMessage> evaluationStream(StreamObserver<EvaluationStreamServerMessage> output) {
        if (closed) {
            output.onError(Status.CANCELLED.withDescription("EvaluationService closed").asRuntimeException());
            return StreamObservers.noop();
        }

        var stream = new GrpcEvaluationStream(output, evaluationService, assignmentTracker, executor);
        activeStreams.add(stream);
        stream.doneFuture().whenComplete((ignore, e) -> activeStreams.remove(stream));
        return stream;
    }

    public CompletableFuture<EvaluationServerStatusResponse> evaluationStatus(EvaluationServerStatusRequest request) {
        return CompletableFuture.completedFuture(evaluationServerStatusHandler.handle(request));
    }

    @Override
    public void close() {
        closed = true;
        for (var stream : Set.copyOf(activeStreams)) {
            stream.close();
        }
    }

    public CompletableFuture<Void> asyncClose() {
        close();
        return activeStreams.stream()
                .map(GrpcEvaluationStream::doneFuture)
                .collect(Collectors.collectingAndThen(Collectors.toList(), CompletableFutures::allOfVoid));
    }
}
