package ru.yandex.solomon.alert.cluster.server.grpc;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import it.unimi.dsi.fastutil.longs.Long2ObjectMap;
import it.unimi.dsi.fastutil.longs.Long2ObjectOpenHashMap;
import org.jetbrains.annotations.NotNull;

import ru.yandex.misc.actor.ActorRunner;
import ru.yandex.solomon.alert.cluster.project.AssignmentConverter;
import ru.yandex.solomon.alert.cluster.server.grpc.AssignmentTracker.ObsoleteListener;
import ru.yandex.solomon.alert.cluster.server.grpc.evaluation.EvaluationStreamServerMessageWriter;
import ru.yandex.solomon.alert.evaluation.EvaluationService;
import ru.yandex.solomon.alert.protobuf.EvaluationAssignmentKey;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage.AssignEvaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamClientMessage.UnassignEvaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.Evaluation;
import ru.yandex.solomon.alert.protobuf.EvaluationStreamServerMessage.EvaluationError;
import ru.yandex.solomon.alert.protobuf.TAssignmentSeqNo;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.balancer.AssignmentSeqNo;
import ru.yandex.solomon.util.CloseableUtils;
import ru.yandex.solomon.util.collection.queue.ArrayListLockQueue;

import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toAlert;
import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toError;
import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toEvaluation;
import static ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationStreamConverter.toState;

/**
 * @author Vladimir Gordiychuk
 */
public class GrpcEvaluationStream implements StreamObserver<EvaluationStreamClientMessage>, AutoCloseable, ObsoleteListener {
    private final EvaluationStreamServerMessageWriter output;
    private final EvaluationService evaluationService;
    private final AssignmentTracker assignmentTracker;
    private final ActorRunner actor;

    // Inbound
    private final ArrayListLockQueue<EvaluationStreamClientMessage> inboundMessages = new ArrayListLockQueue<>();
    private final CompletableFuture<Status> inboundDone = new CompletableFuture<>();

    // Outbound
    private final ArrayListLockQueue<Evaluation> outboundEvaluations = new ArrayListLockQueue<>();
    private final ArrayListLockQueue<EvaluationError> outboundEvaluationErrors = new ArrayListLockQueue<>();
    private final CompletableFuture<Status> outboundDone = new CompletableFuture<>();

    // State
    private final Map<TAssignmentSeqNo, GroupConsumers> consumersGroupBySeqNo = new HashMap<>();
    private final AtomicReference<CompletableFuture<Void>> actFuture = new AtomicReference<>();

    public GrpcEvaluationStream(
            StreamObserver<EvaluationStreamServerMessage> output,
            EvaluationService evaluationService,
            AssignmentTracker assignmentTracker,
            Executor executor)
    {
        this.output = new EvaluationStreamServerMessageWriter(output);
        this.evaluationService = evaluationService;
        this.assignmentTracker =  assignmentTracker;
        this.actor = new ActorRunner(this::act, executor);
        assignmentTracker.subscribe(this);
    }

    @Override
    public void onNext(EvaluationStreamClientMessage value) {
        inboundMessages.enqueue(value);
        actor.schedule();
    }

    @Override
    public void onError(Throwable t) {
        inboundDone.complete(Status.fromThrowable(t));
        actor.schedule();
    }

    @Override
    public void onCompleted() {
        inboundDone.complete(Status.OK);
        actor.schedule();
    }

    private void act() {
        var doneFuture = actFuture.getAndSet(null);
        try {
            processClose();
            processInboundMessages();
            processOutboundMessages();
        } catch (Throwable e) {
            outboundDone.complete(Status.fromThrowable(e));
            if (!output.isClosed()) {
                actor.schedule();
            }
        } finally {
            if (doneFuture != null) {
                doneFuture.complete(null);
            }
        }
    }

    public CompletableFuture<Void> awaitAct() {
        return actFuture.updateAndGet(prev -> {
            if (prev != null) {
                return prev;
            }
            return new CompletableFuture<>();
        });
    }

    private void processInboundMessages() {
        if (output.isClosed()) {
            return;
        }

        if (inboundDone.isDone()) {
            outboundDone.complete(Status.OK);
            return;
        }

        var messages = inboundMessages.dequeueAll();
        for (var message : messages) {
            processInboundMessage(message);
        }
    }

    private void processInboundMessage(EvaluationStreamClientMessage message) {
        for (var assignment : message.getAssignEvaluationsList()) {
            processInboundAssign(assignment);
        }

        for (var unassignment : message.getUnassignEvaluationsList()) {
            processInboundUnassign(unassignment);
        }
    }

    private void processInboundAssign(AssignEvaluation assign) {
        var key = assign.getAssignmentKey();
        try {
            var alert = toAlert(assign);
            var latestState = toState(assign);
            if (!assignmentTracker.isValid(alert.getProjectId(), AssignmentConverter.fromProto(key.getAssignGroupId()))) {
                enqueueError(toError(key, Status.ABORTED.withDescription("assignment seqNo mismatch")));
                return;
            }

            var consumer = new Consumer(key);
            var group = consumersGroupBySeqNo.computeIfAbsent(key.getAssignGroupId(), (ignore) -> new GroupConsumers());
            group.add(key.getAssignId(), consumer);
            evaluationService.assign(alert, latestState, consumer);
        } catch (Throwable e) {
            enqueueError(toError(key, Status.fromThrowable(e)));
            unassign(key);
        }
    }

    private void processInboundUnassign(UnassignEvaluation unassignment) {
        unassign(unassignment.getAssignKey());
    }

    private void processOutboundMessages() {
        if (output.isClosed()) {
            return;
        }

        if (outboundDone.isDone()) {
            var status = outboundDone.join();
            if (status.isOk()) {
                output.onCompleted();
            } else {
                output.onError(status.asRuntimeException());
            }
            return;
        }

        output.onNextEvaluations(outboundEvaluations.dequeueAll());
        for (var error : outboundEvaluationErrors.dequeueAll()) {
            unassign(error.getAssignmentKey());
            output.onNext(error);
        }
        output.flush();
    }

    private void processClose() {
        if (!inboundDone.isDone() && !outboundDone.isDone()) {
            return;
        }

        for (var group : consumersGroupBySeqNo.values()) {
            group.close();
        }
        consumersGroupBySeqNo.clear();
        assignmentTracker.unsubscribe(this);
    }

    private void unassign(EvaluationAssignmentKey key) {
        if (key.getAssignId() == 0) {
            CloseableUtils.close(consumersGroupBySeqNo.remove(key.getAssignGroupId()));
            return;
        }

        var group = consumersGroupBySeqNo.get(key.getAssignGroupId());
        if (group == null) {
            return;
        }

        group.remove(key.getAssignId());
        if (group.isEmpty()) {
            CloseableUtils.close(consumersGroupBySeqNo.remove(key.getAssignGroupId()));
        }
    }

    private void enqueueError(EvaluationError error) {
        outboundEvaluationErrors.enqueue(error);
        actor.schedule();
    }

    private void enqueueEvaluation(Evaluation evaluation) {
        outboundEvaluations.enqueue(evaluation);
        actor.schedule();
    }

    @Override
    public void close() {
        outboundDone.complete(Status.CANCELLED.withDescription("Closed on server side"));
        actor.schedule();
    }

    public CompletableFuture<Void> doneFuture() {
        return CompletableFuture.allOf(inboundDone, outboundDone);
    }

    @Override
    public void obsolete(String projectId, AssignmentSeqNo seqNo) {
        enqueueError(toError(AssignmentConverter.toProto(seqNo), Status.ABORTED.withDescription("Project " + projectId + " seqNo mismatch")));
    }

    private static class GroupConsumers implements AutoCloseable {
        private final Long2ObjectMap<Consumer> consumerByAssignId = new Long2ObjectOpenHashMap<>();

        public void add(long assignId, Consumer consumer) {
            var prev = consumerByAssignId.put(assignId, consumer);
            if (prev != null) {
                prev.close();
            }
        }

        public void remove(long assignId) {
            CloseableUtils.close(consumerByAssignId.remove(assignId));
        }

        public boolean isEmpty() {
            return consumerByAssignId.isEmpty();
        }

        @Override
        public void close() {
            for (var consumer : consumerByAssignId.values()) {
                consumer.close();
            }
        }
    }

    private class Consumer implements EvaluationService.Consumer, AutoCloseable {
        private final EvaluationAssignmentKey key;
        private volatile boolean closed;

        public Consumer(EvaluationAssignmentKey key) {
            this.key = key;
        }

        @Override
        public boolean isCanceled() {
            return closed;
        }

        @Override
        public void consume(@NotNull EvaluationState state) {
            if (!closed) {
                enqueueEvaluation(toEvaluation(key, state));
            }
        }

        @Override
        public void onComplete() {
            if (!closed) {
                enqueueError(toError(key, Status.CANCELLED));
            }
        }

        @Override
        public AssignmentSeqNo getSeqNo() {
            return AssignmentConverter.fromProto(key.getAssignGroupId());
        }

        @Override
        public void close() {
            closed = true;
        }
    }
}
